/****************************************************************************
 *                              DFIOMgr.cc
 *
 * Author: Matthew Ballance
 * Desc:   Implements a manager for multiple DFIO's and DFIOFilters.
 * <Copyright> (c) 2001-2003 Matthew Ballance (mballance@users.sourceforge.net)
 *
 *    This source code is free software; you can redistribute it
 *    and/or modify it in source code form under the terms of the GNU
 *    General Public License as published by the Free Software
 *    Foundation; either version 2 of the License, or (at your option)
 *    any later version.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 *
 * </Copyright>
 ****************************************************************************/
#include "DFIOMgr.h"
#include <vector.h>
#include "ivi_String.h"
#include "ConfigDB.h"
#include <tcl.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include "WidgetManager.h"
#include "TclInterpObj.h"
#include "CmdSwitcher.h"

class DFIOMgr {
    public:
        DFIOMgr(void);

        Vector<DFIOInfo>        *dfioList;
        Vector<DFIOFilter>      *dfioFilterList;

        static DFIOMgr *GetDFIOMgr();

    private:
        static DFIOMgr          *Globl_DFIOMgr;
};

DFIOMgr *DFIOMgr::Globl_DFIOMgr = 0;

/**********************************************************
 * GetDFIOMgr()
 **********************************************************/
DFIOMgr *DFIOMgr::GetDFIOMgr()
{
    if (!Globl_DFIOMgr) {
        Globl_DFIOMgr = new DFIOMgr();
    }

    return Globl_DFIOMgr;
}

/**********************************************************
 * DFIOMgr()
 **********************************************************/
DFIOMgr::DFIOMgr(void)
{
    dfioList       = new Vector<DFIOInfo>();
    dfioFilterList = new Vector<DFIOFilter>();
}

/**********************************************************
 * DFIOMgr_RegisterDFIO()
 **********************************************************/
Int32 DFIOMgr_RegisterDFIO(
        DFIOInfo    *dfio
        )
{
    DFIOMgr *dMgr = DFIOMgr::GetDFIOMgr();
    dMgr->dfioList->append(dfio);

    return 0;
}


/**********************************************************
 * DFIOMgr_NewDFIO()
 **********************************************************/
DFIO *DFIOMgr_NewDFIO(const char *dfioTypeName, Tcl_Interp *interp,
        int argc, char **argv)
{
    DFIOMgr    *dMgr = DFIOMgr::GetDFIOMgr();
    DFIOInfo   *dInfo = 0;
    Uint32      i;
  
    if (!dfioTypeName) {
        dfioTypeName = ConfigDB_GetCurrent("App.DefaultDFIO");
    }

    for (i=0; i<dMgr->dfioList->length(); i++) {
        dInfo = dMgr->dfioList->idx(i);
        if (dInfo->getName().equal(dfioTypeName)) {
            break;
        }
        dInfo = 0;
    }

    if (dInfo) {
        return dInfo->newDFIO(interp, argc, argv);
    } else {
        fprintf(stderr, "ERROR :: dfioFind on \"%s\" failed\n",
                (dfioTypeName)?dfioTypeName:"null");
        if (dMgr->dfioList->length()) {
            dInfo = dMgr->dfioList->idx(0);
            fprintf(stderr, "\tDefaulting to DFIO \"%s\"\n",
                    dInfo->getName().value());
            return dInfo->newDFIO(interp, argc, argv);
        } else {
            fprintf(stderr, "ERROR :: no DFIO's registered\n");
            fprintf(stderr, "\tExiting now...\n");
            exit(1);
        }
    }

    return 0;
}

/**********************************************************
 * DFIOMgr_RegisterDFIOFilter()
 **********************************************************/
Int32 DFIOMgr_RegisterDFIOFilter(
        DFIOFilter        *dfioFilter
        )
{
    DFIOMgr    *dMgr = DFIOMgr::GetDFIOMgr();

    dMgr->dfioFilterList->append(dfioFilter);
    return 0;
}

typedef enum {
    DMC_Create,
    DMC_FilterList,
    DMC_FindExtHandler,
    DMC_FindFilter,
    DMC_NumCmds
} DFIOMgrCmds;

static CmdSwitchStruct    dm_cmds[] = {
    { "create",                DMC_Create           },
    { "filter_list",           DMC_FilterList       },
    { "find_ext_handler",      DMC_FindExtHandler   },
    { "find_filter",           DMC_FindFilter       },
    { "",                      0                    }
};

/********************************************************************
 * DFIO_TclCmd()
 *
 *
 * Creates a new DFIO instance...
 ********************************************************************/
static int DFIO_TclCmd(
        ClientData        clientData,
        Tcl_Interp       *interp,
        int               argc,
        char            **argv)
{
    Char          *typeName = 0;
    Uint32         i;
    TclInterpObj   i_obj(interp);
    DFIO          *dfio;
    Int32          cmd, argidx;
    TclListObj     list(0);
    DFIOMgr       *dMgr = DFIOMgr::GetDFIOMgr();
    DFIOFilter    *filter;
    Int32          filtRestrict = 0;
    Uint32         newArgc=0;
    Char          *newArgv[64];

    if (argc < 2) {
        Tcl_AppendResult(interp, "too fews args", 0);
        return TCL_ERROR;
    }

    cmd = CmdSwitch(dm_cmds, argv[1]);

    switch (cmd) {

        case DMC_Create:


            if (argc < 3) {
                Tcl_AppendResult(interp, "too fews args", 0);
                return TCL_ERROR;
            }

            newArgv[newArgc++] = "dfio";
            newArgv[newArgc++] = argv[2];

            argidx=3;

            if (argc > 3) {
                typeName = argv[3];
                argidx++;
            }

            for (i=argidx; i<argc; i++) {
                newArgv[newArgc++] = argv[i];
            }

            dfio = DFIOMgr_NewDFIO(typeName, interp, newArgc, newArgv);

            if (!dfio->d_ok) {
                return TCL_ERROR;
            }

            break;

        case DMC_FilterList:
            if (argc > 2) {
                if (String::equal(argv[2], "import")) {
                    filtRestrict = 1;
                } else if (String::equal(argv[2], "export")) {
                    filtRestrict = 2;
                } else {
                    Tcl_AppendResult(interp, "no filter spec ", argv[2], 0);
                    return TCL_ERROR;
                }
            }

            list(interp);
            for (i=0; i<dMgr->dfioFilterList->length(); i++) {
                filter = dMgr->dfioFilterList->idx(i);

                if (filtRestrict == 1) {
                    if (filter->doesImport()) {
                        list << filter->get_instName();
                    }
                } else if (filtRestrict == 2) {
                    if (filter->doesExport()) {
                        list << filter->get_instName();
                    }
                } else {
                    list << filter->get_instName();
                }
            }
            Tcl_SetObjResult(interp, list.end());
            break;

        case DMC_FindExtHandler:
            break;

        case DMC_FindFilter:
            break;

        default:
            Tcl_AppendResult(interp, "unknown DFIO-Mgr sub-cmd ", argv[1], 0);
            return TCL_ERROR;
            break;
    }

    return TCL_OK;
}

/**********************************************************
 * DFIOMgr_Init()
 **********************************************************/
extern "C" int DFIOMgr_Init(Tcl_Interp *interp)
{
    DFIOMgr::GetDFIOMgr();

    Tcl_CreateCommand(interp, "dfio_mgr", 
            (Tcl_CmdProc *)DFIO_TclCmd, 0, 0);
    WidgetMgr_AddType(interp, WIDGET_TYPE_DFIO);

    return TCL_OK;
}




