/****************************************************************************
 *                              ivi_main.cc
 * Author: Matthew Ballance
 * Desc:   Main kickoff point for IVI
 *
 * <Copyright> (c) 2001-2003 Matthew Ballance (mballance@users.sourceforge.net)
 *
 *    This source code is free software; you can redistribute it
 *    and/or modify it in source code form under the terms of the GNU
 *    General Public License as published by the Free Software
 *    Foundation; either version 2 of the License, or (at your option)
 *    any later version.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 *
 * </Copyright>
 *
 ****************************************************************************/
#include "ivi_version.h"
#include "StackTrace.h"
#include "config.h"
#include "TclListObj.h"
#include "ConfigDB.h"
#include "CallbackMgr.h"
#include "IviOpts.h"
#include <tcl.h>
#include <tk.h>
#include <stdio.h>
#include <signal.h>

#ifdef MINGW32
#define WIN32_LEAN_AND_MEAN
#include <windows.h>
#undef  WIN32_LEAN_AND_MEAN
#include <locale.h>
#endif

static char *license_notice1[] = {
"*****************************************************************************\n",
"*                     Icarus Verilog Interactive (IVI)                      *\n",
"*                                                                           *\n",
  ""
};

static char *license_notice2[] = {
"*                                                                           *\n",
"* Copyright (c) 2001-2003 Matthew Ballance (mballance@users.sourcefore.net) *\n",
"*                                                                           *\n",
"*                                                                           *\n",
"*               http://www.sourceforge.net/projects/ivi                     *\n",
"*                                                                           *\n",
"* This program is distributed in the hope that it will be useful,           *\n",
"* but WITHOUT ANY WARRANTY; without even the implied warranty of            *\n",
"* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the             *\n",
"* GNU General Public License for more details.                              *\n",
"*****************************************************************************\n",
  ""
};

static char full_exec_name[65536];

/*************************************************************
 * fatal_error_handler()
 *************************************************************/
static void fatal_error_handler(int signum)
{
    StackTrace(full_exec_name);
}

/********************************************************************
 * DumpStackTrace()
 ********************************************************************/
static void DumpStackTrace(Tcl_Interp *interp, FILE *fp)
{
    fprintf(fp, "----> IviApp::DumpStackTrace()\n");
    fprintf(fp, "%s\n", Tcl_GetVar(interp, "errorInfo", TCL_GLOBAL_ONLY));
    fprintf(fp, "<---- IviApp::DumpStackTrace()\n");
}

/*************************************************************
 *
 *************************************************************/
static IviOpts    *cmdLineOptions = 0;
static char      **global_argv;
static int         global_argc;

/*************************************************************
 * SetOptions()
 *************************************************************/
static int SetOptions(Tcl_Interp *interp)
{
    TclListObj   cmd_line(0), sub_list(0);
    int          i;

    /**** Create a list of all the cmd-line args... ****/
    cmd_line(interp);
    for (i=0; i<global_argc; i++) {
        cmd_line << global_argv[i];
    }
    ConfigDB_SetCurrentObj("App.CmdLineArgs", cmd_line.end()); 

    ConfigDB_SetCurrentObj("App.CmdLineMode", 
            Tcl_NewIntObj(cmdLineOptions->commandLine));

    ConfigDB_SetCurrentObj("App.Console",
            Tcl_NewIntObj(cmdLineOptions->console));

    /**** Now, reflect out the list of dofiles ****/
    cmd_line(interp);
    for (i=0; i<cmdLineOptions->doFiles->length(); i++) {
        cmd_line << cmdLineOptions->doFiles->idx(i)->d_file.value();
    }
    ConfigDB_SetCurrentObj("App.DoFiles", cmd_line.end());

    /**** Reflect the list of ViewFiles ****/
    cmd_line(interp);
    for (i=0; i<cmdLineOptions->viewFiles->length(); i++) {
        char *handler = cmdLineOptions->viewFiles->idx(i)->d_handler.value();
        int   hlen    = cmdLineOptions->viewFiles->idx(i)->d_handler.length();

        sub_list(interp);
        sub_list << ((hlen)?handler:"");
        sub_list << cmdLineOptions->viewFiles->idx(i)->d_file.value();

        cmd_line << sub_list.end();
    }
    ConfigDB_SetCurrentObj("App.ViewFiles", cmd_line.end());

    /**** Next, the list of modules... ****/
    cmd_line(interp);
    for (i=0; i<cmdLineOptions->modules->length(); i++) {
        cmd_line << cmdLineOptions->modules->idx(i)->value();
    }
    ConfigDB_SetCurrentObj("App.VpiModules", cmd_line.end());

    /**** Now, the module paths... ****/
    cmd_line(interp);
    for (i=0; i<cmdLineOptions->modulePath->length(); i++) {
        cmd_line << cmdLineOptions->modulePath->idx(i)->value();
    }
    ConfigDB_SetCurrentObj("App.VpiModulePath", cmd_line.end());

    /**** Next the design-path ****/
    ConfigDB_SetCurrent("App.Design", 
            (cmdLineOptions->design)?cmdLineOptions->design->value():"");

    /**** Now, the logfile ****/
    ConfigDB_SetCurrent("App.LogFile", 
            (cmdLineOptions->logFile)?cmdLineOptions->logFile->value():"");

    /**** Set the CMD_LINE global variable... ***/
    ConfigDB_SetCurrent("App.CmdLine",
            (cmdLineOptions->commandLine)?"1":"0");

    return TCL_OK;
}

/*************************************************************
 * StartMain()
 *************************************************************/
int StartMain(Tcl_Interp *interp)
{
    int i, ret;

    /*******************************************************
     * Call all registration callbacks. 
     *******************************************************/
     CallbackMgr_Invoke("App.Register", NULL, 0, 0);

    /*******************************************************
     * Construct all entities requesting a construction 
     * callback
     *******************************************************/
     CallbackMgr_Invoke("App.Construct", NULL, 0, 0);

    /*******************************************************
     * Load the design
     *******************************************************/
     if (cmdLineOptions->design != 0) {
         char buf[256];
         fprintf(stderr, "Loading design %s\n", 
                 cmdLineOptions->design->value());
         sprintf(buf, "load_design %s", cmdLineOptions->design->value());
         if (Tcl_Eval(interp, buf) != TCL_OK) {
             fprintf(stderr, "Error: Problem loading design: %s\n",
                     Tcl_GetStringResult(interp));
         }
     }

    /*******************************************************
     * Load any view files. Note that this is done before
     * the do-files are loaded. This is to allow the do 
     * files to setup the waveform window
     *******************************************************/
     for (i=0; i<cmdLineOptions->viewFiles->length(); i++) {
         ViewFile *viewFile = cmdLineOptions->viewFiles->idx(i);

         if (viewFile->d_handler.length()) {
             ret = Tcl_VarEval(interp,
                 "sdb open [sdb pick_id] ", viewFile->d_file.value(),
                 " -type ", viewFile->d_handler.value(), 0);
         } else {
             ret = Tcl_VarEval(interp,
                 "sdb open [sdb pick_id] ", viewFile->d_file.value(), 0);
         }

         if (ret != TCL_OK) {
             fprintf(stderr, "ERROR: \"%s\"\n",
               Tcl_GetStringResult(interp));
         }
     }


     for (i=0; i<cmdLineOptions->doFiles->length(); i++) {
         DoFile *dofile = cmdLineOptions->doFiles->idx(i);
         ret = Tcl_VarEval(interp, dofile->d_handler.value(), "_do ", 
                 dofile->d_file.value(), 0);
         if (ret != TCL_OK) {
             fprintf(stderr, "ERROR: \"%s\"\n",
               Tcl_GetStringResult(interp));
         }
     }

     return TCL_OK;
}

extern "C" int IviCommon_Init(Tcl_Interp *interp);
extern "C" int IviStaticPkgs_Init(Tcl_Interp *interp);
extern "C" int IviStaticPkgs_Finish(Tcl_Interp *interp);
/*************************************************************
 * IviApp_Init()
 *************************************************************/
static int IviApp_Init(Tcl_Interp *interp)
{
    char   *ivi_home;
    char    tmp[256];
    int     ret;

    if (cmdLineOptions->needHelp) {
	fprintf(stderr, "NOTE: Initializing IVI to get info on filters\n");
    }

    if (Tcl_Init(interp) == TCL_ERROR) {
        fprintf(stderr, "ERROR: while initializing Tcl (%s)\n",
                Tcl_GetStringResult(interp));
        return TCL_ERROR;
    }

    if (Tk_Init(interp) == TCL_ERROR) {
        fprintf(stderr, "ERROR: while initializing Tk (%s)\n",
                Tcl_GetStringResult(interp));
        return TCL_ERROR;
    }

    Tcl_StaticPackage(interp, "Tk", Tk_Init, Tk_SafeInit);
    Tcl_SetVar(interp, "tcl_rcFileName", "~/.wishrc", TCL_GLOBAL_ONLY);

#ifndef ENABLE_SHARED_BUILD
    if (IviStaticPkgs_Init(interp) != TCL_OK) {
        fprintf(stderr, "FATAL ERROR: Cannot initialize static "
                "packages: %s\n", Tcl_GetStringResult(interp));
        exit(1);
    }
#endif

    if ((ivi_home = getenv("IVI_HOME")) == 0) {
        fprintf(stderr, "FATAL ERROR: environment variable IVI_HOME "
                "isn't set\n");
        exit(1);
    } else {
        fprintf(stderr, "IVI_HOME = %s\n", ivi_home);
    }

    if (IviCommon_Init(interp) != TCL_OK) {
        fprintf(stderr, "ERROR :: Problem initializing IviCommon library\n");
        fprintf(stderr, "\t%s\n", Tcl_GetStringResult(interp));
        DumpStackTrace(interp, stdout);
        exit(1);
    }

    IviStaticPkgs_Finish(interp);

    /****************************************************************
     * Set the options based on the parsed argv/argc
     ****************************************************************/
    if (SetOptions(interp) != TCL_OK) {
        return TCL_ERROR;
    }

    /****************************************************************
     * Load the ui package
     ****************************************************************/
    strcpy(tmp, "set dir [file join $env(IVI_HOME) modules ivi_ui]");
    ret = Tcl_EvalEx(interp, tmp, -1, TCL_EVAL_GLOBAL);
    if (ret != TCL_OK) {
        fprintf(stderr, "FATAL ERROR : command %s\n", tmp);
        fprintf(stderr, "\t%s\n", Tcl_GetStringResult(interp));
        DumpStackTrace(interp, stdout);
        exit(1);
    }

    strcpy(tmp, "source [file join $env(IVI_HOME) "
            "modules ivi_ui pkgIndex.tcl]"); 
    ret = Tcl_EvalEx(interp, tmp, -1, TCL_EVAL_GLOBAL);
    if (ret != TCL_OK) {
        fprintf(stderr, "FATAL ERROR : command \"%s\"\n", tmp);
        fprintf(stderr, "\t%s\n", Tcl_GetStringResult(interp));
        DumpStackTrace(interp, stdout);
        exit(1);
    }

    strcpy(tmp, "IviApp_Init");
    if (Tcl_EvalEx(interp, tmp, -1, TCL_EVAL_GLOBAL) != TCL_OK) {
        fprintf(stderr, "IviApp ERROR: IviApp_Init - %s\n", 
                Tcl_GetStringResult(interp));
        DumpStackTrace(interp, stdout);
        exit(1);
    }

    if (StartMain(interp) != TCL_OK) {
        return TCL_ERROR;
    }

    return TCL_OK;
}

#ifdef MAC_OS_X
	int MacOSXargv(char ***pargv);
#endif

/*************************************************************
 * main()
 *************************************************************/
int main(int argc, char **argv)
{
    Uint32   i = 0, lpad_len, rpad_len, slen;
    Char    *version;
    Char     vbuf[1024];
    char    *ivi_home;

#ifdef MAC_OS_X
	argc= MacOSXargv(&argv);
#endif

    global_argc = argc;
    global_argv = argv;


    Tcl_FindExecutable(global_argv[0]);
    /****************************************************************
     * Find IVI_HOME, based on the full path to the executable...
     ****************************************************************/
    {
        const char *exec_name = Tcl_GetNameOfExecutable();
        char        sep = '/';
        char       *p;
        int         exec_len;
        char       *exist_ivi_home, *ivi_home_env;

        if (!exec_name) {
            fprintf(stderr, "IVI FATAL ERROR: Tcl_GetNameOfExecutable() "
                    "failed\n");
            exit(1);
        }

        if (strchr(exec_name, '\\')) {
            sep = '\\';
        }

        exec_len = strlen(exec_name);
        p = (char *)&exec_name[exec_len-1];

        /**** skip any trailing sep ****/
        while (*p && (*p == sep)) { p--; }

        /**** Scan back through 'ivi' ****/
        while (*p && (*p != sep)) { p--; }

        /**** Skip path-separators ****/
        while (*p && (*p == sep)) { p--; }

        while (*p && (*p != sep)) { p--; }

        while (*p && (*p == sep)) { p--; }

        ivi_home = (char *)malloc((p-exec_name+2));
        ivi_home_env = (char *)malloc((p-exec_name+1)+64);

        memcpy(ivi_home, exec_name, p-exec_name+1);
        ivi_home[p-exec_name+1] = 0;

        sprintf(ivi_home_env, "IVI_HOME=%s", ivi_home);
        if ((exist_ivi_home = getenv("IVI_HOME"))) {
            if (strcasecmp(ivi_home, exist_ivi_home)) {
                fprintf(stderr, "IVI WARNING: IVI_HOME set in environment "
                        "differs from installation of 'ivi'\n");
                fprintf(stderr, "\tenvironment IVI_HOME = %s\n",exist_ivi_home);
                fprintf(stderr, "\tdiscovered  IVI_HOME = %s\n", ivi_home);
            }
        }
        putenv(ivi_home_env);
    }

    /****************************************************************
     * Now that we have IVI_HOME, add IVI_HOME/bin to the path...
     ****************************************************************/
    {
        const char *exist_path = getenv("PATH");
        char *new_path;

        if (!exist_path) {
            exist_path = "";
        }

        new_path = (char *)malloc(strlen(exist_path)+2+
                strlen(ivi_home)+strlen("/bin")+strlen("PATH="));
        sprintf(new_path, "PATH=%s/bin%c%s", ivi_home, 
#ifdef __MINGW32__
        ';',
#else
        ':',
#endif
        exist_path);
        putenv(new_path);
    }

    strcpy(full_exec_name, argv[0]);

#ifndef MINGW32
    signal(SIGABRT, fatal_error_handler);
    signal(SIGBUS, fatal_error_handler);
    signal(SIGSEGV, fatal_error_handler);
#endif

    version = ivi_get_version();
    i=0;
    while (license_notice1[i][0]) {
        printf(license_notice1[i]);
        i++;
    }

    /**** Compute the proper version line...
     **** - 72 chars total.
     **** - 2 chars of '*'
     ****/
    sprintf(vbuf, "Version %s", version);
    slen = strlen(vbuf);

    /**** Total length = slen+rpad+lpad+2
     ****/
    rpad_len = (77-2-slen)/2;
    lpad_len = 77-slen-rpad_len-2;

    printf("*%*s%s%*s*\n", rpad_len, "", vbuf, lpad_len, "");

    i=0;
    while (license_notice2[i][0]) {
        printf(license_notice2[i]);
        i++;
    }
    fflush(stdout);

    /**** On windows, get rid of the '\\' on the command-line... ****/
#ifdef MINGW32
    {
        int    i;
        char  *p;
       
        for (i=0; i<global_argc; i++) {
            for (p=global_argv[i]; *p; p++) {
                if (*p == '\\') {
                    *p = '/';
                }
            }
        }
    }
#endif

    cmdLineOptions = new IviOpts(argc, argv);

    if (1) {
        char *dummy_file = (char *)malloc(strlen(ivi_home)+
                strlen("/modules/ivi_ui/ui_dummy.tcl")+16);
        char *new_argv[3];

        sprintf(dummy_file, "%s/modules/ivi_ui/ui_dummy.tcl", ivi_home);

        new_argv[0] = argv[0];
        new_argv[1] = dummy_file;
        new_argv[2] = 0;

        Tk_Main(2, new_argv, IviApp_Init);
    } else {
        Tk_Main(1, argv, IviApp_Init);
    }
}

#ifdef MINGW32

/*
 *-------------------------------------------------------------------------
 *
 * setargv --
 *
 *	Parse the Windows command line string into argc/argv.  Done here
 *	because we don't trust the builtin argument parser in crt0.  
 *	Windows applications are responsible for breaking their command
 *	line into arguments.
 *
 *	2N backslashes + quote -> N backslashes + begin quoted string
 *	2N + 1 backslashes + quote -> literal
 *	N backslashes + non-quote -> literal
 *	quote + quote in a quoted string -> single quote
 *	quote + quote not in quoted string -> empty string
 *	quote -> begin quoted string
 *
 * Results:
 *	Fills argcPtr with the number of arguments and argvPtr with the
 *	array of arguments.
 *
 * Side effects:
 *	Memory allocated.
 *
 *--------------------------------------------------------------------------
 */

static void setargv(int *argcPtr, char ***argvPtr)
{
    char *cmdLine, *p, *arg, *argSpace;
    char **argv;
    int argc, size, inquote, copy, slashes;
    
    cmdLine = GetCommandLine();	/* INTL: BUG */

    /*
     * Precompute an overly pessimistic guess at the number of arguments
     * in the command line by counting non-space spans.
     */

    size = 2;
    for (p = cmdLine; *p != '\0'; p++) {
	if ((*p == ' ') || (*p == '\t')) {	/* INTL: ISO space. */
	    size++;
	    while ((*p == ' ') || (*p == '\t')) { /* INTL: ISO space. */
		p++;
	    }
	    if (*p == '\0') {
		break;
	    }
	}
    }
    argSpace = (char *) Tcl_Alloc(
	    (unsigned) (size * sizeof(char *) + strlen(cmdLine) + 1));
    argv = (char **) argSpace;
    argSpace += size * sizeof(char *);
    size--;

    p = cmdLine;
    for (argc = 0; argc < size; argc++) {
	argv[argc] = arg = argSpace;
	while ((*p == ' ') || (*p == '\t')) {	/* INTL: ISO space. */
	    p++;
	}
	if (*p == '\0') {
	    break;
	}

	inquote = 0;
	slashes = 0;
	while (1) {
	    copy = 1;
	    while (*p == '\\') {
		slashes++;
		p++;
	    }
	    if (*p == '"') {
		if ((slashes & 1) == 0) {
		    copy = 0;
		    if ((inquote) && (p[1] == '"')) {
			p++;
			copy = 1;
		    } else {
			inquote = !inquote;
		    }
                }
                slashes >>= 1;
            }

            while (slashes) {
		*arg = '\\';
		arg++;
		slashes--;
	    }

	    if ((*p == '\0')
		    || (!inquote && ((*p == ' ') || (*p == '\t')))) { /* INTL: ISO space. */
		break;
	    }
	    if (copy != 0) {
		*arg = *p;
		arg++;
	    }
	    p++;
        }
	*arg = '\0';
	argSpace = arg + 1;
    }
    argv[argc] = NULL;

    *argcPtr = argc;
    *argvPtr = argv;
}

/*************************************************************
 * WinMain()
 *************************************************************/
int APIENTRY WinMain(
        HINSTANCE        hInstance,
        HINSTANCE        hPrevInstance,
        LPSTR            lpszCmdLine,
        int              nCmdShow)
{
    char **argv;
    int    argc, i;
    char   buffer[MAX_PATH+1];
    char  *p;

    setlocale(LC_ALL, "C");
    setargv(&argc, &argv); 

    GetModuleFileName(NULL, buffer, sizeof(buffer));
    argv[0] = buffer;
    for (p=buffer; *p; p++) {
        if (*p == '\\') {
            *p = '/';
        }
    }

    for (i=1; i<argc; i++) {
        for (p=argv[i]; *p; p++) {
            if (*p == '\\') {
                *p = '/';
            }
        }
    }

    return main(argc, argv);
}
#endif /* MINGW32 */

