/****************************************************************************
 *                               ShmDFIO.cc
 *
 * Author: Matthew Ballance
 * Desc:   Provides base functionality for the Reader and Writer classes
 * <Copyright> (c) 2001-2003 Matthew Ballance (mballance@users.sourceforge.net)
 *
 *    This source code is free software; you can redistribute it
 *    and/or modify it in source code form under the terms of the GNU
 *    General Public License as published by the Free Software
 *    Foundation; either version 2 of the License, or (at your option)
 *    any later version.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 *
 * </Copyright>
 ****************************************************************************/
#include "ShmDFIO.h"
#include "ShmDFIOWriterInfo.h"
#include "ShmDFIOReaderInfo.h"
#include "DFIOMgr.h"
#include <unistd.h>
#include <stdio.h>

#undef DEBUG_SHM_DFIO

#define FP stderr

#ifdef DEBUG_SHM_DFIO
#define DBG_MSG(x) fprintf x
#else
#define DBG_MSG(x)
#endif


/********************************************************************
 * ShmDFIO()
 ********************************************************************/
ShmDFIO::ShmDFIO(
        Tcl_Interp        *interp,
        Uint32             argc,
        Char             **argv) : DFIO(interp, argc, argv)
{
    const char *filename = 0;

#ifndef __MINGW32__
    d_pagesize = getpagesize();
#else
//    d_pagesize = 4096;
    d_pagesize = 65536;
#endif

    d_header   = 0;
    d_file     = 0;

    for (Uint32 i=2; i<argc; i++) {
        if (!strcmp(argv[i], "-file")) {
            filename = argv[++i];
        }
        else {
            fprintf(stderr, "Unknown option \"%s\"\n", argv[i]);
            return;
        }
    }

    if (!filename) {
        fprintf(stderr, "WARNING: ShmDFIOWriter not given filename\n");
        filename = "ivi.shm";
    }

    d_filename = strdup(filename);
}

/********************************************************************
 * ~ShmDFIO()
 ********************************************************************/
ShmDFIO::~ShmDFIO()
{
    if (d_file) {
        delete d_file;
    }
}

/********************************************************************
 * Remap()
 ********************************************************************/
int ShmDFIO::Remap(Uint32 size)
{
    char *oldMapBase;

    if (size % d_pagesize) {
        DBG_MSG((stderr, "increasing size from %d to %d\n",
                    size, (size+(d_pagesize-(size%d_pagesize)))));
        size += d_pagesize - (size % d_pagesize);
    }

    oldMapBase = d_file->getMapBase();

    d_file->map(size);
    d_header = (ShmDFIOHeader *)d_file->getMapBase();

    d_header->d_mappedSize = size;

    return 0;
}

/********************************************************************
 * Ptr2ShmPtr()
 ********************************************************************/
ShmDFIOPtr ShmDFIO::Ptr2ShmPtr(char *ptr)
{
    char *base = d_file->getMapBase();
    return (ShmDFIOPtr)(ptr-base);
}

/********************************************************************
 * ShmPtr2Ptr()
 ********************************************************************/
char *ShmDFIO::ShmPtr2Ptr(ShmDFIOPtr ptr)
{
    char *base = d_file->getMapBase();
    return (char *)(base+ptr);
}

/********************************************************************
 * getTimeUnit()
 ********************************************************************/
Int32 ShmDFIO::getTimeUnit()
{
    UpdateMapping();

    if (d_header) {
        return d_header->d_timeUnit;
    } else {
        fprintf(stderr, "ERROR: d_header invalid\n");
    }
}

/********************************************************************
 * setTimeUnit()
 ********************************************************************/
void ShmDFIO::setTimeUnit(Int32 unit)
{
    base_setTimeUnit(unit);

    if (!d_header) {
        fprintf(stderr, "ERROR: d_header invalid in setTimeUnit()\n");
    }

    d_header->d_timeUnit = unit;
}

/********************************************************************
 * Malloc()
 ********************************************************************/
char *ShmDFIO::Malloc(Uint32 size)
{
    char *ret;

    DBG_MSG((FP, "----> ShmDFIOWriter::Malloc(%d)\n", size));

    DBG_MSG((FP, "\td_header=%x ; usedSize = %d\n", d_header, 
                d_header->d_usedSize));

    if ((d_header->d_usedSize+size) >= d_header->d_mappedSize) {
        Remap(d_header->d_usedSize+size);
    }

    DBG_MSG((FP, "\td_header=%x ; usedSize = %d\n", d_header, 
                d_header->d_usedSize));

    if (d_header->d_usedSize % 4) {
        d_header->d_usedSize += (4-(d_header->d_usedSize%4));
    }

    ret = ((char *)d_header) + d_header->d_usedSize;
    d_header->d_usedSize += size;


    DBG_MSG((FP, "<---- ShmDFIOWriter::Malloc(%d) => %x\n", size, ret));
    return ret;
}


/********************************************************************
 * Shm_dfio_Init()
 ********************************************************************/
extern "C" int Shm_dfio_Init(Tcl_Interp *interp)
{
    ShmDFIOWriterInfo    *wInfo;
    ShmDFIOReaderInfo    *rInfo;

    if (Tcl_InitStubs(interp, "8.1", 0) == NULL) {
        return TCL_ERROR;
    }
    Tcl_PkgProvide(interp, "shm_dfio", "1.0");

    wInfo = new ShmDFIOWriterInfo();
    rInfo = new ShmDFIOReaderInfo();

    DFIOMgr_RegisterDFIO(wInfo);
    DFIOMgr_RegisterDFIO(rInfo);


    return TCL_OK;
}

/********************************************************************
 * Shm_dfio_SafeInit()
 ********************************************************************/
extern "C" int Shm_dfio_SafeInit(Tcl_Interp *interp)
{
    return Shm_dfio_Init(interp);
}

