/****************************************************************************
 *                             StubTrace.cc
 *
 * Author: Matthew Ballance
 * Desc:
 *
 *
 * <Copyright> (c) 2001-2003 Matthew Ballance (mballance@users.sourceforge.net)
 *
 *    This source code is free software; you can redistribute it
 *    and/or modify it in source code form under the terms of the GNU
 *    General Public License as published by the Free Software
 *    Foundation; either version 2 of the License, or (at your option)
 *    any later version.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 *
 * </Copyright>
 *
 ****************************************************************************/
#include "StubRwTrace.h"

#undef  DEBUG_MODULE

#define FP stderr
#ifdef DEBUG_MODULE
#define DBG_MSG(x) fprintf x 
#else
#define DBG_MSG(x)
#endif

#define BLK_SIZE    1024
// #define BLK_SIZE 4096
// #define BLK_SIZE    65536

/**********************************************************
 * StubTrace()
 **********************************************************/
StubTrace::StubTrace(
    Char     *name,
    Uint32    rows,
    Uint32    msb,
    Uint32    lsb,
    Uint32    flags) : DFIOTrace(name, rows, msb, lsb, flags) 
{
    valChgBlocks    = new Vector<ValChgBlock>();
    currValChgBlock = new ValChgBlock(BLK_SIZE, len);
    currValChgBlock->startTime = 0;
    currValChgBlock->endTime   = 0;
    valChgBlocks->append(currValChgBlock);
    d_traceTime = 0;
    initial = 1;
}


/**********************************************************
 * setValueBitStr()
 *
 * When ending/starting block:
 *   - Know end-time of block.
 *   - Set start-time of block to last_end_time+1
 **********************************************************/
Int32 StubTrace::setValueBitStr(Uint32 row, Char *bitStr)
{
    Uint32         i, slen = strlen(bitStr);
    DFIOValChg    *chg;

    if (!currValChgBlock->currentChg) {
        currValChgBlock->startTime = parent->currentTime;
    }

    chg = DFIOValChg_Idx(currValChgBlock->valChanges, len,
            currValChgBlock->currentChg);

    /**** As long as the GlobalTime for this DFIO stays the same, 
     **** don't create any new transitions...  In the case
     **** of the initial data-entry (which may occur at >0 time),
     **** ensure that no spurious transition is created...
     ****/
    if ((chg->changeTime != parent->currentTime) && !initial) {
        currValChgBlock->currentChg++;
        if (currValChgBlock->currentChg >= (currValChgBlock->numChanges)) {
            currValChgBlock->currentChg--;
            currValChgBlock = new ValChgBlock(BLK_SIZE, len);
            valChgBlocks->append(currValChgBlock);
        }
        chg = DFIOValChg_Idx(currValChgBlock->valChanges, len,
                currValChgBlock->currentChg);
    }

    if (len > slen) {
        for (i=0; i<(len-slen); i++) {
            DFIOValChg_SetBit(chg, i, DFIOBitVal_0);
        }
        for (i=(len-slen); i<len; i++) {
            char bv = bitStr[i-(len-slen)];
            DFIOValChg_SetBit(chg, i, 
                    (bv=='0')?DFIOBitVal_0:
                    (bv=='1')?DFIOBitVal_1:
                    (bv=='Z'||bv=='z')?DFIOBitVal_Z:DFIOBitVal_X);
        }
    } else {
        for (i=0; i<len; i++) {
            char bv = bitStr[len-i-1];
            DFIOValChg_SetBit(chg, i, 
                    (bv=='0')?DFIOBitVal_0:
                    (bv=='1')?DFIOBitVal_1:
                    (bv=='Z'||bv=='z')?DFIOBitVal_Z:DFIOBitVal_X);
        }
    }

    chg->changeTime = parent->currentTime;
    currValChgBlock->endTime = parent->currentTime;
    initial = 0;

    return 1;
}

/**********************************************************
 * findBlock()
 * Find the smallest block that has a timestamp larger
 * (or equal to) the starting time.
 **********************************************************/
Uint32 StubTrace::findBlock(Uint32   reqTime)
{
    Int32           ub, lb, len;
    Int32           checkPt, i;
    Int32           tmp;
    ValChgBlock    *pt, *pt2;

    lb = 0;
    ub = valChgBlocks->length()-1;
    checkPt = (ub-lb)/2;

    len = valChgBlocks->length();
    for (i=0; i<len; i++) {
        DBG_MSG((FP, "\tBlock %d ranges %d - %d\n", i, 
                valChgBlocks->idx(i)->startTime, 
                valChgBlocks->idx(i)->endTime));
    }

    do {
        DBG_MSG((FP, "lb = %d ; ub = %d ; checkPt = %d\n", 
                lb, ub, checkPt));
        pt = valChgBlocks->idx(lb);
        DBG_MSG((FP, "\tlb: %d - %d\n", pt->startTime, pt->endTime));
        pt = valChgBlocks->idx(ub);
        DBG_MSG((FP, "\tub: %d - %d\n", pt->startTime, pt->endTime));

        if (!(pt = valChgBlocks->idx(checkPt))) {
            DBG_MSG((FP, "ERROR: no block @ %d\n", checkPt));
            break;
        }

        if (reqTime > pt->endTime) {
            DBG_MSG((FP, "reqTime %d > endTime %d\n", reqTime, pt->endTime));

            /**** First, see if the next block has a start-time >= the
             **** reqTime...
             ****/
            if ((pt2 = valChgBlocks->idx(checkPt+1))) {
                if (pt2->startTime >= reqTime) {
                    return checkPt+1;
                }
            }

            lb = checkPt;
            tmp = (lb+(ub-lb)/2);

            if (tmp == checkPt) {
                checkPt++;
            } else {
                checkPt = (lb+(ub-lb)/2);
            }
        } else if (reqTime < pt->startTime) {
            DBG_MSG((FP, "reqTime %d < startTime %d\n", reqTime, 
                    pt->startTime));
            ub = checkPt;
            tmp = lb+(ub-lb)/2;

            if (checkPt == tmp) {
                if (checkPt) {
                    checkPt--;
                } else {
                    break;
                }
            } else {
                checkPt = tmp;
            }
        } else {
            DBG_MSG((FP, "else: reqTime %d inside block %d - %d\n",
                    reqTime, pt->startTime, pt->endTime));
            return checkPt;
        }
        DBG_MSG((FP, "lb = %d ; ub = %d ; checkPt = %d\n", 
                lb, ub, checkPt));
    } while ((lb >= 0) && (lb <= ub) && (ub >= 0) && (checkPt < len));

    /*** If we under- or over- flowed, fixup the return... ***/
    if (checkPt >= len) {
        checkPt = len-1;
    } else if (checkPt < 0) {
        checkPt = 0;
    }

    DBG_MSG((FP, "NOTE: fall-through...\n"));

    return checkPt;
}

/**********************************************************
 * getValue()
 *
 * Returns the first value with time <= begin. Vector 
 * extends to the last value <= end
 **********************************************************/
Vector<DFIOValChg> *StubTrace::getValue(
        Uint32         begin, 
        Uint32         end,
        Uint32         flags)
{
    ValChgBlkVector      *ret;
    Uint32                done = 0, startTime = begin, idx, x;
    ValChgBlock          *blk, *tmp_blk; 
    Int32                 i;


    ret = new ValChgBlkVector(len);

    idx = findBlock(startTime);

    if (!(blk = valChgBlocks->idx(idx))) {
        return ret;
    }

//    if ((blk->valChanges[0].changeTime > startTime)) {
    if (DFIOValChg_Idx(blk->valChanges, len, 0)->changeTime > startTime) {

        if (idx) {
            idx--;
        } else {
            /**** See if the end-time of this block somehow falls into
             **** the begin/end range...
             ****/
            if (DFIOValChg_Idx(blk->valChanges, len, 0)->changeTime > end) {
                return ret;
            } else {
                startTime = DFIOValChg_Idx(blk->valChanges, len, 0)->changeTime;
            }
        }
    }

   /**** Find the actual beginning of the data-of-interest
    **** within this block
    **** - Work from the end. Find the first val_chg entry
    ****   that has a timestamp <= the starting time
    ****/
    if (!(blk = valChgBlocks->idx(idx))) {
        goto srw_exit;
    }

    DBG_MSG((FP, "NOTE: Init block ranges %d - %d\n",
            blk->valChanges[0].changeTime,
            blk->valChanges[blk->currentChg].changeTime));
  
    /**** NOTE: Need a much more efficient way of doing
     **** this... Probably should do a bin-search 
     **** here...
     ****/
    for (i=(blk->currentChg); i>=0; i--) {
        if (DFIOValChg_Idx(blk->valChanges, len, i)->changeTime <= startTime) {
            DBG_MSG((FP, "Setting init_idx = %d\n", i));
            DBG_MSG((FP, "\tstartTime=%d ; changeTime=%d\n",
                startTime, blk->valChanges[i].changeTime));
            if ((flags & GVF_OneMinus) && (i > 0)) { 
                i--;
            }
            ret->set_init_idx(i);
            break;
        }
    }
    ret->append(blk);
    idx++;

    /**** Now, append blocks until the time-limit is reached
     ****/
    do {
        if (!(blk = valChgBlocks->idx(idx))) {
            blk = valChgBlocks->idx(idx-1);
            DBG_MSG((FP, "\tStop due to null block\n"));
            goto srw_exit;
        }

        if (blk->startTime <= end) {
            ret->append(blk); 
            idx++;
        } else if (blk->endTime > end) {
            DBG_MSG((FP, "\tStop due to endTime %d >= end %d\n",
                    blk->endTime, end));
            done = 1;
            if (idx) {
                blk = valChgBlocks->idx(--idx);
            }
        }
    } while (!done);

srw_exit:
    /*** Find the value-change that is <= the requested end-time
     ***/
    if (blk) {
        DBG_MSG((FP, "NOTE: final block ranges %d - %d\n",
                blk->startTime, blk->endTime));
        for (i=(blk->currentChg); i>=0; i--) {
            if (DFIOValChg_Idx(blk->valChanges, len, i)->changeTime <= end) {
                DBG_MSG((FP, "NOTE: @ idx %d, changeTime %d <= end %d\n",
                        i, blk->valChanges[i].changeTime, end));
                if (flags & GVF_OnePlus && (i < (blk->currentChg))) {
                    i++;
                }
                break;
            }
        }

        if (i >= 0) {
           DBG_MSG((FP, "Decrementing vector length by %d (length is %d)\n",
                blk->currentChg-i, ret->length()));
            ret->decLen(blk->currentChg-i);
        } else if (!blk->currentChg) {
            ret->incLen(1);
            DBG_MSG((FP, "NOTE: Not decrementing...\n"));
        }
    }
    return ret;
}

/********************************************************************
 * clearTrace()
 ********************************************************************/
void StubTrace::clearTrace()
{
    Uint32 i;

    for (i=0; i<valChgBlocks->length(); i++) {
        /**** Delete each block ****/
        delete valChgBlocks->idx(i);
    }

    valChgBlocks->empty();

    currValChgBlock = new ValChgBlock(BLK_SIZE, len);
    currValChgBlock->startTime = 0;
    currValChgBlock->endTime   = 0;
    valChgBlocks->append(currValChgBlock);
    initial = 1;
    d_traceTime = 0;
}

