#*****************************************************************************
#*                           WaveWindow.tcl
#* Author: Matthew Ballance
#* Desc:   Implements the signal-display widget. Incorporates the wave_widget
#*         and associated widgets...
#*
#*
#* <Copyright> (c) 2001-2003 Matthew Ballance (mballance@users.sourceforge.net)
#*
#*    This source code is free software; you can redistribute it
#*    and/or modify it in source code form under the terms of the GNU
#*    General Public License as published by the Free Software
#*    Foundation; either version 2 of the License, or (at your option)
#*    any later version.
#*
#*    This program is distributed in the hope that it will be useful,
#*    but WITHOUT ANY WARRANTY; without even the implied warranty of
#*    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#*    GNU General Public License for more details.
#*
#*    You should have received a copy of the GNU General Public License
#*    along with this program; if not, write to the Free Software
#*    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
#*
#* </Copyright>
#*
#*****************************************************************************
namespace eval WaveWindow {

    variable configspec {
        {-sdbr        ""}
        {-font        "-*-arial-medium-r-bold-*-*-*-*-*-*-*"}
        {-dflt_cursor "left_ptr"}
        {-move_cursor "hand2"}
        {-sb_width     12    }
        {-pady         2     }
    }
}

#        SigPrefDialog
#        select
#        Zoom
#        SaveSetup

#*********************************************************
#* InitWidgetRec
#*********************************************************
proc WaveWindow::InitWidgetRec {w} {
    upvar #0 $w data

    set data(first_sig) ""
    set data(last_sig)  ""
    set data(zoom_list) ""

    set data(cursor_tolerance) 5

    set data(drag_busy) 0
    set data(drag_list) ""

    set data(orig_cursor) ""

    set data(val_time) 0

    set data(text_fg)  white
}

#*********************************************************
#* yscroll_command
#*********************************************************
proc WaveWindow::yscroll_command args {
    upvar #0 [lindex $args 0] data

    eval $data(w:waveVScroll) set [lrange $args 1 end]
}

#*********************************************************
#* xscroll_command
#*********************************************************
proc WaveWindow::xscroll_command args {
    upvar #0 [lindex $args 0] data

    $data(w:waveHScroll) set [lindex $args 1] [lindex $args 2]
}

#********************************************************************
#* PrintValidate
#********************************************************************
proc WaveWindow::PrintValidate {w} {
    upvar #0 $w data

    after 1 "WaveWindow::ValidatePrintData $w true"
    return true
}
#********************************************************************
#* PrintFileValidate
#********************************************************************
proc WaveWindow::PrintFileValidate {w} {

    after 1 "WaveWindow::ValidatePrintData $w false"
    return true
}

#********************************************************************
#* PrintUpdate
#********************************************************************
proc WaveWindow::PrintUpdate {w} {
    upvar #0 $w data

    set dlg "$w.print"
    set pm  "$w.print_pm"

    set start_time [[$dlg subwidget start_time] get]
    set end_time   [[$dlg subwidget end_time] get]
    set time_per_page [[$dlg subwidget time_per_page] get]

    if {$start_time == "" || $end_time == "" || $time_per_page == ""} {
        return true
    }

    set paper_size [lindex [$pm paper_sizes] \
            [[$dlg subwidget paperSize] getvalue]]
    $pm configure -paper_size [lindex $paper_size 0]

    set paper_orient [lindex {landscape portrait} \
        [[$dlg subwidget paperOrient] getvalue]]
    $pm configure -orientation $paper_orient

    set limits [$data(w:wave_widget) time_bounds]

    #*** Find pix_per_time based on page size and time/page
    catch {set pix_per_time [$data(w:wave_widget) \
            pix_per_time $pm $time_per_page]} res

    set pages [$data(w:wave_widget) page_layout \
        $pm $start_time $end_time $pix_per_time]

    set out_dim [$dlg subwidget out_dim]
    $out_dim config -text "[lindex $pages 0]x[lindex $pages 1]"

    return true
}

#********************************************************************
#* SetEntryColors
#********************************************************************
proc WaveWindow::SetEntryColors { e good } {
    if {$good} {
        $e config -bg white -fg black
    } else {
        set val [$e get]
        if {$val == ""} {
            set bg "red"
        } else {
            set bg "white"
        }

        $e config -bg $bg -fg red
    }
}

#********************************************************************
#* ValidatePrintData
#********************************************************************
proc WaveWindow::ValidatePrintData {w do_update} {
    upvar #0 $w data

    set dlg $w.print
    set disabled 0

    set start_time_w [$dlg subwidget start_time]
    set start_time [$start_time_w get]
    if {[string is integer -strict $start_time]} {
        SetEntryColors $start_time_w true
    } else {
        SetEntryColors $start_time_w false
        incr disabled
    }

    set end_time_w [$dlg subwidget end_time]
    set end_time [$end_time_w get]
    if {[string is integer -strict $end_time]} {
        SetEntryColors $end_time_w true
    } else {
        SetEntryColors $end_time_w false
        incr disabled
    }

    set timepp_w [$dlg subwidget time_per_page]
    set timepp [$timepp_w get]
    if {[string is integer -strict $timepp]} {
        SetEntryColors $timepp_w true
    } else {
        SetEntryColors $timepp_w false
        incr disabled
    }

    set file_w [$dlg subwidget file]
    set file [$file_w get]
    if {$file == ""} {
        SetEntryColors $file_w false
        incr disabled
    } else {
        SetEntryColors $file_w true
    }

    set ok [$dlg subwidget ok]
    if {$disabled} {
        $ok configure -state disabled
    } else {
        $ok configure -state normal
        if {$do_update} {
            PrintUpdate $w
        }
    }
}

#********************************************************************
#* Print
#********************************************************************
proc WaveWindow::Print {w} {
    upvar #0 $w data

    set pm [ps_pixmap $w.print_pm]

    $pm margins 35 35 35 35

    set psizes ""
    foreach ps [$pm paper_sizes] {
        lappend psizes [lindex $ps 0]
    }

    set dlg [DialogMgr::create WaveWidget::Print $w.print \
        -var [list w $w]]

    set start_time [$dlg subwidget start_time]
    set end_time   [$dlg subwidget end_time]

    set limits [$data(w:wave_widget) time_bounds]

    $start_time delete 0 end
    $start_time insert 0 [lindex $limits 0]
    $end_time delete 0 end
    $end_time insert 0 [lindex $limits 1]

    set time_per_page [$dlg subwidget time_per_page] 
    $time_per_page delete 0 end
    $time_per_page insert 0 \
        [$data(w:wave_widget) time_per_page \
            $pm [$data(w:wave_widget) pix_per_time]]

    set psc [$dlg subwidget paperSize]
    $psc configure -values $psizes
    $psc setvalue first

    set pso [$dlg subwidget paperOrient]
#    $pso configure -values {Landscape Portrait}
    $pso configure -values {Landscape}
    $pso setvalue first

    foreach c {paperSize paperOrient} {
        set c [$dlg subwidget $c]
	$c configure -modifycmd "WaveWindow::PrintUpdate $w"
    }

    set file_e [$dlg subwidget file]

    $file_e delete 0 end
    $file_e insert end "print.ps"

    foreach e {start_time end_time time_per_page} {
        set e [$dlg subwidget $e]
        $e config -validate key -validatecommand "WaveWindow::PrintValidate $w"
    }
    [$dlg subwidget file] config -validate key \
        -validatecommand "WaveWindow::PrintFileValidate $w"

    PrintUpdate $w


    $dlg popup

    if {[$dlg wait] != ""} {
        $pm configure -file [$file_e get]
        $pm paper_sizes

        $data(w:wave_widget) print $pm 0 [lindex $limits 1]

        destroy $dlg 
    }

    $pm delete
}

#********************************************************************
#* PrintFileBrowse
#********************************************************************
proc WaveWindow::PrintFileBrowse {w} {
    upvar #0 $w data

    set fileTypes ""
    set filename [tk_getSaveFile -filetypes $fileTypes]

    if {$filename != ""} {
        set file [$w.print subwidget file]
        $file delete 0 end
        $file insert end $filename
    }
}

#*********************************************************
#* CursorControl
#*********************************************************
proc WaveWindow::CursorControl {w cmd} {
    upvar #0 $w data

    switch $cmd {
        add {
            set c [$data(-sdbr) add_cursor]
            set wave_time [$data(w:wave_widget) get_limits]
            set mid [expr ([lindex $wave_time 1] - [lindex $wave_time 0]) / 2 \
                + [lindex $wave_time 0]]
            $c set_time $mid
            $c selected 1
        }

        del {
            set clist [$data(-sdbr) get_cursor_list]
            foreach c $clist {
                if [$c selected] {
                    $c delete
                }
            }
        }

        prev_edge - 
        next_edge {
            set hlist $data(w:sigWinSigList)
            set sel [$hlist selection get]
            if {$sel == ""} {
                set sig [lindex [$data(-sdbr) list] 0]
            } else {
                set sig [lindex $sel 0]
            }

            if {$sig != ""} {
                foreach c [$data(-sdbr) get_cursor_list] {
                    if [$c selected] {

                        if {$cmd == "next_edge"} {
                            $c next_edge $sig
                        } else {
                            $c prev_edge $sig
                        }

                        set wave_time [$data(w:wave_widget) get_limits]

                        set ctime [$c get_time]

                        if {$ctime < [lindex $wave_time 0]} {
                            set delta [expr [lindex $wave_time 1] - \
                                            [lindex $wave_time 0]]
                            set llimit [expr $ctime - (($delta * 3)/4)]
                            set rlimit [expr $ctime + (($delta * 1)/4)]

                            if {$llimit < 0} {
                                set rlimit [expr $rlimit - $llimit]
                                set llimit 0
                            }
                            $data(w:wave_widget) set_limits $llimit $rlimit
                        } elseif {$ctime > [lindex $wave_time 1]} {
                            set delta [expr [lindex $wave_time 1] - \
                                            [lindex $wave_time 0]]

                            set llimit [expr $ctime - (($delta * 1)/4)]
                            set rlimit [expr $ctime + (($delta * 3)/4)]

                            if {$llimit < 0} {
                                set rlimit [expr $rlimit - $llimit]
                                set llimit 0
                            }
                            $data(w:wave_widget) set_limits $llimit $rlimit
                        }
                    }
                }
            }
        }
    }
}

#*********************************************************
#* CursorMoveCB 
#*********************************************************
proc WaveWindow::CursorMoveCB {w t} {
    upvar #0 $w data

    SetValList $w $t
    set data(val_time) $t
}

#*********************************************************
#*
#*********************************************************
proc WaveWindow::SetCursorMode {w mode} {
    upvar #0 $w data

    set select_mode [$data(w:toolbar) subwidget select_mode]
    set zoom_mode   [$data(w:toolbar) subwidget zoom_mode]

    if {$mode == "select" && $data(currSelMode) != "select"} {
        $select_mode configure -relief sunken
        $zoom_mode configure -relief link
        $data(w:wave_widget) configure -cursor_mode "select" 

        if {[catch {$data(w:wave_widget) configure -cursor2 ""} res]} {
            puts "configure failed: $res"
        }
        set data(currSelMode) "select"
    }

    if {$mode == "zoom_mode" && $data(currSelMode) != "zoom_mode"} {
        $select_mode configure -relief link
        $zoom_mode configure -relief sunken
        $data(w:wave_widget) configure -cursor_mode "zoom"

        if {[catch {$data(w:wave_widget) configure -cursor2 sizing} res]} {
            puts "configure failed: $res"
        }
        set data(currSelMode) "zoom_mode"
    }
}

#*********************************************************
#* ConstructToolbar
#*********************************************************
proc WaveWindow::ConstructToolbar {w p} {
    upvar #0 $w data
    global IVI_HOME
    global CurrWaveWindow

    set CurrWaveWindow $w

    uplevel #0 {set ImgPath [list [file join $IVI_HOME modules wave_widget] \
                                  [file join $IVI_HOME modules ivi_ui]]}

#    uplevel #0 {set ImgPath [file join $IVI_HOME modules ivi_ui]}
    set toolbar [MenuMgr::create toolbar WaveWidget::Toolbar $p \
        -var [list w $w]]

    set select_mode [$toolbar subwidget select_mode]

    $select_mode configure -relief sunken 

    return $toolbar
}

#*********************************************************
#* GetSchemeValue
#*********************************************************
proc WaveWindow::GetSchemeValue {w value} {
    set scheme [IviConfig::current WaveWidget.ColorScheme]
    set scheme "WaveWidget.ColorSchemes.$scheme"

    return [IviConfig::current $scheme.$value]
}

#*********************************************************
#* ConstructSigTree
#*********************************************************
proc WaveWindow::ConstructSigTree {w p} {
    upvar #0 $w data
    global ivi_global

    set top_f [frame $p.frame]
    set tree_f [frame $top_f.frame]
    set tree [tree_widget $tree_f.tree \
        -opencmd  "after 1 WaveWindow::SigBrowse $w open"  \
        -closecmd "after 1 WaveWindow::SigBrowse $w close" \
        -selectmode extended -movedropcmd "WaveWindow::move_done $w"]

    set hlist $tree
    $hlist config -yscrollcommand "WaveWindow::SetScroll $w" -width 16

    set bg [GetSchemeValue $w siglist_bg]
    set fg [GetSchemeValue $w siglist_fg]

    if {$bg == ""} {
        set bg grey50
    }

    if {$fg == ""} {
        set fg "white"
    }

    $hlist config -background $bg -foreground $fg

    set data(w:sigl_pad) [frame $tree_f.pad_label -borderwidth 2]
    $data(w:sigl_pad) config -width 128 -height 2 -relief flat -bg $bg
    

    pack $data(w:sigl_pad) -side bottom  -expand no -fill x
    pack $tree  -side bottom -expand yes -fill both

    set hsb [scrollbar $top_f.hsb -orient horizontal]
    $hsb config -command "$hlist xview"
    $hlist config -xscrollcommand "$hsb set"

    pack $hsb -side bottom -expand no -fill x
    pack $tree_f -side bottom -expand yes -fill both

    pack $top_f -expand yes -fill both

    set data(w:sigWinSigList) $tree
}

#*********************************************************
#* ConstructWaveWin
#*********************************************************
proc WaveWindow::ConstructWaveWin {w p} {
    upvar #0 $w data
    global ivi_global
    global tcl_platform

    #***************************************************************
    #*** WaveWidget Pane
    #***************************************************************
    set top [frame $p]

    set pw [panedwindow $top.pw -orient horizontal]
    set data(w:outer_pane) $pw

    set data(w:val_list_pane) [frame $pw.val_list_pane]
    $pw add $data(w:val_list_pane) -width 120

    set data(w:wave_win_pane) [frame $pw.wave_win_pane]
    $pw add $data(w:wave_win_pane)

    #**** Create value list
    set list_tf [frame $data(w:val_list_pane).tf]
    set hlist $data(w:sigWinSigList)

    set bg [GetSchemeValue $w siglist_bg]
    set fg [GetSchemeValue $w siglist_fg]

    if {$bg == ""} {
        set bg grey50
    }

    if {$fg == ""} {
        set fg "white"
    }

    set data(w:val_list) [tree_widget $list_tf.listbox \
        -pady 1 -background $bg -foreground $fg \
        -selectmode disabled]

    set list_bf [frame $list_tf.bf]
    set pl [frame $list_bf.pl -borderwidth 2 -relief flat -bg $bg]
    $pl config -width 64 -height 18 

    set data(w:val_list_sb)    [scrollbar $list_bf.hsb \
        -orient horizontal]
    $data(w:val_list_sb) config -command "$data(w:val_list) xview"
    $data(w:val_list) config -xscrollcommand "$data(w:val_list_sb) set"

    pack $data(w:val_list_sb) -side bottom -expand no -fill x
    pack $pl -side bottom -fill x
    set data(w:val_list_pad) $pl
    pack $list_bf -side bottom -expand no -fill both
    pack $list_tf -expand yes -fill both

    if {$tcl_platform(platform) == "windows"} {
        set cursor "arrow"
    } else {
        set cursor "left_ptr"
    }

    set data(w:wave_widget)    [wave_widget $data(w:wave_win_pane).wave \
        -sdbr $data(-sdbr) \
        -xscrollcommand "WaveWindow::xscroll_command $w" \
        -cursor_move_cmd "WaveWindow::CursorMoveCB $w"   \
        -select_box_cmd "WaveWindow::SelBoxCmd $w"       \
        -sig_tree        $data(w:sigWinSigList)          \
        -sigval_tree     $data(w:val_list)               \
        -active_cursor   $cursor                         \
        -color_scheme    \
         "WaveWidget.ColorSchemes.[IviConfig::current WaveWidget.ColorScheme]"]

    set data(w:waveHScroll)    [scrollbar $data(w:wave_win_pane).hsb \
        -orient horizontal -command "WaveWindow::HScroll $w" ]

    set vsb_frame              [frame $data(w:wave_win_pane).vsb_frame]
    set data(w:waveVScroll)    [scrollbar $vsb_frame.vsb \
        -orient vertical -command "WaveWindow::VScroll $w" ]
    set data(w:vsb_pad)        [frame $vsb_frame.vsb_pad \
        -relief groove -borderwidth 2]
    
    pack $data(w:vsb_pad) -side bottom -expand no -fill x
    pack $data(w:waveVScroll) -side bottom -expand yes -fill y

    pack $data(w:val_list) -expand yes -fill both
    pack $pw -expand yes -fill both


    grid $data(w:wave_widget) $vsb_frame -sticky news

    grid $data(w:waveHScroll) -sticky news
    grid columnconfigure $data(w:wave_win_pane) 0 -weight 1
    grid rowconfigure $data(w:wave_win_pane) 0 -weight 1

    return $top
}

#********************************************************************
#* PadCB
#*
#* This proc is called when the geometry of the window changes, or
#* when a cursor is added to or deleted from the wave window
#*
#* The task of this proc is to update the height of the pad elements
#* at the bottom of the window to match the height of the cursor
#* pane
#********************************************************************
proc WaveWindow::PadCB {w} {
    upvar #0 $w data

    set c_h [$data(w:wave_widget) cget -cursorpane_height]

    $data(w:sigl_pad) config -height $c_h
    $data(w:val_list_pad) config -height $c_h
    $data(w:vsb_pad) config -height $c_h
}

#*********************************************************
#* WaveWindow
#*********************************************************
proc WaveWindow::WaveWindow {w args} {
    global CallbackTypes
    variable configspec
    global   IVI_HOME

    set w [frame $w]

    bind $w <Destroy> "WaveWindow::Destroy $w"

    array set $w {_dummy _dummy}
    upvar #0 $w data

    set data(ww_dir) [file join $IVI_HOME modules wave_widget]

    foreach cs $configspec {
        set data([lindex $cs 0]) [lindex $cs 1]
    }

    InitWidgetRec $w

    eval BaseClass::configure $w $args

    set data(sdbr_cb) [callback add $CallbackTypes(SDBR_SIGLIST_UPDATE) \
        $data(-sdbr) [list WaveWindow::update_sigpanes $w] ]

    set data(w:toolbar)     [ConstructToolbar $w $w.toolbar]

    set data(w:waveWinPane) [panedwindow $w.wave]

    set data(w:waveWinSigList) [frame $data(w:waveWinPane).siglist]
    $data(w:waveWinPane) add $data(w:waveWinSigList) -width 120

    set data(w:waveWinSigDisp) [frame $data(w:waveWinPane).sigdisp]
    $data(w:waveWinPane) add $data(w:waveWinSigDisp)


    #***************************************************************
    #*** SigList Pane
    #***************************************************************
    ConstructSigTree $w $data(w:waveWinSigList)

    set data(w:SigTopFrame)    [ConstructWaveWin $w \
        $data(w:waveWinSigDisp).top_frame]

    pack $data(w:SigTopFrame) -expand yes -fill both -side top

    bind $data(w:wave_widget) <Configure> "WaveWindow::PadCB $w"
    set data(cursor_cb) [callback add \
        $CallbackTypes(CBTYPE_CURSOR_LIST_UPDATE) $data(-sdbr) \
        "WaveWindow::PadCB $w"]

    UpdateSigTree $w

    pack $data(w:toolbar)       -side top -expand no -fill x
    pack $data(w:sigWinSigList) -expand yes -fill both     
    
    pack $data(w:waveWinPane) -expand yes -fill both

    set $w.sl_popup [MenuMgr::create popup WaveWidget::SigListPopup \
        $w.sl_popup -var [list w $w]]
    set $w.d_popup  [MenuMgr::create popup WaveWidget::WavePopup \
        $w.d_popup -var [list w $w]]

    SetBindings $w

    $data(w:wave_widget) configure -trace_height \
        [$data(w:val_list) cget -lineheight] -pady $data(-pady)

    $data(w:sigWinSigList) configure -pady $data(-pady)
    $data(w:val_list) configure -pady $data(-pady)

    rename $w ::$w:cmd
    proc ::$w { cmd args } "return \[eval WaveWindow::cmd $w \$cmd \$args\]"

    set data(currSelMode) ""

    return $w
}

#*********************************************************
#* Destroy
#*********************************************************
proc WaveWindow::Destroy {w} {
    upvar #0 $w data

    callback destroy $data(cursor_cb)
}

#*********************************************************
#* cmd
#*********************************************************
proc WaveWindow::cmd {path cmd args} {
    upvar #0 $path data
    set args [lrange $args 0 end]

    switch $cmd {
        config -
        configure {
            eval BaseClass::configure $path $args
        }

        cget {
            eval BaseClass::cget $path $args
        }

        subwidget {
            set val [array get data "w:[lindex $args 0]"]
            if {$val != ""} {
                return [lindex $val 1]
            } else {
                error "no subwidget \"[lindex $args 0]\" of WaveWindow"
            }
        }


        SigPrefDialog {
            SigPrefDialog $path
        }

        SaveSetup {
            SaveSetup $path [lindex $args 0] [lindex $args 1]
        }

        Zoom {
            eval Zoom $path [lindex $args 0] [lindex $args 1]
        }

        select {
            eval select $path $args
        }
    }
}

#*********************************************************
#* SelBoxCmd
#*********************************************************
proc WaveWindow::SelBoxCmd { w xi yi xf yf } {
    upvar #0 $w data

    set exist [$data(w:wave_widget) get_limits]
    WaveWindow::AddZoom $w [lindex $exist 0] [lindex $exist 1]

    $data(w:wave_widget) set_limits \
        [$data(w:wave_widget) pixels_to_time $xi] \
        [$data(w:wave_widget) pixels_to_time $xf]
}

#*********************************************************
#* SigBrowse
#*********************************************************
proc WaveWindow::SigBrowse { w action path } {
    upvar #0 $w data

    set tree $data(w:sigWinSigList)

    if {$action == "open"} {
        $path config -expanded 1
    } else {
        $path config -expanded 0
    }
}

#*********************************************************
#* AddCursor
#*********************************************************
proc WaveWindow::AddCursor {w menu} {
    upvar #0 $w data

    foreach c [$data(-sdbr) get_cursor_list] {
        $c selected 0
    }

    set cursor [$data(-sdbr) add_cursor]
    $cursor set_time [$data(w:wave_widget) pixels_to_time [$menu x]]

    $cursor selected 1
}

#********************************************************************
#* LockSelected
#********************************************************************
proc WaveWindow::LockSelected {w} {
    upvar #0 $w data

    foreach cur [$data(-sdbr) get_cursor_list] {
        if {[$cur selected]} {
            $cur lock 1
        }
    }
}

#********************************************************************
#* UnlockSelected
#********************************************************************
proc WaveWindow::UnlockSelected {w} {
    upvar #0 $w data

    foreach cur [$data(-sdbr) get_cursor_list] {
        if {[$cur selected]} {
            $cur lock 0
        }
    }
}

#*********************************************************
#* AddZoom
#*
#* Adds a zoom to the history-list of zooms
#*********************************************************
proc WaveWindow::AddZoom {w start end} {
    upvar #0 $w data

    set zlist [list $start $end]

    lappend new_zlist $zlist

    foreach zm $data(zoom_list) {
        lappend new_zlist $zm
    }

    set data(zoom_list) $new_zlist

    set data(cursor_tolerance) \
        [$data(w:wave_widget) pixels_to_time -no_offset 20]
}

#*********************************************************
#* Zoom
#*********************************************************
proc WaveWindow::Zoom args {
    upvar #0 [lindex $args 0] data
    set w [lindex $args 0]

    set cmd [lindex $args 1]
    if {$cmd == "full"} {
        set exist [$data(w:wave_widget) get_limits]
        WaveWindow::AddZoom $w [lindex $exist 0] [lindex $exist 1]

        set bounds [$data(w:wave_widget) time_bounds]
        $data(w:wave_widget) set_limits [lindex $bounds 0] [lindex $bounds 1]
    } elseif {$cmd == "in"} {
        WaveWindow::ZoomIn $w
    } elseif {$cmd == "out"} {
        WaveWindow::ZoomOut $w
    } elseif {$cmd == "last"} {
        WaveWindow::ZoomLast $w
    } elseif {$cmd == "gui_range"} {
        set limits [$data(w:wave_widget) get_limits]
        set diag [DialogMgr::create WaveWidget::ZoomRange $w.zoom_range \
           -var [list start [lindex $limits 0] end [lindex $limits 1]]]

        $diag center [winfo toplevel $w]
        $diag popup

        if {[$diag wait] != ""} {
            set from [$diag.vbox0.f.vbox0.f.hbox0.f.from get_text]
            set to   [$diag.vbox0.f.vbox0.f.hbox1.f.to   get_text]
            $data(w:wave_widget) set_limits $from $to
            destroy $diag
        }
    } elseif {$cmd == "get"} {
        set exist [$data(w:wave_widget) get_limits]
        return $exist
    } elseif {$cmd == "range_quiet"} {
        eval $data(w:wave_widget) set_limits [lrange $args 2 end]
    } else {
        set exist [$data(w:wave_widget) get_limits]
        eval WaveWindow::AddZoom $w $exist
        eval $data(w:wave_widget) set_limits [lrange $args 1 end]
    }
}

#*********************************************************
#* SaveSetup
#*********************************************************
proc WaveWindow::SaveSetup { w fp win_name args} {
    upvar #0 $w data
    global CallbackTypes
    set    save_cursors 1

    foreach arg $args {
        if {$arg == "-no_cursors"} {
            set save_cursors 0
        }
    }

    if {$win_name != "" && $win_name != " "} {
        set win "-win $win_name "
    } else {
        set win ""
    }

    if {[llength [sdb_mgr list]] > 1} {
        set do_sdb 1
    } else {
        set do_sdb 0
    }

    puts $fp "callback disable $CallbackTypes(SDBR_SIGLIST_UPDATE) \
        \[wave get_sdbr $win\]"
    puts $fp "wave zoom [$data(w:wave_widget) get_limits] $win"

    #*** Needed info:
    #*** -radix
    #*** -visel
    #*** -sig_dispname
    #*** -expanded

    foreach sig [$data(-sdbr) list] {
        if {[$sig cget -separator]} {
            set dname [$sig cget -sig_dispname]
            regsub {^---- } $dname {} dname
            regsub { ----$} $dname {} dname

            puts $fp "wave add_sep $win \"$dname\""
        } else {
            set fullname [$sig cget -sig_fullname]
            if {$do_sdb} {
                set sdb_str "-sdb [$sig cget -sdb] "
            } else {
                set sdb_str ""
            }
            puts $fp "set _sig \[wave add $sdb_str$win$fullname -radix [$sig cget -radix] -visel [$sig cget -visel] -sig_dispname \"[$sig cget -sig_dispname]\" -expanded [$sig cget -expanded]\]"
            if {[$sig cget -has_enum] == 1} {
                puts $fp "\$_sig set_enum [$sig get_enum]"
            }
        }
    }

    if {$save_cursors} {
        foreach cur [$data(-sdbr) get_cursor_list] {
            set line "set c \[wave add_cursor $win\]"
            puts $fp $line
            puts $fp "\$c set_time [$cur get_time]"
            if {[$cur lock]} {
                puts $fp "\$c lock 1"
            }

            if {[$cur selected]} {
                puts $fp "\$c selected 1"
            }
        }
    }

    puts $fp "callback enable $CallbackTypes(SDBR_SIGLIST_UPDATE) \
        \[wave get_sdbr $win\]"
    puts $fp "callback invoke $CallbackTypes(SDBR_SIGLIST_UPDATE) \
        \[wave get_sdbr $win\]"
}

#*********************************************************
#* ZoomLast
#*********************************************************
proc WaveWindow::ZoomLast {w} {
    upvar #0 $w data

    set zlist [lindex $data(zoom_list) 0]
    if {$zlist != ""} {
        $data(w:wave_widget) set_limits \
            [lindex $zlist 0] [lindex $zlist 1] 
        set data(zoom_list) [lrange $data(zoom_list) 1 end]
    }
}

#*********************************************************
#* ZoomIn
#*********************************************************
proc WaveWindow::ZoomIn {w} {
    upvar #0 $w data

    set exist [$data(w:wave_widget) get_limits]
    WaveWindow::AddZoom $w [lindex $exist 0] [lindex $exist 1]

    set limits [$data(w:wave_widget) get_limits]

    set delta [expr [lindex $limits 1] - [lindex $limits 0]]

    set llimit [expr [lindex $limits 0] + ($delta/4)]
    set rlimit [expr $llimit + ($delta/2)]

    $data(w:wave_widget) set_limits $llimit $rlimit
}

#*********************************************************
#* ZoomOut
#*********************************************************
proc WaveWindow::ZoomOut {w} {
    upvar #0 $w data

    set exist [$data(w:wave_widget) get_limits]
    WaveWindow::AddZoom $w [lindex $exist 0] [lindex $exist 1]

    set limits [$data(w:wave_widget) get_limits]

    set delta [expr [lindex $limits 1] - [lindex $limits 0]]

    if {[lindex $limits 0] >= [expr $delta/2]} {
        set llimit [expr [lindex $limits 0] - ($delta/2)]
    } else {
        set llimit 0
    }
    set rlimit [expr $llimit + ($delta*2)]

    $data(w:wave_widget) set_limits $llimit $rlimit
}


#*********************************************************
#* SigPrefDialog
#*
#* Construct a preferences dialog. When closed, update
#* the config of signals
#*********************************************************
proc WaveWindow::SigPrefDialog {w} {
    upvar #0 $w data
    global CallbackTypes

    set diag [DialogMgr::create WaveWidget::SigPrefs $w.sigprefs]
    $diag popup

    set sigs [$data(w:sigWinSigList) selection get]

    if {[$diag wait] != ""} {

        set visel [[$diag subwidget path_el] get_text]

        callback disable $CallbackTypes(SDBR_SIGLIST_UPDATE) $data(-sdbr)
        foreach sig $sigs {
           $sig config -visel $visel
        }

        callback enable $CallbackTypes(SDBR_SIGLIST_UPDATE) $data(-sdbr)
        callback invoke $CallbackTypes(SDBR_SIGLIST_UPDATE) $data(-sdbr)

        destroy $diag
    }
}

#*********************************************************
#* CursorMouseAction
#* 
#*********************************************************
proc WaveWindow::CursorMouseAction {w x y mode} {
    upvar #0 $w data

    if {[$data(w:wave_widget) cget -cursor_mode] == "zoom"} {
        return
    }

    if {[llength [$data(-sdbr) get_cursor_list]] > 0} {
        set b_time [$data(w:wave_widget) pixels_to_time $x]
        set cursor [$data(-sdbr) get_cursor $b_time $data(cursor_tolerance)]

        if {$mode == "mod"} {
            if {$cursor != ""} {
                if {[$cursor selected] == 0} {
                    $cursor selected 1
                } else {
                    $cursor selected 0
                }
            }
        } else {
            set clist [$data(-sdbr) get_cursor_list]
            foreach c $clist {
                $c selected 0
            }

            if {$cursor != ""} {
                $cursor selected 1
            }
        }
    } else {
        #**** Add a new cursor...
        set cursor [$data(-sdbr) add_cursor]
        $cursor set_time [$data(w:wave_widget) pixels_to_time $x]
        $cursor selected 1
    }
}

#*********************************************************
#* AddSepCmd
#*********************************************************
proc WaveWindow::AddSepCmd {w menu} {
    upvar #0 $w data

    #***
    #*** Open a dialog to ask for name...
    #***

    set diag [DialogMgr::create WaveWindow::SepName $w.sepdlg]
    $diag withdraw
    $diag center $diag
    $diag popup

    if {[$diag wait] != ""} {
        set hlist $data(w:sigWinSigList)
        set near  [$hlist nearest [$menu y]]

        set label "---- [[$diag subwidget sep_name] get_text] ----"
        set sep [$data(-sdbr) add_separator $label]

        destroy $diag

        set s_index [$sep index]
        if {$near != ""} {
            if {[regexp {!.*$} $near]} {
                regsub {!.*$} $near {} near
            } 
            set s_dest  [$near index]
            $data(-sdbr) move $s_index $s_dest
        }
    }
}

#*********************************************************
#* SetBindings
#*********************************************************
proc WaveWindow::SetBindings {w} {
    upvar #0 $w data
 
    set tree $data(w:sigWinSigList)
    set wave_widget $data(w:wave_widget)
    bind $tree <Control-a>   "WaveWindow::select $w 0 end"
    bind $tree <Control-u>   "WaveWindow::select $w"

    $w.sl_popup bind $tree <<RMB-Popup>>
    $w.d_popup  bind $wave_widget <<RMB-Popup>>

    bind $wave_widget <minus> "WaveWindow::ZoomOut $w"
    bind $wave_widget <equal> "WaveWindow::ZoomIn $w"
    bind $wave_widget <plus>  "WaveWindow::ZoomIn $w"

#    bind $wave_widget <<Cursor-Add-Select>> \
#       "WaveWindow::CursorMouseAction $w %x %y mod"
#    bind $wave_widget <Button-1> \
#       "WaveWindow::CursorMouseAction $w %x %y single"

    bind $wave_widget <Enter> "focus $wave_widget"
}

#*********************************************************
#* VScroll
#*********************************************************
proc WaveWindow::VScroll args {
    upvar #0 [lindex $args 0] data
    set tree $data(w:sigWinSigList)
    eval $tree yview [lrange $args 1 end]
    eval $data(w:val_list) yview [lrange $args 1 end]
}

#*********************************************************
#* HScroll
#*********************************************************
proc WaveWindow::HScroll args {
    upvar #0 [lindex $args 0] data

    if {[llength $args] > 2 } {
        eval $data(w:wave_widget) xview [lrange $args 1 end] 
    }
}

#*********************************************************
#* SetScroll
#*********************************************************
proc WaveWindow::SetScroll args {
    upvar #0 [lindex $args 0] data

#    puts "WaveWindow::SetScroll $args"

    eval $data(w:waveVScroll) set [lrange $args 1 end]

    eval $data(w:wave_widget) yview moveto [lindex $args 1]
}

#*********************************************************
#* update_sigpanes
#*********************************************************
proc WaveWindow::update_sigpanes {w} {
    upvar #0 $w data
    set hlist $data(w:sigWinSigList)

    WaveWindow::UpdateSigTree $w
    WaveWindow::UpdateValList $w
}

#*********************************************************
#* SetRadix
#*********************************************************
proc WaveWindow::SetRadix {w radix} {
    upvar #0 $w data
    global CallbackTypes
    set sig_list ""
    set tree $data(w:sigWinSigList)

    # Don't recognize selected sub-signals
    foreach sig [$tree selection get] {
        if {[regexp {@.*@.+} $sig] == 0} {
            lappend sig_list $sig
        }
    }
   
    callback disable $CallbackTypes(SDBR_SIGLIST_UPDATE) $data(-sdbr)
    foreach sig $sig_list {
        $sig config -radix $radix
    }
    callback enable $CallbackTypes(SDBR_SIGLIST_UPDATE) $data(-sdbr)
    callback invoke $CallbackTypes(SDBR_SIGLIST_UPDATE) $data(-sdbr)
}

#*********************************************************
#* select
#*********************************************************
proc WaveWindow::select args {
    upvar #0 [lindex $args 0] data


    set hlist $data(w:sigWinSigList)
    set len [llength $args]
    if {$len > 1} {
        set from [lindex $args 1]
        set to   [lindex $args 2]
        if {$data(first_sig) != ""} {
            $hlist selection set $data(first_sig) $data(last_sig)
        }
    } else {
        $hlist selection clear
    }
}

#*********************************************************
#* move
#*********************************************************
proc WaveWindow::move {w} {
    upvar #0 $w data

    $data(w:sigWinSigList) start_move
}

#*********************************************************
#* move_done
#*********************************************************
proc WaveWindow::move_done {w put_idx} {
    upvar #0 $w data
    global CallbackTypes

    callback disable $CallbackTypes(SDBR_SIGLIST_UPDATE) $data(-sdbr)
    set sel_list_i [$data(w:sigWinSigList) selection get]

    set sel_list ""
    foreach item $sel_list_i {
        if {![regexp {[^!]*![^!]*} $item]} {
            lappend sel_list $item
        }
    }

    eval $data(-sdbr) append_cut_buffer $sel_list

    if {[regexp {[^!]*![^!]*} $put_idx]} {
        regsub {!.*$} $put_idx {} put_idx

        set sig_list [$data(-sdbr) list]

        set idx [lsearch $sig_list $put_idx]

        incr idx

        set put_idx [lindex $sig_list $idx]

        if {$put_idx == ""} {
            set put_idx "end"
        }
    }


    $data(-sdbr) put_cut_buffer $put_idx

    callback enable $CallbackTypes(SDBR_SIGLIST_UPDATE) $data(-sdbr)
    callback invoke $CallbackTypes(SDBR_SIGLIST_UPDATE) $data(-sdbr)
}

#*********************************************************
#* delete
#*********************************************************
proc WaveWindow::delete {w} {
    upvar #0 $w data
    global CallbackTypes

    callback disable $CallbackTypes(SDBR_SIGLIST_UPDATE) $data(-sdbr)

    set hlist $data(w:sigWinSigList)
    set selected [$hlist selection get]

    foreach sig $selected {
        $sig delete
    }

    callback enable $CallbackTypes(SDBR_SIGLIST_UPDATE) $data(-sdbr)
    callback invoke $CallbackTypes(SDBR_SIGLIST_UPDATE) $data(-sdbr)
}

#*********************************************************
#* SetValList
#*********************************************************
proc WaveWindow::SetValList {w t} {
    upvar #0 $w data

    set vlist $data(w:val_list) 

    $vlist config -redraw false

    foreach child [$vlist info children] {
        if {[$child cget -separator]} {
            set val_list "--------"
            set val $val_list
        } else {
            set val_list [$child get_value $t]

            if {[regexp {No Data} $val_list]} {
                set val $val_list
            } else {
                set val [bitvector [lindex $val_list 0] "-[$child cget -radix]"]
            }
        }

        set t_val [lindex $val 0]
        if {$t_val == ""} {
            set t_val "--"
        }

        $vlist itemconfig $child -text $t_val

        if {[$vlist itemcget $child -open]} {
            foreach bit [$vlist info children $child] {
                set val_list [lrange $val_list 1 end]
                set t_val [lindex $val_list 0]
                if {$t_val == ""} {
                    set t_val "--"
                }
                $vlist itemconfig $bit -text $t_val
            }
        }
    }

    $vlist config -redraw true
}

#*********************************************************
#* UpdateValList
#*********************************************************
proc WaveWindow::UpdateValList {w} {
    upvar #0 $w data

    set t 0
    set hlist $data(w:sigWinSigList)
    set vlist $data(w:val_list) 

    $vlist config -pady [$hlist cget -pady] -drawbranch 0 

    $vlist delete root

    foreach child [$hlist info children] {
        $vlist insert end root $child -text "--" -drawcross never

        if {[$hlist itemcget $child -open]} {
            $vlist itemconfig $child -open true
            foreach bit [$hlist info children $child] {
                $vlist insert end $child $bit -text "--"
            }
        }
    }

    $data(w:wave_widget) update_sigvals
#    WaveWindow::SetValList $w $data(val_time)
}

#*********************************************************
#* UpdateSigTree
#*********************************************************
proc WaveWindow::UpdateSigTree {w} {
    upvar #0 $w data
    set max_len 0

    set hlist $data(w:sigWinSigList)

    set sel_list [$hlist selection get]
    $hlist configure -redraw false
    $hlist delete root

    if {$data(-sdbr) != ""} { 
        set sig_list [$data(-sdbr) list]
        set data(first_sig) [lindex $sig_list 0]
        set data(last_sig)  [lindex $sig_list end]
        foreach sig [$data(-sdbr) list] {
            set msb [$sig cget -msb]
            set lsb [$sig cget -lsb]
            set name [$sig cget -sig_dispname]
            if {$msb != $lsb} {
                set name "$name\[$msb:$lsb\]"
            }

            if {[$sig cget -separator] == 1} {
                $hlist insert end root $sig -text $name -drawbranch 0
            } else {
                if {$max_len < [string length $name]} {
                    set max_len [string length $name]
                }
                $hlist insert end root $sig -text $name -drawbranch 0
            }
            if {[lsearch $sel_list $sig] > -1} {
                $hlist selection add $sig
            }

            if {[$sig cget -expanded]} {
                $hlist itemconfig "$sig" -open true
            } else {
                $hlist itemconfig "$sig" -open false
            }

            if {$msb > $lsb} {
                for {set i $msb} {$i >= $lsb} {incr i -1} {
                    set child [$hlist insert end $sig "$sig!$i" -text $i]
                    if {[lsearch $sel_list "$sig!$i"] > -1} {
                        $hlist selection add "$sig!$i"
                    }
                }
            } elseif {$msb < $lsb} {
                for {set i $msb} {$i <= $lsb} {incr i} {
                    set child [$hlist insert end $sig "$sig!$i" -text $i]
                    if {[lsearch $sel_list "$sig!$i"] > -1} {
                        $hlist selection add "$sig!$i"
                    }
                }
            }
        }
    }

    $hlist configure -redraw true
}

