
/*************************************************************************
 *
 * All portions are copyrighted by their respective authors.
 *
 * This file is part of IVMan (ivm).
 *
 * This file may be distributed under the terms of the Q Public License
 * as defined by Troll Tech AS of Norway and appearing in the file
 * LICENSE.QPL included in the packaging of this file.
 * 
 * See http://www.troll.no/qpl for QPL licensing information.
 *
 * $Id: manager.c,v 1.72 2006/02/10 01:20:51 ro_han Exp $
 ************************************************************************/

#define _GNU_SOURCE
#include <stdio.h>
#include <unistd.h>
#include <signal.h>
#include <string.h>
#include <stdlib.h>

#include <fcntl.h>
#include <sys/ioctl.h>
#include <sys/types.h>
#include <dirent.h>
#include <mntent.h>
#include <errno.h>
#include <assert.h>

#include <linux/cdrom.h>
#include <linux/sysctl.h>

#include <glib.h>
#include <dbus/dbus.h>
#include <dbus/dbus-glib.h>
#include <libhal.h>


#include "common.h"
#include "hal_interface.h"
#include "hal_compat.h"
#include "manager.h"
#include "debug.h"
#include "IvmConfig/IvmConfigProperties.h"
#include "IvmConfig/IvmConfigActions.h"
#include "hal_psi.h"
#include "log.h"
#include "daemonize.h"

#include "userconfigs.h"

gchar *homeDir;

/// Command to be used for mounting when desired mount point exists, as passed to ivm_run_command() .
char *known_mount_command;

/// Command to be used for mounting when desired mount point does not exist, as passed to ivm_run_command() .
char *unknown_mount_command;

/// Command to be used for unmounting, as passed to ivm_run_command() .
char *umount_command;

/// Command-line option: run in 'root mode' when not root.
gboolean systemOpt = FALSE;

/// Command-line option: don't fork.
gboolean noForkOpt = FALSE;

/// Command-line option: output debugging info.
gboolean debugOpt = FALSE;

/// Command-line option: configuration directory.
gchar *confdirOpt = NULL;


/**
 * Get the correct configuration directory.
 * @return	Directory containing configuration files for this instance of Ivman.
 */
char * get_confdir()
{
    GString * configDir = NULL;

    if (usermode) {
        configDir = g_string_new(homeDir);
        configDir = g_string_append(configDir, "/.ivman/");
    }

    else {
        configDir = g_string_new(SYSCONFDIR);
        configDir = g_string_append(configDir, "/ivman/");
    }

    char * ret = configDir->str;
    g_string_free(configDir, FALSE);
    return ret;
}

void signal_handler(int sig)
{
	DEBUG( _("Got signal %d"), sig );
//    switch (sig) {
//    case SIGTERM:
//    case SIGINT:
//    case SIGHUP:
        // Quit the main loop
        g_main_loop_quit( loop );
//        break;
//    }
}

void xmlReadFailed(char const * filename) {
    assert(filename);
    DEBUG(_("Could not parse %s!\n"
    "Please check this file for errors - read 'man %s' for some tips "
    "(e.g., how to use special characters such as <,> and &). "
    "Execution will continue, but rules in %s may not be executed."),
    filename, filename, filename);
}


/**
 * Test if the given command succeeds.
 * It is very important that the command can't possibly hang.
 * @param   command The command to execute.  Executed by /bin/sh, so
 *                  doesn't have to be a full path if in default PATH.
 * @return  TRUE if command is executed and has exit status 0,
 *          FALSE otherwise.
 */
gboolean command_succeeds( char * command ) {
    /*
     * This method does:
     *  1) put together argv[] for execution with /bin/sh
     *  2) execute, using glib function
     *  3) check exit status
     */
    assert( command );

    /* 1) put together argv[] for execution with /bin/sh */
    gchar *argv[4];

    argv[0] = "/bin/sh";
    argv[1] = "-c";
    argv[2] = command;
    argv[3] = 0;
    gint exitStatus;
    GError *error = 0;
    /* end part 1) */


    /* 2) execute, using glib function */
    gboolean success = g_spawn_sync( NULL, argv, NULL, 0, NULL, NULL, NULL,
                                     NULL, &exitStatus, &error );
    /* end part 2) */

    /* 3) check exit status */
    if ( !success ) {
        DEBUG(_("Error while trying to run '%s': %s\n"),
                  argv[2], error->message);
        g_error_free(error);
        return FALSE;
    }
    if ( error ) g_error_free( error );

    if ( WIFEXITED(exitStatus) && ( WEXITSTATUS(exitStatus) == 0) ) {
        return TRUE;
    }

    return FALSE;
}


/**
 * Test if the given command exists.
 * It is very important that the command can't possibly hang.
 * @param   command The command to execute.  Executed by /bin/sh, so
 *                  doesn't have to be a full path if in default PATH.
 * @return  TRUE if command can be found and executed,
 *          FALSE otherwise.
 */
gboolean command_exists( char * command ) {
    /*
     * This method does:
     *  1) put together argv[] for execution with /bin/sh
     *  2) execute, using glib function
     *  3) check exit status
     */
    assert( command );

    /* 1) put together argv[] for execution with /bin/sh */
    gchar *argv[4];

    argv[0] = "/bin/sh";
    argv[1] = "-c";
    argv[2] = command;
    argv[3] = 0;
    gint exitStatus;
    GError *error = 0;
    /* end part 1) */


    /* 2) execute, using glib function */
    gboolean success = g_spawn_sync( NULL, argv, NULL, 0, NULL, 
                                     NULL, NULL, NULL, &exitStatus, &error );
    /* end part 2) */

    /* 3) check exit status */
    if ( !success ) {
        DEBUG(_("Error while trying to run '%s': %s\n"),
                  argv[2], error->message);
        g_error_free(error);
        return FALSE;
    }
    if ( error ) g_error_free( error );

    /* 127 is the error code returned by /bin/sh for command not found. */
    if (WEXITSTATUS(exitStatus) == 127) {
        return FALSE;
    }

    return TRUE;
}



/**
 * Find the commands used to mount and unmount volumes on this system.
 */
void set_mount_command()
{
    known_mount_command = NULL;
    unknown_mount_command = NULL;
    umount_command = NULL;
    // If command used to mount is preset, just return it
    if ( cfg_base->mountcommand
         && cfg_base->umountcommand ) {
        known_mount_command = cfg_base->mountcommand;
        unknown_mount_command = known_mount_command;
        umount_command = cfg_base->umountcommand;
        return;
    }

    if ( cfg_base->mountcommand 
         || cfg_base->umountcommand )
    {
        DEBUG(_
              ("Only one of 'mountcommand', 'umountcommand' was specified "
               "in IvmConfigBase.xml.  Both must be specified for the "
               "options to take effect.\n"));
    }

    DEBUG(_
          ("No mount command was specified in IvmConfigBase.xml.  Ivman "
           "will try to automatically detect the command to use. "
           "If Ivman incorrectly detects the program(s) available on your "
           "system, first make sure the program(s) are in the default "
           "shell PATH, then please report it as a bug.\n"));

    // OK, now try to find a mount program.
    // mount programs in order of preference:
    // - pmount-hal (for user-mode Ivman only)
    // - pmount
    // - mount

    // Test if pmount-hal exists
    if ( usermode ) {
        if ( !command_exists( "pmount-hal &> /dev/null" ) ) {
            DEBUG(_("pmount-hal was not found on your system.\n"));
        }
        else {
            if ( !command_exists( "pumount &> /dev/null" ) ) {
                DEBUG(_("pmount-hal was found on your system, but pumount "
                        "wasn't!\n"));
            }
            else {
                DEBUG(_
                      ("pmount-hal was found on your system. It will be "
                       "used for mounting.\n"));
                known_mount_command = "pmount-hal '$hal.info.udi$'";
                unknown_mount_command = known_mount_command;
                umount_command = "pumount '$hal.block.device$'";
                return;
            }
        }
    }
    else {
        DEBUG(_
              ("pmount-hal detection skipped, as we are a root instance "
               "of Ivman.  pmount-hal is only used for user instances.\n"));
    }


    // Now try looking for regular pmount.
    if ( !command_exists( "pmount &> /dev/null" ) ) {
        DEBUG(_("pmount was not found on your system.\n"));
    }
    else {
        if ( !command_exists( "pumount &> /dev/null" ) ) {
            DEBUG(_
                  ("pmount was found on your system, but pumount "
                   "wasn't!\n"));
        }

        else {
            // If not root, fine; if root, make sure -u exists.
            if ( usermode ) {
                DEBUG(_
                      ("pmount was found on your system. It will be used "
                       "for mounting.\n"));
                known_mount_command =
                    "pmount '$hal.block.device$' "
                    "'$hal.volume.policy.desired_mount_point$'";
                unknown_mount_command = "pmount '$hal.block.device$'";
                umount_command = "pumount '$hal.block.device$'";
                return;
            }


            GString *pmount_umask_check = g_string_new( "pmount" );
            pmount_umask_check =
                g_string_append( pmount_umask_check, " --help | " );
            pmount_umask_check =
                g_string_append( pmount_umask_check, GREP_PATH
                                " -- '-u <' &> /dev/null");

            gboolean umask_accepted =
                command_succeeds( pmount_umask_check->str );
            g_string_free(pmount_umask_check, TRUE);

            if ( umask_accepted ) {
                DEBUG(_("pmount accepts -u <umask>\n"));
                DEBUG(_
                      ("pmount was found on your system. It will be used "
                       "for mounting.\n"));

                GString *command = g_string_new("pmount -u ");
                g_string_append(command, cfg_base->umask);
                g_string_append(command, " '$hal.block.device$' "
                                "'$hal.volume.policy.desired_mount_point$'");
                known_mount_command = command->str;
                g_string_free(command, FALSE);

                command = g_string_new("pmount -u ");
                g_string_append(command, cfg_base->umask);
                g_string_append(command, " '$hal.block.device$'");
                unknown_mount_command = command->str;
                g_string_free(command, FALSE);

                umount_command = "pumount '$hal.block.device$'";
                return;
            }


            DEBUG(_("pmount does not accept -u <umask>"));

            DEBUG(_
                  ("WARNING: pmount does not accept -u <umask>, so I will "
                   "not mount volumes with it (since they would only be "
                   "accessible by the %s user account, which is probably "
                   "not what you want). Upgrade pmount.\n"),
                  cfg_base->user);
        }
    }


    // At the bottom of the barrel is plain old mount ...
    if ( !command_exists( "mount &> /dev/null" ) ) {
        DEBUG(_("mount was not found on your system.\n"));
    }

    else {
        if ( !command_exists( "umount &> /dev/null" ) ) {
            DEBUG(_
                  ("mount was found on your system, but umount was "
                   "not!\n"));
        }
        else {
            DEBUG(_
                  ("mount was found on your system. It will be used for "
                   "mounting.  This means you need fstab entries for the "
                   "devices you want mounted, or you need to use "
                   "fstab-sync. It is highly recommended you get pmount "
                   "instead.\n"));

            known_mount_command = "mount '$hal.block.device$'";
            unknown_mount_command = known_mount_command;
            umount_command = "umount '$hal.block.device$'";
            return;
        }
    }


    // Couldn't find anything !!!
    return;

}



/**
 * Find the full path of a desired configuration file.
 * @param filename	Name of the desired configuration file.
 */
char *ivm_get_config_file(char const *const filename)
{

    GString *configFile;

    configFile = g_string_new(confdirOpt);
    configFile = g_string_append(configFile, filename);

    char *ret = configFile->str;

    g_string_free(configFile, FALSE);
    return ret;

}



// when running in user mode, tests for config files, and creates them
// if they do not already exist.
void ivm_test_configs()
{

    FILE *file;
    GString *filename;
    char *content;

    DIR *dir = opendir(confdirOpt);

    if (dir == NULL)
    {
        DEBUG(_("Settings directory does not exist, attempting "
                "to create it...\n"));

        // make the necessary directory if it does not exist
        GString *command = g_string_new(MKDIR_PATH);

        command = g_string_append(command, " -p ");
        command = g_string_append(command, confdirOpt);
        ivm_run_command(command->str, NULL, FALSE);
        g_string_free(command, TRUE);

        // wait 5 seconds for the directory to be created
        int i = 0;

        dir = opendir(confdirOpt);
        while (dir == NULL && i < 5)
        {
            sleep(1);
            dir = opendir(confdirOpt);
            i++;
        }

        if ((void *) opendir(confdirOpt) == NULL)
        {
            DEBUG(_("Couldn't create new directory %s\n"), confdirOpt);
            exit(1);
        }
        else
            closedir(dir);
    }

    else
        closedir(dir);

    char *filenames[] = {
        "IvmConfigActions.xml", "IvmConfigBase.xml",
        "IvmConfigProperties.xml",
        "IvmConfigConditions.xml", NULL
    };

    char *contents[] = {
        IVMCONFIGACTIONS, IVMCONFIGBASE,
        IVMCONFIGPROPERTIES, IVMCONFIGCONDITIONS, NULL
    };

    for (int i = 0; filenames[i] != NULL; i++)
    {
        filename = g_string_new(confdirOpt);
        filename = g_string_append(filename, filenames[i]);

        file = fopen(filename->str, "r");

        if ( !file )
        {
            DEBUG(_
                  ("Configuration file %s not found, creating one "
                   "with default content...\n"), filename->str);

            file = fopen(filename->str, "w");

            if ( file ) {
                content = contents[i];
                fwrite(content, 1, strlen(content), file);
                fclose(file);
            }
            else {
                DEBUG(_
                  ("Couldn't create configuration file %s!\n"), filename->str);
            }
        }
        else
            fclose(file);

        g_string_free(filename, TRUE);

    }
}



/**
 * Load IvmConfigBase.xml configuration.
 */
void ivm_load_config()
{
    char *file = ivm_get_config_file( "IvmConfigBase.xml" );
    cfg_base = parseIvmConfigBase(file);
    free(file);

    if ( !cfg_base ) {
        DEBUG(_("Could not parse IvmConfigBase.xml! "
                "Please check this file for errors! "
                "Since I need IvmConfigBase.xml to function, I "
                "will now exit.\n"));
        exit(1);
    }
    // if we're not running as root, set pidFile to ~/.ivman/.lock
    if (usermode) {
        GString *pidFile = g_string_new(homeDir);

        pidFile = g_string_append(pidFile, "/.ivman/.lock");
        cfg_base->pidFile = pidFile->str;
        g_string_free(pidFile, FALSE);
    }
}



/**
 * Free an array of strings.
 */
void ivm_free_str_array(char **strings)
{
    if (strings == NULL)
        return;

    int i = 0;

    while (strings[i] != NULL)
    {
        free(strings[i]);
        i++;
    }

    free(strings);
}

/**
 * Run an array of commands.
 */
void ivm_run_commands(char **commands, char const *udi,
                      gboolean const execun)
{
    if (commands == NULL)
        return;

    int i = 0;

    while (commands[i] != NULL)
    {
        ivm_run_command(commands[i], udi, execun);
        i++;
    }

    if (execun)
        g_hash_table_remove(pss, udi);
}


/**
 * Run a command.
 */
gboolean ivm_run_command(char const *const command,
                         char const *udi, gboolean const execun)
{
    char *argv[4];
    char *new_command = (char *)strdup(command);

    LibHalPropertySet *hal_ps = NULL;
    DBusError dbus_error;

    dbus_error_init(&dbus_error);
    GError *error = NULL;
    GString *currentCommand;
    int length;
    int i;

    char c;
    char *substitution;
    char *propertyname = NULL;
    char *substring = strstr(new_command, "$hal.");
    char *substring2;
    int pos = (int) ((substring - new_command) / sizeof(char));
    char *beginning = (char *) strndup(new_command, pos);

    if (execun)
        hal_ps = (LibHalPropertySet *) g_hash_table_lookup(pss,
                                                           (gpointer) udi);

    while (substring != NULL)
    {
        // get rid of $hal.
        substring[0] = ' ';
        substring[1] = ' ';
        substring[2] = ' ';
        substring[3] = ' ';
        substring[4] = ' ';

        substring = (char *) g_strchug(substring);
        substring2 = strstr(substring, "$");

        length = (int) ((substring2 - substring) / sizeof(char));
        propertyname = (char *) strndup(substring, length);

        i = 0;

        while (i < length)
        {
            substring[i] = ' ';
            i++;
        }

        substring[i] = ' ';

        // hack so we don' t eat an extra space
        c = substring[i + 1];
        substring[i + 1] = 'a';

        substring = (char *) g_strchug(substring);
        substring[0] = c;



        if ((execun && psi_property_exists(hal_ps, propertyname))
            || libhal_device_property_exists(hal_ctx, udi,
                                             propertyname, &dbus_error))
        {
            unsigned int type = execun ? psi_get_property_type(hal_ps,
                                                               propertyname)
                : libhal_device_get_property_type(hal_ctx,
                                                  udi,
                                                  propertyname,
                                                  &dbus_error);

            if (type == DBUS_TYPE_STRING)
            {
                substitution =
                    execun ? (char *)
                    psi_get_property_string(hal_ps,
                                            propertyname)
                    : (char *)
                    libhal_device_get_property_string
                    (hal_ctx, udi, propertyname, &dbus_error);
            }
            else if (type == DBUS_TYPE_BOOLEAN)
            {
                if ((execun
                     && psi_get_property_bool(hal_ps,
                                              propertyname))
                    ||
                    libhal_device_get_property_bool
                    (hal_ctx, udi, propertyname, &dbus_error))
                    substitution = (char *) strdup("true");
                else
                    substitution = (char *) strdup("false");
            }
            else if (type == DBUS_TYPE_INT32 || type == DBUS_TYPE_INT64)
            {
                asprintf(&substitution, "%d",
                         execun ?
                         psi_get_property_int(hal_ps,
                                              propertyname)
                         :
                         libhal_device_get_property_int
                         (hal_ctx, udi, propertyname, &dbus_error));
            }
            else if (type == DBUS_TYPE_UINT32 || type == DBUS_TYPE_UINT64)
            {
                asprintf(&substitution, "%u",
                         execun ? (unsigned int)
                         psi_get_property_uint64(hal_ps,
                                                 propertyname)
                         : (unsigned int)
                         libhal_device_get_property_uint64
                         (hal_ctx, udi, propertyname, &dbus_error));
            }
            else
            {
                // we don 't worry about other types of
                // properties,
                // they' re pretty uncommon}
                substitution = (char *) strdup("NULL");
                DEBUG(_
                      ("Warning: unhandled HAL type encountered, NULL "
                       "substituted for value!\n"));
            }
        }
        else
        {
            DEBUG(_
                  ("Warning: nonexistent HAL property encountered, NULL "
                   "substituted for value!\n"));
            substitution = (char *) strdup("NULL");
        }



        if (propertyname != NULL)
            free(propertyname);

        // Since this will be passed as a shell argument, replace ' and " so
        // we can put the property within quotes.
        while (strstr(substitution, "'") != 0)
        {
            char * substr = strstr(substitution, "'");
            substr[0] = '?';
        }
        while (strstr(substitution, "\"") != 0)
        {
            char * substr = strstr(substitution, "\"");
            substr[0] = '?';
        }


        currentCommand = g_string_new(substring);
        currentCommand = g_string_prepend(currentCommand, substitution);
        currentCommand = g_string_prepend(currentCommand, beginning);
        free(new_command);
        new_command = (char *) strdup(currentCommand->str);
        g_string_free(currentCommand, TRUE);

        if (substitution)
            free(substitution);

        libhal_free_string(beginning);

        substring = strstr(new_command, "$hal.");
        int pos = (int) ((substring - new_command) / sizeof(char));

        beginning = (char *) strndup(new_command, pos);

    }

    libhal_free_string(beginning);

    ivm_check_dbus_error(&dbus_error);

    argv[0] = "/bin/sh";
    argv[1] = "-c";
    argv[2] = new_command;
    argv[3] = NULL;

    DEBUG(_("Running: %s\n"), argv[2]);

    if (!g_spawn_async(NULL, argv, NULL, 0, NULL, NULL, NULL, &error))
    {
        DEBUG(_("Execution of %s failed with error: %s\n"),
              new_command, error->message);
        return FALSE;
    }

    g_free(new_command);
    return TRUE;
}







// returns TRUE if DVD 'udi' is a video DVD, FALSE otherwise
gboolean ivm_is_dvd(char const *const udi)
{
    char *path;
    gboolean retval;

    DBusError dbus_error;

    dbus_error_init(&dbus_error);

    // return false if DVD is not mounted
    if (!libhal_device_property_exists
        (hal_ctx, udi, "volume.is_mounted", &dbus_error))
        return FALSE;

    if (libhal_device_get_property_bool
        (hal_ctx, udi, "volume.is_mounted", &dbus_error) == FALSE)
        return FALSE;

    if (!libhal_device_property_exists
        (hal_ctx, udi, "volume.mount_point", &dbus_error))
        return FALSE;

    char *mount_point = (char *)
        libhal_device_get_property_string(hal_ctx, udi,
                                          "volume.mount_point",
                                          &dbus_error);

    if (!libhal_device_property_exists
        (hal_ctx, udi, "block.device", &dbus_error))
        return FALSE;

    char *device = libhal_device_get_property_string(hal_ctx,
                                                     udi,
                                                     "block.device",
                                                     &dbus_error);

    DEBUG(_("Checking for video DVD in device '%s' mounted at '%s'\n"),
          device, mount_point);

    path = g_build_path(G_DIR_SEPARATOR_S, mount_point, "video_ts", NULL);
    retval = g_file_test(path, G_FILE_TEST_IS_DIR);
    g_free(path);

    /* try the other name, if needed */

    if (retval == FALSE)
    {
        path =
            g_build_path(G_DIR_SEPARATOR_S, mount_point, "VIDEO_TS", NULL);
        retval = g_file_test(path, G_FILE_TEST_IS_DIR);
        g_free(path);
    }

    if (retval == TRUE)
        DEBUG(_("%s looks like a video DVD\n"), device);
    else
        DEBUG(_("%s does not look like a video DVD\n"), device);

    libhal_free_string(mount_point);
    libhal_free_string(device);
    ivm_check_dbus_error(&dbus_error);

    return retval;
}




/**
 * Mount device 'udi'
 */
void ivm_device_mount(char const *const udi)
{
    if (!known_mount_command || !unknown_mount_command)
    {
        DEBUG(_("Can't mount %s; no mount command available!\n"), udi);
        return;
    }
    
    if ( 0 == strcmp( known_mount_command, unknown_mount_command) ) {
        ivm_run_command( known_mount_command, udi, FALSE );
        return;
    }

    DBusError dbus_error;
    dbus_error_init(&dbus_error);

    if ( libhal_device_property_exists(
            hal_ctx,
            udi,
            "volume.policy.desired_mount_point",
            &dbus_error ) ) {
        ivm_run_command( known_mount_command, udi, FALSE );
    }
    else {
        // Now check if the parent device has a desired mount point.
        if ( libhal_device_property_exists(
            hal_ctx,
            udi,
            "block.storage_device",
            &dbus_error ) ) {
            
            char * parent_udi = libhal_device_get_property_string(
                hal_ctx,
                udi,
                "block.storage_device",
                &dbus_error);
            if ( libhal_device_property_exists(
                hal_ctx,
                parent_udi,
                "storage.policy.desired_mount_point",
                &dbus_error) ) {
                
                char * mount_point = libhal_device_get_property_string(
                    hal_ctx,
                    parent_udi,
                    "storage.policy.desired_mount_point",
                    &dbus_error);
                
                GString * command = g_string_new( known_mount_command );
                int strsize = strlen( "$hal.volume.policy.desired_mount_point$" );
                int index =
                    (int)strstr(command->str, "$hal.volume.policy.desired_mount_point$" ) -
                    (int)command->str;
                command = g_string_erase( command, index, strsize );
                command = g_string_insert( command, index, mount_point );
                libhal_free_string( mount_point );
                
                ivm_run_command( command->str, udi, FALSE );
                g_string_free( command, TRUE );
            }
            else {
                libhal_free_string( parent_udi );
                ivm_run_command( unknown_mount_command, udi, FALSE );
            }
        }
        else {
            ivm_run_command( unknown_mount_command, udi, FALSE );
        }
    }

    ivm_check_dbus_error(&dbus_error);

    return;
}



/**
 * Handle media change / device attach on 'udi'
 */
void ivm_media_changed(char const *const udi)
{
    DBusError dbus_error;

    dbus_error_init(&dbus_error);

    char *file = ivm_get_config_file("IvmConfigActions.xml");
    IvmConfigActions *cfg = parseIvmConfigActions(file, (char *) udi);
    
    if ( cfg->execdvd ) {
        DEBUG( _("WARNING: execdvd rule found.  execdvd is deprecated.  Use HAL property volume.disc.is_videodvd instead.") );
    }

    if (!cfg)
    {
        xmlReadFailed("IvmConfigActions.xml");
        free(file);
        return;
    }

    char *device = NULL;

    if (libhal_device_property_exists
        (hal_ctx, udi, "block.device", &dbus_error))
        device =
            (char *)
            libhal_device_get_property_string(hal_ctx,
                                              udi,
                                              "block.device", &dbus_error);

    if (cfg->mount)
    {
        // If we 're running in system-wide mode, sleep a few
        // seconds
        // to allow any user-mode instances to mount first.

        if (!usermode && cfg_base->sleep)
        {
            DEBUG(_("Giving other programs a chance to mount...\n"));
            sleep(5);
        }

        if (libhal_device_property_exists(hal_ctx, udi,
                                          "volume.is_mounted",
                                          &dbus_error)
            && !libhal_device_get_property_bool(hal_ctx, udi,
                                                "volume.is_mounted",
                                                &dbus_error))
        {
            DEBUG(_("Attempting to mount device %s\n"), device);
            ivm_device_mount(udi);
            sleep(1);
        }
    }

    if ((!libhal_device_property_exists(hal_ctx, udi,
                                        "volume.is_mounted",
                                        &dbus_error))
        ||
        (!libhal_device_get_property_bool
         (hal_ctx, udi, "volume.is_mounted", &dbus_error))
        ||
        (!libhal_device_property_exists(hal_ctx, udi, "volume.mount_point", &dbus_error))
       )
    {
        DEBUG(_("%s wasn't mounted, by us or by others..."), udi);
    }
    else 
    {
        // After mounting the filesystem, it takes HAL a while to
        // update, so we poll once every second for five seconds
        // to find out the mount point

        int i = 0;

        char *mount_point = (char *)
            libhal_device_get_property_string(hal_ctx, udi,
                                              "volume.mount_point",
                                              &dbus_error);

        while (( !mount_point || (strlen(mount_point) < 2) )
               && (i < 5))
        {
            sleep(1);
            mount_point = (char *)
                libhal_device_get_property_string(hal_ctx, udi,
                                                  " volume.mount_point ",
                                                  &dbus_error);
            i++;
        }


        if ( !mount_point || strlen(mount_point) < 2)
        {
            DEBUG(_("Couldn't get mount point of device %s after "
                    "5 seconds\n"), device);
        }
        else
        {
            DEBUG(_("Device %s is mounted at %s\n"), device, mount_point);
        }

        libhal_free_string(mount_point);
    }
    
    // Now that we've mounted, reparse the file, as some properties may have changed
    // (e.g. mount point)
    ivm_free_str_array(cfg->execdvd);
    ivm_free_str_array(cfg->exec);
    if (cfg->execun)
        free(cfg->execun);
    free(cfg);
    
    cfg = parseIvmConfigActions(file, (char *) udi);
    free(file);
    if (!cfg) {
        xmlReadFailed("IvmConfigActions.xml");
        return;
    }

    if ( ivm_is_dvd(udi) )
        ivm_run_commands(cfg->execdvd, udi, FALSE);
    ivm_free_str_array(cfg->execdvd);

    ivm_run_commands(cfg->exec, udi, FALSE);
    ivm_free_str_array(cfg->exec);

    if (cfg->execun) {
        g_hash_table_insert(execuns, (gpointer)
                            strdup(udi), (gpointer) cfg->execun);
    }
    LibHalPropertySet *hal_ps =
        libhal_device_get_all_properties(hal_ctx, udi, &dbus_error);

    g_hash_table_insert(pss, (gpointer) strdup(udi), (gpointer) hal_ps);

    free(cfg);

    libhal_free_string(device);
    ivm_check_dbus_error(&dbus_error);
}



/** Unmount if required */
void ivm_umount_if_needed(char const *const udi)
{
    if (udi == NULL)
        return;

    DEBUG(_("Unmounting %s\n"), udi);

    ivm_run_command(umount_command, udi, TRUE);
}

/**
  * Set up some gettext stuff.
  */
void setupI18N()
{
    if ( !bindtextdomain(PACKAGE, DATADIR "/locale/") )
        DEBUG(_("bindtextdomain() failed: %s\n"), strerror(errno));

    if ( !textdomain(PACKAGE) )
        DEBUG(_("textdomain() failed: %s\n"), strerror(errno));

    if ( !setlocale(LC_ALL, "") )
        DEBUG(_("setlocale() failed: %s\n"), strerror(errno));
}


/**
 * Set up connection to HAL.
 */
void setupHAL()
{
    DBusError dbus_error;
    dbus_error_init(&dbus_error);

#ifdef HAL_0_4
    LibHalFunctions hal_functions = {
        hal_mainloop_integration,
        hal_device_added,
        hal_device_removed,
        hal_device_new_capability,
        hal_device_lost_capability,
        hal_device_property_modified, hal_device_condition
    };

    hal_ctx = (LibHalContext *) hal_initialize(&hal_functions, FALSE);
    if (!hal_ctx)
    {
        DEBUG(_("Failed to create HAL context!\n"));
        exit(1);
    }
#else
    hal_ctx = (LibHalContext *) libhal_ctx_new();
    if (!hal_ctx)
    {
        DEBUG(_("Failed to create HAL context!\n"));
        exit(1);
    }

    if (!hal_mainloop_integration(hal_ctx, &dbus_error))
    {
        DEBUG(_("Couldn't connect to HAL!\n"));
        exit(1);
    }

    // Set up all the callbacks in hal_interface.c
    libhal_ctx_set_device_added(hal_ctx, hal_device_added);
    libhal_ctx_set_device_removed(hal_ctx, hal_device_removed);
    libhal_ctx_set_device_new_capability(hal_ctx,
                                         hal_device_new_capability);
    libhal_ctx_set_device_lost_capability(hal_ctx,
                                          hal_device_lost_capability);
    libhal_ctx_set_device_property_modified(hal_ctx,
                                            hal_device_property_modified);
    libhal_ctx_set_device_condition(hal_ctx, hal_device_condition);

    if (!libhal_ctx_init(hal_ctx, &dbus_error))
    {
        DEBUG(_("Couldn't initialise HAL!\n"));
        exit(1);
    }
#endif



#ifdef HAL_0_4
    if (libhal_device_property_watch_all(hal_ctx, &dbus_error)) {
        DEBUG(_("Failed to watch all HAL properties!\n"));
        exit(1);
    }
#else
    if (!libhal_device_property_watch_all(hal_ctx, &dbus_error)) {
        DEBUG(_("Failed to watch all HAL properties!\n"
                "Error: %s\n"), dbus_error.message);
        exit(1);
    }
#endif                          // HAL_0_4

    devices = g_hash_table_new_full(g_str_hash, g_str_equal, free, free);
    execuns = g_hash_table_new_full(g_str_hash, g_str_equal,
                                    free,
                                    (GDestroyNotify) ivm_free_str_array);
#ifndef HAL_0_4

    pss =
        g_hash_table_new_full(g_str_hash,
                              g_str_equal, free,
                              (GDestroyNotify) libhal_free_property_set);
#else

    pss =
        g_hash_table_new_full(g_str_hash,
                              g_str_equal, free,
                              (GDestroyNotify) hal_free_property_set);
#endif

    // Now try to watch all properties, if necessary.
    char *file = ivm_get_config_file("IvmConfigProperties.xml");
    IvmConfigProperties *prop_cfg =
        parseIvmConfigProperties(file, NULL, NULL);

    if ( !prop_cfg ) {
        xmlReadFailed(" IvmConfigProperties.xml ");
    }

    else {
        if ( prop_cfg->checkOnInit ) {
            DEBUG(_
                  (" Running through rules for every current property.\n"));

            // run through every device
            int num_devices;
            char **devices =
                (char **) libhal_get_all_devices(hal_ctx, &num_devices,
                                                 &dbus_error);

            if ( !devices ) {
                DEBUG(_
                  (" Couldn't enumerate all devices!\n"));
#ifndef HAL_0_4
                ivm_check_dbus_error(&dbus_error);
#endif
            }

            int i = 0;

            while ( devices && ( i < num_devices ) )
            {
                //DEBUG(_("Currently checking device %s"), devices[i]);
                IvmConfigProperties *newProp_cfg =
                    IvmConfigPropertiesAll(file, devices[i]);

                ivm_run_commands(newProp_cfg->exec, devices[i], FALSE);
                ivm_free_str_array(newProp_cfg->exec);
                free(newProp_cfg);

                i++;
            }

            if ( devices )
                libhal_free_string_array(devices);

        }


        free(prop_cfg);
    }

    free(file);

    ivm_check_dbus_error(&dbus_error);
}



/**
 * Parse command-line options.
 */
void parseArguments(int argc, char *argv[])
{
    GError *optParseError = NULL;

    GOptionEntry entries[] = {
        {"system", 's', 0, G_OPTION_ARG_NONE, &systemOpt,
        _("Force Ivman to run in system-wide mode. Use this if you "
          "want Ivman to behave like you started it as root, even if you did not."),
         NULL},
        {"debug", 'd', 0, G_OPTION_ARG_NONE, &debugOpt,
         _("Force Ivman to show debugging output."), NULL},
        {"confdir", 'c', 0, G_OPTION_ARG_STRING, &confdirOpt,
         _("Force Ivman to look in a specific directory for configuration (default: /etc/ivman or ~/.ivman)"),
         NULL},
        {"nofork", (char) NULL, 0, G_OPTION_ARG_NONE, &noForkOpt,
         _("Force Ivman not to daemonize itself."), NULL},
        {(const gchar *) NULL, (gchar) NULL, (gint) NULL, (GOptionArg) NULL,
         (gpointer) NULL, (const gchar *) NULL, (const gchar *) NULL}
    };

    GOptionContext *context = g_option_context_new(_("- start ivman"));

    g_option_context_add_main_entries(context, entries, NULL);

    confdirOpt = NULL;
    if (!g_option_context_parse(context, &argc, &argv, &optParseError))
    {
        DEBUG("%s\n", optParseError->message);
        exit(1);
    }

    if ( confdirOpt && ( strlen(confdirOpt) == 0 ) ) {
        DEBUG(_("-c option given with no directory!"));
        exit(1);
    }
}










/**
 * Do all configuration necessary for initialisation.
 * Must be called after parseArguments.
 */
void do_startup_configure()
{
    // Assume user mode
    usermode = TRUE;

    // are we running as root or with --system option?
    if ( systemOpt || ( geteuid() == 0 ) )
        usermode = FALSE;

    homeDir = (gchar *) g_get_home_dir();

    if ( !homeDir ) {
        DEBUG( _("Can't find home directory, exiting!\n") );
        exit(1);
    }

    // Set config dir, if not passed as a commandline option
    if ( !confdirOpt )
        confdirOpt = get_confdir();

    // Now make sure last character is '/'.
    if ( confdirOpt[strlen(confdirOpt)-1] != '/' ) {
        GString * newopt = g_string_new(confdirOpt);
        newopt = g_string_append(newopt, "/");
        free(confdirOpt);
        confdirOpt = newopt->str;
        g_string_free(newopt, FALSE);
    }

    DEBUG(_("Directory %s will be used for configuration files."), confdirOpt);

    // Create default configs if necessary
    ivm_test_configs();

    // Load configuration
    ivm_load_config();

    if ( debugOpt )
        cfg_base->debug = TRUE;

    isdaemon = FALSE;

    if ( noForkOpt )
        cfg_base->fork = FALSE;



    // Find mount command to use.
    set_mount_command();
    if ( !known_mount_command || !unknown_mount_command || !umount_command ) {
        DEBUG(_("An appropriate mount command could not be found! "
                "Please make sure you have mount/umount or pmount/pumount in "
                "Ivman's PATH. Mounting currently will not work.\n"));
    }

    // Permanently unlock CD / DVD drive
    if ( geteuid() == 0 )
        ivm_run_command( "echo 0 > /proc/sys/dev/cdrom/lock", NULL, FALSE );


}



/**
 * main function.
 */
int main(int argc, char *argv[])
{
    setupI18N();
    parseArguments(argc, argv);
    do_startup_configure();

    // Connect to HAL
    setupHAL();

    DEBUG("%s, http:/ivman.sourceforge.net\n", PACKAGE_STRING);

#ifdef HAL_0_4
    DEBUG(_("Compiled against HAL 0.4.x or earlier\n"));
#else
    DEBUG(_("Compiled against HAL 0.5.x or later\n"));
#endif

    if ( !usermode ) {
        DEBUG(_("Running in system mode.\n"));
    }
    else {
        DEBUG(_("Running in user mode.\n"));
    }


    if ( cfg_base->fork ) {
        // Daemonize
        isdaemon = daemonize();

        if ( !isdaemon ) {
            DEBUG(_("Couldn't daemonize, exiting..."));
            return 1;
        }
    }

    // Drop privileges.  Must do this _after_ daemonizing!
    if ((geteuid() == 0) && (!dropPrivileges(cfg_base->user, cfg_base->group)))
    {
        DEBUG(_("Couldn't drop privileges, exiting!"),
              cfg_base->user, cfg_base->group);
        return 1;
    }

    loop = g_main_loop_new(NULL, FALSE);
    if ( !loop ) {
        DEBUG( _("Error creating main loop!\n") );
        return 1;
    }

    signal(SIGTERM, signal_handler);
    signal(SIGINT, signal_handler);
    
    if ( !cfg_base->fork )
        signal(SIGHUP, signal_handler);

    DEBUG(_("Entering main loop.\n"));
    g_main_loop_run(loop);
    DEBUG(_("Exiting normally.\n"));

    // When we get here, main loop is finished.
    if ( cfg_base->fork )
        clear_pidfile(cfg_base->pidFile);

    g_hash_table_destroy(devices);

    return 0;

}
