/*
 * jabberd - Jabber Open Source Server
 * Copyright (c) 2002 Jeremie Miller, Thomas Muldowney,
 *                    Ryan Eatmon, Robert Norris
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA02111-1307USA
 */

#include "sm.h"

/** @file sm/object.c
  * @brief object sets
  * @author Robert Norris
  * $Date: 2004/04/30 02:42:57 $
  * $Revision: 1.4.2.1 $
  */

os_t os_new(void) {
    pool p;
    os_t os;

    p = pool_new();
    os = (os_t) pmalloco(p, sizeof(struct os_st));

    os->p = p;

    return os;
}

void os_free(os_t os) {
    pool_free(os->p);
}

int os_count(os_t os) {
    return os->count;
}

int os_iter_first(os_t os) {
    os->iter = os->head;

    if(os->iter == NULL)
        return 0;

    return 1;
}

int os_iter_next(os_t os) {
    if(os->iter == NULL)
        return 0;

    os->iter = os->iter->next;

    if(os->iter == NULL)
        return 0;

    return 1;
}

os_object_t os_iter_object(os_t os) {
    return os->iter;
}

os_object_t os_object_new(os_t os) {
    os_object_t o;

    log_debug(ZONE, "creating new object");

    o = (os_object_t) pmalloco(os->p, sizeof(struct os_object_st));
    o->os = os;

    o->hash = xhash_new(51);

    /* make sure that the hash gets freed when the os pool gets freed */
    pool_cleanup(os->p, (pool_cleaner) pool_free, (void *) xhash_pool(o->hash));

    /* insert at the end, we have to preserve order */
    o->prev = os->tail;
    if(os->tail != NULL) os->tail->next = o;
    os->tail = o;
    if(os->head == NULL) os->head = o;
    
    os->count++;

    return o;
}

void os_object_free(os_object_t o) {
    log_debug(ZONE, "dropping object");

    if(o->prev != NULL)
        o->prev->next = o->next;
    if(o->next != NULL)
        o->next->prev = o->prev;

    if(o->os->head == o)
        o->os->head = o->next;
    if(o->os->tail == o);
        o->os->tail = o->prev;

    if(o->os->iter == o)
        o->os->iter = o->next;

    o->os->count--;
}

void os_object_put(os_object_t o, const char *key, const void *val, os_type_t type) {
    os_field_t osf;
    nad_t nad;

    log_debug(ZONE, "adding field %s (val %x type %d) to object", key, val, type);

    osf = pmalloco(o->os->p, sizeof(struct os_field_st));
    osf->key = pstrdup(o->os->p, key);

    switch(type) {
        case os_type_BOOLEAN:
        case os_type_INTEGER:
            osf->val = (void *) pmalloco(o->os->p, sizeof(int));
            * (int *) osf->val = * (int *) val;
            break;

        case os_type_STRING:
            osf->val = (void *) pstrdup(o->os->p, (char *) val);
            break;

        case os_type_NAD:
            nad = nad_copy((nad_t) val);

            /* make sure that the nad gets freed when the os pool gets freed */
            pool_cleanup(o->os->p, (pool_cleaner) nad_free, (void *) nad);

            osf->val = (void *) nad;
            break;
    }

    osf->type = type;

    xhash_put(o->hash, osf->key, (void *) osf);
}

int os_object_get(os_object_t o, const char *key, void **val, os_type_t *type) {
    os_field_t osf;

    osf = (os_field_t) xhash_get(o->hash, key);
    if(osf == NULL) {
        *val = NULL;
        return 0;
    }

    *type = osf->type;

    switch(osf->type) {
        case os_type_BOOLEAN:
        case os_type_INTEGER:
            * (int *) val = * (int *) osf->val;
            break;

        case os_type_STRING:
        case os_type_NAD:
            *val = osf->val;
            break;

        default:
            *val = NULL;
    }

    log_debug(ZONE, "got field %s (val %x type %d) to object", key, *val, *type);

    return 1;
}

int os_object_iter_first(os_object_t o) {
    return xhash_iter_first(o->hash);
}

int os_object_iter_next(os_object_t o) {
    return xhash_iter_next(o->hash);
}

void os_object_iter_get(os_object_t o, char **key, void **val, os_type_t *type) {
    os_field_t osf;

    xhash_iter_get(o->hash, (const char **) key, (void **) &osf);
    if(*key == NULL) {
        *val = NULL;
        return;
    }

    *type = osf->type;

    switch(osf->type) {
        case os_type_BOOLEAN:
        case os_type_INTEGER:
            * (int *) val = * (int *) osf->val;
            break;

        case os_type_STRING:
        case os_type_NAD:
            *val = osf->val;
            break;

        default:
            *val = NULL;
    }
    
    log_debug(ZONE, "got iter field %s (val %x type %d) to object", *key, *val, *type);
}
