#include <jmp-config.h>

#include <stdio.h>
#include <stdlib.h>

#include <gtk/gtk.h>

#include <jmp.h>
#include <hash.h>
#include <method.h>
#include <cls.h>
#include <comparators.h>
#include <stats_context.h>
#include <stats_personality.h>
#include <jmptime.h>

#include <ui_gtk.h>
#include <ui_gtk_prefs.h>
#include <ui_gtk_gtkutils.h>
#include <ui_gtk_method_window.h>
#include <ui_gtk_methodlist_menu.h>

static stats_context_t stats_context;

static GtkWidget* method_window;

static GtkListStore *method_list = NULL;
static int method_list_size = 0;
static GtkWidget *method_statusbar;

static int max_method_rows = 100;


int get_method_rows () {
    return max_method_rows;
}

void set_method_rows (int rows) {
    max_method_rows = rows;
}

/** array of method compare functions... */
static int ((*method_comprs[])(const void* v1, const void* v2)) = { method_compr_class,
								    method_compr_class,
								    method_compr_time,
								    method_compr_calls,
								    method_compr_hold_time,
								    method_compr_total_time,
								    method_compr_total_time_per_call,
								    method_compr_objects,
								    method_compr_objpercall,
								    method_compr_bytes,
};

static int ((*method_comprs_r[])(const void* v1, const void* v2)) = { method_compr_class_r,
								    method_compr_class_r,
								    method_compr_time_r,
								    method_compr_calls_r,
								    method_compr_hold_time_r,
								    method_compr_total_time_r,
								    method_compr_total_time_per_call_r,
								    method_compr_objects_r,
								    method_compr_objpercall_r,
								    method_compr_bytes_r,
};


static void method_column_clicked (GtkWidget *treeviewcolumn, gpointer user_data) {
    int column = (int)user_data;
    if (method_comprs[column] != NULL) {
        int (*old_func)(const void* v1, const void* v2);
        old_func = stats_context_get_compr(&stats_context);
        if(old_func == method_comprs[column])
          stats_context_set_compr(&stats_context, method_comprs_r[column]);
        else
          stats_context_set_compr(&stats_context, method_comprs[column]);
        update_method_tree (get_methods ());
    } else {
	fprintf (stdout, "Sort order not yet implemented.\n");
    }
}

void toggle_method_window () {
    if(method_window == NULL)
	return;
    if(GTK_WIDGET_VISIBLE(method_window)) {
	gtk_widget_hide_all (method_window);
    } else {
	gtk_widget_show_all (method_window);
	update_method_tree (get_methods ());    
    }
}

static void destroy (GtkWidget *widget, gpointer data) {
    /* NOP */
}

static void init () {
    stats_context_init (&stats_context, &stats_personality_method_usage);
    stats_context_set_compr (&stats_context, method_compr_time);
}

static void done () {
    stats_context_end (&stats_context);
}

static void build_method_window () {
    GtkWidget* scroller;
    GtkWidget *tree;
    GtkTreeSelection *select;
    GtkWidget *vbox;

    GtkWidget* jmpwin = gtk_window_new (GTK_WINDOW_TOPLEVEL);
    gtk_signal_connect (GTK_OBJECT (jmpwin), "delete-event",
			GTK_SIGNAL_FUNC (gtk_widget_hide_on_delete), NULL);    
    gtk_signal_connect (GTK_OBJECT (jmpwin), "destroy",
			GTK_SIGNAL_FUNC (destroy), NULL);
    gtk_window_set_title (GTK_WINDOW (jmpwin), _("Java Memory Profiler - Methods"));
    scroller = gtk_scrolled_window_new (NULL, NULL);
    vbox = gtk_vbox_new (FALSE, 0);
    gtk_container_add (GTK_CONTAINER (jmpwin), vbox);
    gtk_box_pack_start (GTK_BOX (vbox), scroller, TRUE, TRUE, 0);
    
    method_list = gtk_list_store_new (MN_COLUMNS, G_TYPE_STRING, G_TYPE_STRING, 
				      G_TYPE_STRING, G_TYPE_LONG, G_TYPE_STRING,
				      G_TYPE_STRING, G_TYPE_STRING, G_TYPE_INT, G_TYPE_INT, 
				      G_TYPE_STRING, G_TYPE_POINTER);
    
    tree = gtk_tree_view_new_with_model (GTK_TREE_MODEL (method_list));
    add_column (tree, _("Class"), MCLASS_COLUMN, (gpointer)0, method_column_clicked, 200, 0);
    add_column (tree, _("Method"), MMETHOD_NAME_COLUMN, (gpointer)1, method_column_clicked, 200, 0);
    add_column (tree, _("secs"), MSECS_COLUMN, (gpointer)2, method_column_clicked, 80, 1);
    add_column (tree, _("calls"), MCALLS_COLUMN, (gpointer)3, method_column_clicked, 80, 1);
    add_column (tree, _("subs sec"), MSUBS_COLUN, (gpointer)4, method_column_clicked, 80, 1);
    add_column (tree, _("total"), MTOTAL_COLUMN, (gpointer)5, method_column_clicked, 80, 1);
    add_column (tree, _("total/call"), MTOTAL_CALL_COLUMN, (gpointer)6, method_column_clicked, 80, 1);
    add_column (tree, _ ("objects"), MOBJECTS_COLUMN, (gpointer) 7, method_column_clicked, 80, 1);
    add_column (tree, _ ("objs/call"), MOBJSCALL_COLUMN, (gpointer) 8, method_column_clicked, 80, 1);
    add_column (tree, _ ("bytes"), MBYTES_COLUMN, (gpointer) 9, method_column_clicked, 80, 1);
    gtk_tree_view_set_headers_clickable (GTK_TREE_VIEW (tree), TRUE);
    gtk_container_add (GTK_CONTAINER (scroller), tree);    

    select = gtk_tree_view_get_selection (GTK_TREE_VIEW (tree));
    g_signal_connect (G_OBJECT (select), "changed",
		      G_CALLBACK (mlist_row_changed),
		      method_list);
    gtk_signal_connect (GTK_OBJECT(tree), "button_press_event",
			GTK_SIGNAL_FUNC (mlist_button_handler), 
			NULL);

    method_statusbar = gtk_statusbar_new ();
    gtk_box_pack_start (GTK_BOX (vbox), method_statusbar, FALSE, FALSE, 0);

    gtk_window_set_default_size (GTK_WINDOW (jmpwin), 800, 200);
    ui_gtk_prefs_load_window (UI_GTK_PREFS_METHOD_WINDOW, ui_gtk_state(), GTK_WINDOW (jmpwin));
    method_window = jmpwin;
}

void setup_method_tracing () {
    if (method_list == NULL) {
        init ();
	build_method_window ();
    }
}

void quit_method_window () {
    if (method_window) {
        ui_gtk_prefs_save_window (UI_GTK_PREFS_METHOD_WINDOW, (GtkWindow *)method_window);
	gtk_widget_destroy (method_window);
	method_window = NULL;
        done ();
    }
}

static void get_time_string (char* buf, size_t len, jlong val) {
    jlong jsec, jnsec;
    jmptime_set_time_values (val, &jsec, &jnsec);
    snprintf (buf, len, "%lld.%06lld", jsec, jnsec);
}

static long safe_div(long a, int b) {
    if (b > 0)
	return a / b;
    return 0;
}

static jlong safe_div_jlong(jlong a, int b) {
    if (b > 0)
	return a / b;
    return 0;
}

/** Add a row to the gtk CList (update the ui). */
static void add_method_row_to_list (method* m, int row, GtkTreeIter* iter) {
    char sec[64];
    char hsec[64];
    char tsec[64];
    char tsec_call[64];
    methodtime* mt;
    int fast = 0;

    if (row < method_list_size) {
	method* m2;
	gtk_tree_model_get (GTK_TREE_MODEL (method_list), iter, MMETHOD_COLUMN, &m2, -1);
	if (m == m2) {
	    if (!method_check_modified (m)) {
		gtk_tree_model_iter_next (GTK_TREE_MODEL (method_list), iter);
		return;
	    }
	    fast = 1;	    
	}
    } else {
	gtk_list_store_append (method_list, iter);
    }

    mt = method_get_time_used (m);
#ifdef DLMDEBUG
{char *name = cls_get_name(method_get_owner (m));
if(name != NULL && strcmp(name, "java.lang.Object") == 0)
 fprintf(stderr, "MMMM Methtime tvsec=%lld hsec=%lld tsec=%lld %s\n", mt->tv, mt->tv_hold, mt->tv + mt->tv_hold, name);
else if(name != NULL && strcmp(name, "java.lang.Class") == 0)
 fprintf(stderr, "MMMM Methtime tvsec=%lld hsec=%lld tsec=%lld %s\n", mt->tv, mt->tv_hold, mt->tv + mt->tv_hold, name);
 }
#endif
 
    get_time_string (sec, 64, mt->tv);
    get_time_string (hsec, 64, mt->tv_hold);
    get_time_string (tsec, 64, mt->tv + mt->tv_hold);
    get_time_string (tsec_call, 64, safe_div_jlong(mt->tv + mt->tv_hold, method_get_calls (m)));
    if (fast) {
	gtk_list_store_set (method_list, iter,
			    MSECS_COLUMN, sec,
			    MCALLS_COLUMN, method_get_calls (m),
			    MSUBS_COLUN, hsec,
			    MTOTAL_COLUMN, tsec,
			    MTOTAL_CALL_COLUMN, tsec_call,
			    MOBJECTS_COLUMN, method_get_allocated_objects (m),
			    MOBJSCALL_COLUMN, safe_div(method_get_allocated_objects (m), method_get_calls (m)),
			    MBYTES_COLUMN, format_num (method_get_allocated_memory (m)),
			    -1);
    } else {
	cls* c = method_get_owner (m);
	gtk_list_store_set (method_list, iter,
			    MCLASS_COLUMN, cls_get_name (c),
			    MMETHOD_NAME_COLUMN, method_get_method_jmpname (m),
			    MSECS_COLUMN, sec,
			    MCALLS_COLUMN, method_get_calls (m),
			    MSUBS_COLUN, hsec,
			    MTOTAL_COLUMN, tsec,
			    MMETHOD_COLUMN, m,
			    MTOTAL_CALL_COLUMN, tsec_call,
			    MOBJECTS_COLUMN, method_get_allocated_objects (m),
			    MOBJSCALL_COLUMN, safe_div(method_get_allocated_objects (m), method_get_calls (m)),
			    MBYTES_COLUMN, format_num (method_get_allocated_memory (m)),
			    -1);
    }
    method_set_modified (m, 0);
    gtk_tree_model_iter_next (GTK_TREE_MODEL (method_list), iter);
}

void update_method_tree (hashtab* methods) {
    char buf[64];

    setup_method_tracing ();

    if (method_window == NULL || !GTK_WIDGET_VISIBLE (method_window))
        return;

    /* Single pass filter, count and copy (with total) to expandable arraylist datums */
    stats_context_calc (&stats_context, methods);

    update_tree (method_list, stats_context.resultlist_count, max_method_rows, (void**)stats_context.resultlist, 
		 (add_row_func)add_method_row_to_list, method_list_size);
    method_list_size = 
	max_method_rows < stats_context.resultlist_count ? max_method_rows : stats_context.resultlist_count;
    snprintf (buf, 64, _("Showing %d methods out of %d"), 
	      method_list_size, stats_context.resultlist_total);
    set_status_internal (method_statusbar, buf);
}

/* Emacs Local Variables: */
/* Emacs mode:C */
/* Emacs c-indentation-style:"gnu" */
/* Emacs c-hanging-braces-alist:((brace-list-open)(brace-entry-open)(defun-open after)(substatement-open after)(block-close . c-snug-do-while)(extern-lang-open after)) */
/* Emacs c-cleanup-list:(brace-else-brace brace-elseif-brace space-before-funcall) */
/* Emacs c-basic-offset:4 */
/* Emacs End: */
