/*
 * Copyright (C) 2003 Joo-won Jung <sanori@sanori.net>
 * 
 *     This program is free software; you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation; either version 2 of the License, or
 *     (at your option) any later version.
 * 
 *     This program is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 * 
 *     You should have received a copy of the GNU General Public License
 *     along with this program; if not, write to the Free Software
 *     Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111, USA.
 */

#ifndef _MIME_CODER_H_
#define _MIME_CODER_H_
#include <stdlib.h>

typedef enum {ENC_NONE=0, ENC_QP=1, ENC_B64=2} EncodingType;
/* Standard encoding name. The type of the argument is EncodingType */
extern const char *EncodingName[];

/* From http://www.iana.org/assignments/character-sets
 * This limits the string size of 'charset' argument */
#define CHARSET_LEN 40 

/* 
 * Each 'encode_*' function encodes the string 'in' with the length 'ilen'
 * and store the result to 'out' whose allocated space is 'olen'.
 * Each function returns the number of bytes encoded.
 *
 * Each 'decode_*' function decodes the string 'in' and store the result
 * to the character array 'out' whose length is 'olen'. If the arguement
 * 'ilen' is not supplied, 'in' is considered as null-terminated string.
 * Each fuction returns the number of bytes decoded (<= olen)
 * or -1 (bad encoding).
 */

size_t encode_b64 (char *out, size_t olen, const char *in, size_t ilen);
size_t _decode_b64(char *out, size_t olen, const char *in, size_t ilen);
#define decode_b64(o, ol, i) _decode_b64((o), (ol), (i), strlen(i))

size_t _encode_qp(char *out, size_t olen, const char *in, size_t ilen,
		int rfc2047);
size_t _decode_qp(char *out, size_t olen, const char *in, size_t ilen,
		int rfc2047);

#define encode_qp(o, ol, i, il) _encode_qp((o), (ol), (i), (il), 0)
#define decode_qp(o, ol, i) _decode_qp((o), (ol), (i), strlen(i), 0)

/*
 * RFC-2047 coding fuctions inserts '\n' not to exceed the maximum
 * characters in a line, 76, of RFC-822 and RFC-2047.
 * The 'header_size' argument denotes the number of the characters of
 * header which occupies the first line.
 */
size_t decode_rfc2047(char *out, size_t olen, char *charset, const char *in);
size_t encode_rfc2047(char *out, size_t olen, const char *in,
		size_t ilen, const char *charset, EncodingType enc, size_t header_len);

/*
 * The following functions return the number of bytes that each encoding
 * function requires. The termination null character is not included in
 * the number of bytes.
 */
#define enclen_b64(ilen) ((((ilen)+2)/3)*4)
#define enclen_qp(ilen) ((ilen)*3)
size_t enclen_rfc2047(size_t ilen, size_t charsetlen, EncodingType enc,
		size_t header_len);
#endif
