#include "jpilot_helpers.h"
#include "../jpilot_src/libplugin.h"
#include "pytype_basics.h"
#include <pi-buffer.h>
#include <Python.h>

PyObject *DeleteObjectFromJpilotDatabase(PyObject *record, char *database) {
  if (((PyPiBase*)record)->saved_br.size > 0) {
    if (jp_delete_record(database, 
			 &(((PyPiBase*)record)->saved_br), 
			 DELETE_FLAG) == EXIT_FAILURE) {
      PyErr_SetString(PyExc_IOError,
		      "Unable to write delete record in jpilot database.");
      return NULL;
    } else {
      /* Successful delete. Make this record not think it came from the db anymore */
      if (((PyPiBase*)record)->saved_br.buf)
	free(((PyPiBase*)record)->saved_br.buf);
      ((PyPiBase*)record)->saved_br.buf = NULL;
      ((PyPiBase*)record)->saved_br.size = 0;
    }
  } else {
    PyErr_SetString(PyExc_ValueError,
		    "This record was not loaded from the database, no record to delete.");
    return NULL;
  }
  Py_INCREF(Py_None);
  return Py_None;
}

PyObject *SaveObjectToJpilotDatabase(PyObject *record, char *database, int recordtype) {
  int rec_len;
  buf_rec br;
  pi_buffer_t *buffer;
  
  if (((PyPiBase*)record)->packer == NULL) {
    PyErr_SetString(PyExc_RuntimeError,
		    "Unable to pack record, no packer given.");
    return NULL;
  }
  
  /* Keep unique ID intact */
  if (((PyPiBase*)record)->unique_id) {
    br.unique_id = ((PyPiBase*)record)->unique_id;
  } else {
    br.unique_id = 0;
  }
  
  /* http://cvs.coldsync.org/cgi-bin/viewcvs.cgi/coldsync/libpdb/
     pdb.c?rev=1.44&content-type=text/vnd.viewcvs-markup 
     The archived bit is only set, if deleted is set. Otherwise,
     the bit is used as part of the category */
  
  if (((PyPiBase*)record)->deleted) {
    /* Deleted is set, so we can set archived, and ignore cat */
    br.attrib = ((PyPiBase*)record)->archived ? 0x08 : 0;
  } else {
    /* record is not deleted, so ignore archived bit, and set category */
    br.attrib = (((PyPiBase*)record)->category);
  }
  
  br.attrib = ((PyPiBase*)record)->deleted ? (br.attrib | 0x80) : br.attrib;
  br.attrib = ((PyPiBase*)record)->modified ? (br.attrib | 0x40) : br.attrib;
  br.attrib = ((PyPiBase*)record)->busy ? (br.attrib | 0x20) : br.attrib;
  br.attrib = ((PyPiBase*)record)->secret ? (br.attrib | 0x10) : br.attrib;
  
  if (((PyPiBase*)record)->saved_br.size > 0) {
    br.rt = REPLACEMENT_PALM_REC;
  } else {
    br.rt = NEW_PC_REC;
  }
  
  buffer = pi_buffer_new(0);
  if (((PyPiBase*)record)->packer(&(((PyPiBase*)record)->a), 
				     buffer, 
				     recordtype) != 0) {
    PyErr_SetString(PyExc_RuntimeError,"Unable to pack record.");
    pi_buffer_free(buffer);    
    return NULL;
  }
  br.buf = buffer->data;
  br.size = buffer->used;
  
  /* need to delete the old one first ? */
  if (((PyPiBase*)record)->saved_br.size > 0) {
    if (jp_delete_record(database, 
			 &(((PyPiBase*)record)->saved_br), 
			 MODIFY_FLAG) == EXIT_FAILURE) {
      PyErr_SetString(PyExc_IOError,
		      "Unable to write delete-previous record in jpilot database.");
      return NULL;
    }
  }

  /* careful, br.buf is pointing at buffer */
  if (jp_pc_write(database, &br) == EXIT_FAILURE) {
    PyErr_SetString(PyExc_IOError,
		    "Unable to write record to jpilot database.");
    pi_buffer_free(buffer);    
    return NULL;
  }
  
  /* Collect the new unique id and reset the dirty bit */
  ((PyPiBase*)record)->unsaved_changes = 0;
  ((PyPiBase*)record)->unique_id = br.unique_id;

  /* Used to delete this record, or to recognise the next write is a replacement */
  if (((PyPiBase*)record)->saved_br.size > 0) {
    if (((PyPiBase*)record)->saved_br.buf)
      free(((PyPiBase*)record)->saved_br.buf);
    ((PyPiBase*)record)->saved_br.size = 0;
  }
  ((PyPiBase*)record)->saved_br.buf = malloc(br.size);
  if (((PyPiBase*)record)->saved_br.buf == NULL) {
    PyErr_SetString(PyExc_MemoryError,"Unable to allocate memory for saved_br.buf.");    
    return NULL;
  }
  memcpy(((PyPiBase*)record)->saved_br.buf,br.buf,br.size);
  ((PyPiBase*)record)->saved_br.size = br.size;
  ((PyPiBase*)record)->saved_br.attrib = br.attrib;
  ((PyPiBase*)record)->saved_br.rt = br.rt;
  ((PyPiBase*)record)->saved_br.unique_id = br.unique_id;

  pi_buffer_free(buffer);

  Py_INCREF(record);
  return record;
}


int read_DB_and_rewind_list(char *database, GList **records) {
  GList *temp_list;
  int n;
  
  n = jp_read_DB_files(database, records);
  if (n > 0)
    for (temp_list = *records; temp_list; temp_list = temp_list->prev) {
      *records = temp_list;
    }
  return n;
}

PyObject *AppInfoCategories_to_PyList(struct CategoryAppInfo *cai) {
  PyObject *cats;
  int i;
  
  cats = PyList_New(16);
  for (i = 0; i < 16; i++) {
    PyList_SET_ITEM(cats,i,PyUnicode_Decode(cai->name[i],
					    strlen(cai->name[i]),
					    PALMOSCHARENCODING,
					    NULL));
  }
  return cats;
}
