#include <Python.h>
#include <vobject.h>

#include "move_into_libpisock.h"
#include "pytype_addresses.h"
#include "pytype_basics.h"
#include "structmember.h"
#include "python_compatibility.h"
#include <pi-buffer.h>
#include "python_compatibility.h"

/*******************************************************
 * Create and delloc methods for objects 
 ******************************************************/

extern PyObject* PyPiAddress_New(PyTypeObject *type, PyObject *args, PyObject *kwds) {
  int i;
  PyPiAddress* self;

  /* Why do we have to do this here? The one in the swig init doesn't seem
     to work ?! */
  mxDateTime_ImportModuleAndAPI();

  AddressType.ob_type = &PyType_Type;
  self = (PyPiAddress *)type->tp_alloc(type, 0);
  new_Address(&(self->a));
  SetBasicRecordObjectAttributeDefaults((PyObject*) self, pack_Address);  

  return (PyObject*)self;
}

extern int PyPiAddress_Init(PyObject *self, PyObject *args, PyObject *kwds) {
  PyPiAddress* fromaddress = NULL;
  PyPiAddress* address = NULL;
  PyObject* filters = NULL;
  int i;
  int malloc_failed = 0;
  
  static char *kwlist[] = {"address","record_field_filters", NULL};
  if (!PyArg_ParseTupleAndKeywords(args, kwds, "|OO", kwlist, 
				   &fromaddress, &filters)) {
    return -1;
  }
  
  address = (PyPiAddress*)self;
  /* we have to support calling __init__ more than once */
  free_Address(&(address->a));
  if (address->saved_br.size > 0 && address->saved_br.buf) {	
    free(address->saved_br.buf);					
  }

  if ((fromaddress == NULL) || ((PyObject *)fromaddress == Py_None)) {
    /* Initialise attributes custom to this type of object */
    new_Address(&(address->a));
    SetBasicRecordObjectAttributeDefaults((PyObject*) address, pack_Address);    
  } else {
    if (!PyPiAddress_Check(fromaddress)) {
      PyErr_SetString(PyExc_TypeError,"Must provide a Address object to share");
      return -1;
    }
  
    /* copy all the database agnostic record attributes */
    address->saved_br.size = fromaddress->saved_br.size;
    address->saved_br.attrib = fromaddress->saved_br.attrib;
    address->saved_br.rt = fromaddress->saved_br.rt;
    address->saved_br.unique_id = fromaddress->saved_br.unique_id;

    address->rt = fromaddress->rt;
    address->unique_id = fromaddress->unique_id;
    
    address->saved_br.buf = malloc(fromaddress->saved_br.size);
    memcpy(address->saved_br.buf,
	   fromaddress->saved_br.buf,
	   fromaddress->saved_br.size);
    
    address->category = fromaddress->category;
    address->unsaved_changes = fromaddress->unsaved_changes;
    address->deleted = fromaddress->deleted;
    address->modified = fromaddress->modified;
    address->busy = fromaddress->busy;
    address->secret = fromaddress->secret;
    
    /* copy non-pointer data */
    memcpy(&(address->a), &(fromaddress->a), sizeof(struct Address));
    
    /* Now do the pointers */
    for(i=0;i<19;i++) {
      if (fromaddress->a.entry[i]) {
	if ((address->a.entry[i] = malloc(strlen(fromaddress->a.entry[i])+1)) == NULL) {
	  malloc_failed = 1;
	} else {
	  strcpy(address->a.entry[i],fromaddress->a.entry[i]);
	}
      } else {
	address->a.entry[i] = NULL;
      }
    }
    
    if (malloc_failed) { /* Oh well, undo the mallocs that worked */
      for(i=0;i<19;i++) {
	if (fromaddress->a.entry[i]) {
	  free(fromaddress->a.entry[i]);
	}
      }
      PyErr_SetString(PyExc_MemoryError,"Unable to allocate memory for address entires");
      return -1;
    }
  }

  return 0;
}

static PyObject * PyPiAddress_Allocate(PyTypeObject *type, int nitems) {
  PyPiAddress *address;
  if (type == &AddressType) {
    address = PyObject_New(PyPiAddress, &AddressType);
    return (PyObject *) address;
  } else {
    /* Is this for subclasses ? */
    address = (PyPiAddress *)PyType_GenericAlloc(type, nitems);
    return (PyObject*)address;
  }
}

/**
 * Wrap an existing Address in a jppy address. This makes
 * a copy of a into a newly allocated struct inside the python object.
 *
 * @param a the palm object
 * @param rt
 * @param unique_id
 * @param attrib
 * @param size
 * @param buf
 * @return a newly created python event object that represents the data in a
 */
extern PyObject* PyPiAddress_Wrap(struct Address *a, PCRecType rt, 
				 unsigned int unique_id, unsigned char attrib,
				 int size, void* buf, PyObject *filters) {
  PyPiAddress* address;
  int i;
  int malloc_failed = 0;
  

 PyObject *python_mod, *python_mdict, *address_class, *python_args, *python_kw;
  python_mod = PyImport_Import(PyString_FromString("jppy.jpilot.legacy"));
  if (python_mod == NULL) {
    PyErr_Print();
    return NULL;
  }
  python_mdict = PyModule_GetDict(python_mod);
  if (python_mdict == NULL) {
    PyErr_Print();
    Py_DECREF(python_mod);
    return NULL;
  }  
  Py_INCREF(python_mdict);
  Py_DECREF(python_mod);
  address_class = PyDict_GetItemString(python_mdict, "Address"); /* borrowed reference */
  if (address_class == NULL) {
    PyErr_Print();
    Py_DECREF(python_mdict);
    return NULL;
  }
  Py_INCREF(address_class);
  python_args = Py_BuildValue("()");
  python_kw = Py_BuildValue("{s:O}","record_field_filters",filters);
  address = (PyPiAddress*) PyObject_Call(address_class, python_args, python_kw);
  Py_DECREF(address_class);
  Py_DECREF(python_args);
  Py_DECREF(python_kw);
  if (address == NULL) {
    PyErr_Print();
    return NULL;
  }
  Py_INCREF(address);

  /* copy birthday stuff, phoneLabels and showPhone */
  memcpy(&(address->a), a, sizeof(struct Address));

  /* set saved_br stuff, and rt and unique_id, and attrib derived details for the current address */
  SetSavedBrAndRTandUniqueIDandAttribs(rt, unique_id, attrib, size, buf, (PyObject *)address);

  for(i=0;i<19;i++) {
    if (a->entry[i]) {
      if ((address->a.entry[i] = malloc(strlen(a->entry[i])+1)) == NULL) {
	malloc_failed = 1;
      } else {
	strcpy(address->a.entry[i],a->entry[i]);
      }
    } else {
      address->a.entry[i] = NULL;
    }
  }

  if (malloc_failed) { /* Oh well, undo the mallocs that worked */
    for(i=0;i<19;i++) {
      if (address->a.entry[i]) {
	free(address->a.entry[i]);
      }
    }
    PyErr_SetString(PyExc_MemoryError,"Unable to allocate memory for address entires");
    return NULL;	    
  }

  return (PyObject*)address;
}

static void PyPiAddress_Dealloc(PyPiAddress* self) {
  free_Address(&(self->a));
  if (self->filters != NULL) {
    Py_DECREF(self->filters);
    self->filters = NULL;
  }
  if (self->saved_br.size > 0 && self->saved_br.buf) {
    free(self->saved_br.buf);
  }
  self->ob_type->tp_free((PyObject*)self);
}


#define JPPY_ADDRESS_SORT_FIELD contLastname
static int PyPiAddress_Compare(PyPiAddress* self,PyPiAddress *other) {
  int res;

  if ((self->a.entry[JPPY_ADDRESS_SORT_FIELD]) &&
      (other->a.entry[JPPY_ADDRESS_SORT_FIELD])) {
    
    res = strcasecmp(self->a.entry[JPPY_ADDRESS_SORT_FIELD],
		 other->a.entry[JPPY_ADDRESS_SORT_FIELD]);
    if (res > 0) {
      res = 1;
    } else if (res < 0) {
      res = -1;
    } else {
      res = 0;
    }
  } else if (self->a.entry[JPPY_ADDRESS_SORT_FIELD]) {
    res = -1;
  } else if (other->a.entry[JPPY_ADDRESS_SORT_FIELD]) {
    res = 1;
  } else {
    res = 0;
  }

  return res;

}

static char *PyPiAddress_key_list[] = {
  "lastname",
  "firstname",
  "company",
  "title",
  "phone1",
  "phone2",
  "phone3",
  "phone4",
  "phone5",
  "address1",
  "city1",
  "state1",
  "zip1",
  "country1",
  "custom1",
  "custom2",
  "custom3",
  "custom4",
  "note",
  "currentphone",
  "showphone",
  NULL};

static PyObject* PyPiAddress_keys(PyObject* self) {
  PyObject *list = PyList_New(0);
  int n = 0;

  while (PyPiAddress_key_list[n]) {
    PyObject *value;
    value = PyString_FromString(PyPiAddress_key_list[n++]);
    PyList_Append(list, value);
    Py_DECREF(value);
  }
  PyPi_extend_keys_from_filters((PyPiBase*)self, list);
  return list;
}

PyObject *PyPiAddress_GetItem(PyPiAddress* self,  PyObject* key);

static PyObject* PyPiAddress_values(PyObject* self) {
  PyObject *list = PyList_New(0);
  int n = 0;

  while (PyPiAddress_key_list[n]) {
    PyObject *key;
    PyObject *value;
    key   = PyString_FromString(PyPiAddress_key_list[n++]);
    value = PyPiAddress_GetItem((PyPiAddress *)self, key);
    PyList_Append(list, value);
    Py_DECREF(key);
    Py_DECREF(value);
  }
  return list;
}

static PyObject* PyPiAddress_items(PyObject* self) {
  PyObject *list = PyList_New(0);
  int n = 0;

  while (PyPiAddress_key_list[n]) {
    PyObject *key, *value, *tuple;
    key = PyString_FromString(PyPiAddress_key_list[n++]);
    value = PyPiAddress_GetItem((PyPiAddress *)self, key);
    tuple = Py_BuildValue("(OO)", key, value);
    PyList_Append(list, tuple); /* get it's own ref */
    Py_DECREF(key);
    Py_DECREF(value);
    Py_DECREF(tuple);
  }
  return list;
}


/*******************************************************
 * Delete (from file), save and log methods for the Address objects
 ******************************************************/

static PyObject* PyPiAddress_Log(PyPiAddress* self, PyObject *args) {
  char *newnote;
  char *type,*entry,*p;
  time_t tt;

  entry = NULL;

  if (!PyArg_ParseTuple(args, "s|s:log", &type, &entry))
    return NULL;
  
  if(entry==NULL) {
    entry=type;
    type=NULL;
  }
  
  tt = time(NULL);
  p = ctime(&tt);
  if (p[24] == '\n')
    p[24] = '\0';

  if(self->a.entry[contNote] == NULL) {
    self->a.entry[contNote] = malloc(1);
    self->a.entry[contNote][0] = '\0';
  }

  newnote = malloc(4096);
  if ((newnote == NULL) || (self->a.entry[contNote] == NULL)) {
    PyErr_SetString(PyExc_MemoryError,"Unable to allocate memory for new note string");
    return NULL;	    
  }

  if(type) 
    snprintf(newnote,4096,"%s\n\n%s\n%s: %s", self->a.entry[contNote], p, type, entry);
  else
    snprintf(newnote,4096,"%s\n\n%s\n%s", self->a.entry[contNote], p, entry);    

  free(self->a.entry[contNote]);
  self->a.entry[contNote] = newnote;

  self->unsaved_changes = 1;
  Py_INCREF(Py_None);
  return Py_None;  
}

/************************
 * vCard generation
 ************************/


static PyObject* PyPiAddress_Generate_VCard(PyPiAddress* self, PyObject *args, PyObject *kw) {
  VObject *prop;
  VObject *vcard;
  char strbuf[255];
  char* clipboard;
  int i;
  int minimal = 0;

  char *keywords[] = {"minimal", NULL};

  if (!PyArg_ParseTupleAndKeywords(args, kw, "|i:vcard", keywords, &minimal))
    return NULL;  
  
  vcard = newVObject(VCCardProp);

  safeAddPropValue(vcard,VCVersionProp,"2.1");

  if ((self->a.entry[contFirstname]) || (self->a.entry[contLastname])) {
    prop = addProp(vcard,VCNameProp);
    
      safeAddPropValue(prop,VCFamilyNameProp,self->a.entry[contLastname]);
      safeAddPropValue(prop,VCGivenNameProp,self->a.entry[contFirstname]);
    
      if (1) {
	/* Used to have if (full) */
	if ((self->a.entry[contFirstname]) && (self->a.entry[contLastname])) {
	  snprintf(strbuf,254,"%s %s", self->a.entry[contFirstname], self->a.entry[contLastname]);
	  safeAddPropValue(vcard,VCFullNameProp,strbuf);
	} else if (self->a.entry[contFirstname]) {
	  safeAddPropValue(vcard,VCFullNameProp,self->a.entry[contFirstname]);
	}  else if (self->a.entry[contLastname]) {
	  safeAddPropValue(vcard,VCFullNameProp,self->a.entry[contLastname]);
	}
      }
  }

  prop = addProp(vcard,VCOrgProp);
  safeAddPropValue(prop,VCOrgNameProp,self->a.entry[contCompany]);
  safeAddPropValue(vcard,VCTitleProp,self->a.entry[contTitle]);

  for (i=0;i<5;i++) {
    if (self->a.entry[entryPhone1 + i]) {
      switch(self->a.phoneLabel[i]) {
      case 0:
	prop = safeAddPropValue(vcard,VCTelephoneProp,self->a.entry[entryPhone1 + i]);
	addProp(prop,VCWorkProp);
	continue;
      case 1:
	prop = safeAddPropValue(vcard,VCTelephoneProp,self->a.entry[entryPhone1 + i]);
	addProp(prop,VCHomeProp);
	continue;
      case 2:
	prop = safeAddPropValue(vcard,VCTelephoneProp,self->a.entry[entryPhone1 + i]);
	addProp(prop,VCFaxProp);
	continue;
      case 3:
	safeAddPropValue(vcard,VCTelephoneProp,self->a.entry[entryPhone1 + i]); /* other ? */
	continue;
      case 4:
	prop = safeAddPropValue(vcard,VCEmailAddressProp,self->a.entry[entryPhone1 + i]);
	addProp(prop,VCInternetProp);
	continue;
      case 5:
	safeAddPropValue(vcard,VCTelephoneProp,self->a.entry[entryPhone1 + i]); /* main */
	continue;
      case 6:
	prop = safeAddPropValue(vcard,VCTelephoneProp,self->a.entry[entryPhone1 + i]);
	addProp(prop,VCPagerProp);
	continue;
      case 7:
	prop = safeAddPropValue(vcard,VCTelephoneProp,self->a.entry[entryPhone1 + i]);
	addProp(prop,VCCellularProp);
	continue;
      }
    }
  }
  
  if (!minimal) {
    char date_string[9];
    if (self->a.entry[contNote]) {
      prop = safeAddPropValue(vcard,VCCommentProp,self->a.entry[contNote]);
      addProp(prop,VCQuotedPrintableProp);
    }
  
    prop = addProp(vcard, VCAdrProp);
    safeAddPropValue(prop, VCStreetAddressProp, self->a.entry[entryAddress]);
    safeAddPropValue(prop, VCCityProp, self->a.entry[entryCity]);
    safeAddPropValue(prop, VCRegionProp, self->a.entry[entryState]);
    safeAddPropValue(prop, VCPostalCodeProp, self->a.entry[entryZip]);
    safeAddPropValue(prop, VCCountryNameProp, self->a.entry[entryCountry]);


    for (i=0;i<4;i++) {  
      if (self->a.entry[entryCustom1 + i]){
	snprintf(strbuf,20,"X-Palm-Custom%d",i + 1);
	prop = safeAddPropValue(vcard,strbuf,self->a.entry[entryCustom1 + i]);
	addProp(prop,VCQuotedPrintableProp);
      }
    }
  }
  clipboard = writeMemVObject(0,0,vcard);
  return PyString_FromString(clipboard);
}



static PyMethodDef PyPiAddress_Methods[] = {
  { "log",  (PyCFunction)PyPiAddress_Log,    METH_VARARGS, "Make a log entry"},
  { "vcard",(PyCFunction)PyPiAddress_Generate_VCard, METH_KEYWORDS, "Generate a vcard"},
  { "keys", (PyCFunction)PyPiAddress_keys, METH_NOARGS, "Return a list of available keys"},
  { "items",(PyCFunction)PyPiAddress_items, METH_NOARGS, "Return a list of available items"},
  { "values",(PyCFunction)PyPiAddress_values, METH_NOARGS, "Return a list of available items"},
  {NULL,NULL,0,NULL}
};


static PyMemberDef PyPiAddress_Members[] = {
  PYPI_MEMBERS_HEAD,
  {NULL}  /* Sentinel */
};

static PyGetSetDef PyPiAddress_Getseters[] = {
  PYPI_GETSETERS_HEAD,
  {NULL}  /* Sentinel */
};

/**** mapping interface ****/
int PyPiAddress_Len(PyObject* self) {
  int len=0;
  // quite expensive way to do it, but we need to get filters too
  PyObject *keys = PyPiAddress_keys(self);
  len = PySequence_Size(keys);
  Py_DECREF(keys);
  return len;
}


PyObject *PyPiAddress_GetItem(PyPiAddress* self,  PyObject* key) {
  char *keystring;
  PyObject *result;
  PyObject* py_list;
  int n, i;

  if (!PyString_Check(key)) {
    Py_INCREF(Py_None);
    return Py_None;  
  }

  if ((result = PyPi_GetItem_from_filters((PyPiBase *)self, key)) != NULL)
    return result;
  else if (PyErr_Occurred() != NULL)
    return NULL;


  Py_INCREF(key);
  keystring = PyString_AsString(key);

  GET_STRING_ATTR(keystring,"lastname", a.entry[entryLastname]);
  GET_STRING_ATTR(keystring,"firstname", a.entry[entryFirstname]);
  GET_STRING_ATTR(keystring,"company", a.entry[entryCompany]);
  GET_STRING_ATTR(keystring,"title", a.entry[entryTitle]);

  GET_STRING_ATTR(keystring,"phone1", a.entry[entryPhone1]);
  GET_STRING_ATTR(keystring,"phone2", a.entry[entryPhone2]);
  GET_STRING_ATTR(keystring,"phone3", a.entry[entryPhone3]);
  GET_STRING_ATTR(keystring,"phone4", a.entry[entryPhone4]);
  GET_STRING_ATTR(keystring,"phone5", a.entry[entryPhone5]);

  GET_STRING_ATTR(keystring,"address1", a.entry[entryAddress]);
  GET_STRING_ATTR(keystring,"city1", a.entry[entryCity]);
  GET_STRING_ATTR(keystring,"state1", a.entry[entryState]);
  GET_STRING_ATTR(keystring,"zip1", a.entry[entryZip]);
  GET_STRING_ATTR(keystring,"country1", a.entry[entryCountry]);

  GET_STRING_ATTR(keystring,"custom1", a.entry[entryCustom1]);
  GET_STRING_ATTR(keystring,"custom2", a.entry[entryCustom2]);
  GET_STRING_ATTR(keystring,"custom3", a.entry[entryCustom3]);
  GET_STRING_ATTR(keystring,"custom4", a.entry[entryCustom4]);

  GET_STRING_ATTR(keystring,"note", a.entry[entryNote]);

  GET_STRING_ATTR(keystring,"currentphone",a.entry[self->a.showPhone+entryPhone1]);
  GET_INT_ATTR(keystring,"showphone",a.showPhone);

  GET_INT_ATTR(keystring,"type1",a.phoneLabel[0]);
  GET_INT_ATTR(keystring,"type2",a.phoneLabel[1]);
  GET_INT_ATTR(keystring,"type3",a.phoneLabel[2]);
  GET_INT_ATTR(keystring,"type4",a.phoneLabel[3]);
  GET_INT_ATTR(keystring,"type5",a.phoneLabel[4]);

  if (strcasecmp(keystring,"phones") == 0) {
    py_list = PyList_New(5);
    PyList_SET_ITEM(py_list, 0, PyPiAddress_GetItem(self, PyString_FromString("phone1")));
    PyList_SET_ITEM(py_list, 1, PyPiAddress_GetItem(self, PyString_FromString("phone2")));
    PyList_SET_ITEM(py_list, 2, PyPiAddress_GetItem(self, PyString_FromString("phone3")));
    PyList_SET_ITEM(py_list, 3, PyPiAddress_GetItem(self, PyString_FromString("phone4")));
    PyList_SET_ITEM(py_list, 4, PyPiAddress_GetItem(self, PyString_FromString("phone5")));
    Py_DECREF(key);
    return py_list;
  }  

  if (strcasecmp(keystring,"phones_with_labels") == 0) {
    py_list = PyList_New(5);
    PyList_SET_ITEM(py_list, 0, Py_BuildValue("(Oi)",PyPiAddress_GetItem(self, PyString_FromString("phone1")),
					      self->a.phoneLabel[0]));
    PyList_SET_ITEM(py_list, 1, Py_BuildValue("(Oi)",PyPiAddress_GetItem(self, PyString_FromString("phone2")),
					      self->a.phoneLabel[1]));
    PyList_SET_ITEM(py_list, 2, Py_BuildValue("(Oi)",PyPiAddress_GetItem(self, PyString_FromString("phone3")),
					      self->a.phoneLabel[2]));
    PyList_SET_ITEM(py_list, 3, Py_BuildValue("(Oi)",PyPiAddress_GetItem(self, PyString_FromString("phone4")),
					      self->a.phoneLabel[3]));
    PyList_SET_ITEM(py_list, 4, Py_BuildValue("(Oi)",PyPiAddress_GetItem(self, PyString_FromString("phone5")),
					      self->a.phoneLabel[4]));
    Py_DECREF(key);
    return py_list;
  } 

  n = -1;
  if (strcasecmp(keystring,"email") == 0) {
    n = 4;
  } else if (strcasecmp(keystring,"mobile") == 0) {
    n = 7;
  }
  if (n > 0) {
    for(i=0;i<5;i++) {
      if (n == self->a.phoneLabel[i]) {
	if (self->a.entry[entryPhone1 + i] != NULL) {
	  Py_DECREF(key);
	  return PyString_FromString(self->a.entry[entryPhone1 + i]);
	}
      }
    }
    Py_DECREF(key);
    Py_INCREF(Py_None);
    return Py_None;  
  }
  
  PyErr_Format(PyExc_KeyError,"no such key '%s'", keystring);
  Py_DECREF(key);
  return NULL;
}

int PyPiAddress_SetItem(PyPiAddress* self, PyObject* key, PyObject* value) {
  char buf[255];
  char *keystring;
  PyObject *pytup, *pyint, *pystr;
  int n;

  if (!PyString_Check(key)) {
    PyErr_SetString(PyExc_TypeError,"key must be a String");
    return -1;
  }

  if (PyPi_SetItem_from_filters((PyPiBase *)self, key, value) > 0)
    return 0;
  else if (PyErr_Occurred() != NULL)
    return -1;

  Py_INCREF(key);
  keystring = PyString_AsString(key);

  if (value == NULL) {
    PyErr_Format(PyExc_ValueError,"Can't delete value %s", keystring);
    return -1;
  }
  
  SET_STRING_ATTR(keystring,"lastname",a.entry[entryLastname],value, 4096); 
  SET_STRING_ATTR(keystring,"firstname",a.entry[entryFirstname],value, 4096);
  SET_STRING_ATTR(keystring,"company",a.entry[entryCompany],value, 4096);
  SET_STRING_ATTR(keystring,"title",a.entry[entryTitle],value, 4096);

  SET_STRING_ATTR(keystring,"phone1",a.entry[entryPhone1],value, 4096);
  SET_STRING_ATTR(keystring,"phone2",a.entry[entryPhone2],value, 4096);
  SET_STRING_ATTR(keystring,"phone3",a.entry[entryPhone3],value, 4096);
  SET_STRING_ATTR(keystring,"phone4",a.entry[entryPhone4],value, 4096);
  SET_STRING_ATTR(keystring,"phone5",a.entry[entryPhone5],value, 4096);

  SET_STRING_ATTR(keystring,"address1",a.entry[entryAddress],value, 4096);
  SET_STRING_ATTR(keystring,"city1",a.entry[entryCity],value, 4096);
  SET_STRING_ATTR(keystring,"state1",a.entry[entryState],value, 4096);
  SET_STRING_ATTR(keystring,"zip1",a.entry[entryZip],value, 4096);
  SET_STRING_ATTR(keystring,"country1",a.entry[entryCountry],value, 4096);

  SET_STRING_ATTR(keystring,"custom1",a.entry[entryCustom1],value, 4096);
  SET_STRING_ATTR(keystring,"custom2",a.entry[entryCustom2],value, 4096);
  SET_STRING_ATTR(keystring,"custom3",a.entry[entryCustom3],value, 4096);
  SET_STRING_ATTR(keystring,"custom4",a.entry[entryCustom4],value, 4096);

  SET_STRING_ATTR(keystring,"note",a.entry[entryNote],value, 32768);
  SET_STRING_ATTR(keystring,"currentphone",a.entry[self->a.showPhone+entryPhone1],value, 4096);

  SET_BOUNDED_INT_ATTR(keystring,"type1",a.phoneLabel[0],value,0,7,buf,255);
  SET_BOUNDED_INT_ATTR(keystring,"type2",a.phoneLabel[1],value,0,7,buf,255);
  SET_BOUNDED_INT_ATTR(keystring,"type3",a.phoneLabel[2],value,0,7,buf,255);
  SET_BOUNDED_INT_ATTR(keystring,"type4",a.phoneLabel[3],value,0,7,buf,255);
  SET_BOUNDED_INT_ATTR(keystring,"type5",a.phoneLabel[4],value,0,7,buf,255);

  SET_BOUNDED_INT_ATTR(keystring,"showphone",a.showPhone,value,0,4,buf,255);

  if (strcasecmp(keystring,"phones_with_labels") == 0) {
    if (!PyList_Check(value)) {
      PyErr_SetString(PyExc_TypeError,"phone_with_labels must be set to a list");
      Py_DECREF(key);
      return -1;
    }
    if (PyList_GET_SIZE(value) != 5) {
      PyErr_SetString(PyExc_IndexError,"phone_with_labels must have 5 elements");
      Py_DECREF(key);
      return -1;
    }

    for (n=0;n<5;n++) {
      pytup = PyList_GET_ITEM(value,n);
      if (!PyTuple_Check(pytup)) {
	PyErr_SetString(PyExc_TypeError,"phone_with_labels must be set to a list containing tuples");
	Py_DECREF(key);
	return -1;
      }
      if (PyTuple_Size(pytup) != 2) {
	PyErr_SetString(PyExc_IndexError,"phone_with_labels' tuples must contain two elements");
	Py_DECREF(key);
	return -1;
      }      
      pystr = PyTuple_GET_ITEM(pytup,0);
      pyint = PyTuple_GET_ITEM(pytup,1);
      if (!PyInt_Check(pyint) || !PyString_Check(pystr) || 
	  (PyInt_AS_LONG(pyint) > 7 || PyInt_AS_LONG(pyint) < 0)) {
	PyErr_SetString(PyExc_TypeError,
			"phone_with_labels' tuples must be (string, int) where int 0<=x<=7");
	Py_DECREF(key);
	return -1;
      }
      self->a.entry[entryPhone1+n] = realloc(self->a.entry[entryPhone1+n],
					     strlen(PyString_AS_STRING(pystr))+1);
      if(self->a.entry[entryPhone1+n] != NULL) {
	strcpy(self->a.entry[entryPhone1+n],PyString_AS_STRING(pystr));
	self->a.phoneLabel[n] = PyInt_AS_LONG(pyint);
	self->unsaved_changes = 1;
      } else {
	PyErr_SetString(PyExc_MemoryError,
			"Unable to save attribute, out of memory (realloc failed)!");
	Py_DECREF(key);
	return -1;
      };
      
    }
    Py_DECREF(key);
    return 0;
  }

  PyErr_SetString(PyExc_KeyError,"no such key");
  Py_DECREF(key);
  return -1;
}

static PyMappingMethods PyPiAddress_Mapping = {
  (lenfunc)PyPiAddress_Len,
  (binaryfunc)PyPiAddress_GetItem, 
  (objobjargproc)PyPiAddress_SetItem,
};



/*******************************************************
 * Provide a repr method
 ******************************************************/

static PyObject *PyPiAddress_Repr(PyPiAddress* self) {
  static PyObject *format = NULL;
  PyObject *attrib, *args, *result;
  int len1, len2;

  if (format == NULL) {
    format = PyString_FromString("<%s %r %r %s>");
    if (format == NULL)
      return NULL;
  }
  
  if (self->a.entry[contFirstname]) {
    len1 = strlen(self->a.entry[contFirstname]) > 25 ? 25 : strlen(self->a.entry[contFirstname]);
  } else {
    len1 = 0;
  }

  if (self->a.entry[contLastname]) {
    len2 = strlen(self->a.entry[contLastname]) > 25 ? 25 : strlen(self->a.entry[contLastname]);
  } else {
    len2 = 0;
  }

  args = Py_BuildValue("ss#s#O", 
		       (self->ob_type)->tp_name,
		       self->a.entry[contFirstname],
		       len1,
		       self->a.entry[contLastname],
		       len2,
		       Attribute_Repr((PyObject *)self));

  if (args == NULL)
    return NULL;

  result = PyString_Format(format, args);
  Py_DECREF(args);
  return result;
}


/*******************************************************
 * Declare the type
 ******************************************************/


PyTypeObject AddressType = {
  PyObject_HEAD_INIT(NULL)
  0,
  "jppy._jpilot.__jpilot.Address",
  sizeof(PyPiAddress),
  0,
  (destructor)PyPiAddress_Dealloc,      /*tp_dealloc*/
  0,                                /*tp_print*/
  0, /*tp_getattr*/
  0, /*tp_setattr*/
  (cmpfunc)PyPiAddress_Compare,     /*tp_compare*/
  (reprfunc)PyPiAddress_Repr,       /*tp_repr*/
  0,                                /*tp_as_number*/
  0,                                /*tp_as_sequence*/
  &PyPiAddress_Mapping,                                /*tp_as_mapping*/
  0,                                /*tp_hash */
  0,                         /*tp_call*/
  0,                         /*tp_str*/
  0,                         /*tp_getattro*/
  0,                         /*tp_setattro*/
  0,                         /*tp_as_buffer*/
  Py_TPFLAGS_DEFAULT | Py_TPFLAGS_BASETYPE, /*tp_flags*/
  "Address objects",           /* tp_doc */
  0,		               /* tp_traverse */
  0,		               /* tp_clear */
  0,		               /* tp_richcompare */
  0,		               /* tp_weaklistoffset */
  0,		               /* tp_iter */
  0,		               /* tp_iternext */
  PyPiAddress_Methods,             /* tp_methods */
  PyPiAddress_Members,            /* tp_members */
  PyPiAddress_Getseters,          /* tp_getset */
  0,                         /* tp_base */
  0,                         /* tp_dict */
  0,                         /* tp_descr_get */
  0,                         /* tp_descr_set */
  0,                         /* tp_dictoffset */
  (initproc)PyPiAddress_Init,      /* tp_init */
  (allocfunc)PyPiAddress_Allocate,                 /* tp_alloc */
  (newfunc)PyPiAddress_New,                 /* tp_new */
  0, /* Low-level free-memory routine */
};
