import re
import types
import datetime
import pytz
from jppy.core import Component, ExtensionPoint, Interface, implements, JppyError
from jppy.config import ExtensionOption
import jppy.vcard
import jppy.ical

class IDatabaseAccessor(Interface):
    def records(self,
                search=None,
                sort=None,
                modified=None,
                deleted=None,
                privates=None,
                categories=None):
        pass

    def new(self, *args, **kwargs):
        pass
    
    def fetch(self,uid):
        pass

    def getCategories(self):
        pass

    def save(self, record):
        pass

    def delete(self, record):
        pass

class IAddressBook(IDatabaseAccessor):
    def getLabels(self):
        pass

    def getContactLabels(self):
        pass
    
    def getPhoneLabels(self):
        pass

    def getIMLabels(self):
        pass

    def getAddressTypeLabels(self):
        pass

    def getAddressLabels(self):
        pass

    def loadVCards(self, stream):
        pass

class ITaskList(IDatabaseAccessor):
    pass

class IMemoList(IDatabaseAccessor):
    pass

class ICalendar(IDatabaseAccessor):
    def eventsFromDate(self, date, **kwargs):
        pass
    
    def eventsToday(self):
        pass

    def eventsThisWeek(self):
        pass

    def eventsOnDay(self, day):
        pass

class IInstallFile(Interface):
    def install(self, filename):
        pass

class IRecordExtension(Interface):
    def fields(self):
        """Return a list of fields this extension handles"""

    def handles_field(self, field):
        """Query to see if this extension wants to handle this field"""

    def get_field(self, record, fieldname):
        """Read a field from record"""

    def set_field(self, record, fieldname, value):
        """Set a field on a record"""

class IAddressExtension(IRecordExtension):
    pass

class IMemoExtension(IRecordExtension):
    pass

class ITodoExtension(IRecordExtension):
    pass

class IEventExtension(IRecordExtension):
    pass

class databaseAccessor(Component):
    implements(IDatabaseAccessor)
    record_field_filters = ExtensionPoint(IRecordExtension)
    
    showModified   = 0
    showDeleted    = 0
    showPrivates   = 0
    showCategories = []
    showSorted     = 1
    searchString   = ""
    
    def __iter__(self):
        return iter(self.records())

    def __getitem__(self, key):
        return self.records()[key]

    def new(self, *args, **kwargs):
        kwargs['record_field_filters'] = self.record_field_filters
        return self._record_class(*args, **kwargs)

    def records(self,
                search=None,
                sort=None,
                modified=None,
                deleted=None,
                privates=None,
                categories=None):
        if search == None: search  = re.compile(self.searchString)
        if sort   == None: sort    = self.showSorted
        if modified==None: modified= self.showModified
        if deleted== None: deleted = self.showDeleted
        if categories==None: categories = self.showCategories[:]
        if privates==None: privates= self.showPrivates
        if not hasattr(search,'search'):
            search = re.compile(str(search),re.I)
        self._metadata = self._appInfoFunction(self._database)            
        return self._readDatabaseFunction(search,0,sort,modified,
                                          deleted,privates,categories,
                                          self._database,self._metadata['_storageversion'],
                                          self.record_field_filters)
    
    def fetch(self,uid):
        self._metadata = self._appInfoFunction(self._database)                    
        return self._readDatabaseFunction(None,uid,1,0,0,1,[],
                                          self._database,self._metadata['_storageversion'],
                                          self.record_field_filters)

    def getCategories(self):
        self._metadata = self._appInfoFunction(self._database)
        return self._metadata['categories']

    def save(self, record):
        self._metadata = self._appInfoFunction(self._database)
        return self._writeDatabaseFunction(record, self._database,
                                           self._metadata['_storageversion'])

    def delete(self, record):
        return self._deleteFromDatabaseFunction(record, self._database)

class addressBook(databaseAccessor):
    record_field_filters = ExtensionPoint(IAddressExtension)
    
    def getLabels(self):
        self._metadata = self._appInfoFunction(self._database)        
        return self._metadata['customlabels']

    def getContactLabels(self):
        self._metadata = self._appInfoFunction(self._database)        
        return self._metadata['contactlabels']

    def getPhoneLabels(self):
        self._metadata = self._appInfoFunction(self._database)        
        return self._metadata['phonelabels']

    def getIMLabels(self):
        self._metadata = self._appInfoFunction(self._database)        
        return self._metadata['imlabels']

    def getAddressTypeLabels(self):
        self._metadata = self._appInfoFunction(self._database)        
        return self._metadata['addresstypelabels']

    def getAddressLabels(self):
        self._metadata = self._appInfoFunction(self._database)        
        return self._metadata['addresslabels']

    def loadVCards(self, stream):
        loader = jppy.vcard.vcardLoader(self.env)
        return loader.parse(stream)

class memoList(databaseAccessor):
    record_field_filters = ExtensionPoint(IMemoExtension)

class memo32List(databaseAccessor):
    record_field_filters = ExtensionPoint(IMemoExtension)

class taskList(databaseAccessor):
    record_field_filters = ExtensionPoint(ITodoExtension)

class calendar(databaseAccessor):
    record_field_filters = ExtensionPoint(IEventExtension)


    def eventsFromDate(self, date, **kwargs):
        events = list()
        bandSize = datetime.timedelta(**kwargs)
        startBound_tz = date.replace(hour=0,
                                     minute=0,
                                     second=0,
                                     microsecond=0,
                                     tzinfo=pytz.utc)
        startBound_tz = startBound_tz - datetime.timedelta(seconds=1)
        endBound_tz = (date + bandSize).replace(hour=0,
                                                minute=0,
                                                second=0,
                                                microsecond=0,
                                                tzinfo=pytz.utc)
        startBound = startBound_tz.replace(tzinfo=None)
        endBound   = endBound_tz.replace(tzinfo=None)
        #print "Finding between %s and %s" % (startBound, endBound)
        for e in self:
            try:
                tzinfo = e['dates'][0].tzinfo
            except IndexError:
                tzinfo = None
                
            if tzinfo is None:
                for hit in e['dates'].between(startBound, endBound, inc=False):
                    hit = hit.replace(tzinfo=pytz.utc)
                    events.append((hit,e))
            else:
                for hit in e['dates'].between(startBound_tz, endBound_tz, inc=False):
                    hit = hit.replace(tzinfo=pytz.utc)
                    events.append((hit,e))
        events.sort()
        return events

    def eventsToday(self):
        return self.eventsOnDay(datetime.datetime.now())

    def eventsThisWeek(self):
        startOfWeek = datetime.datetime.now()-datetime.timedelta(
            days=datetime.datetime.now().weekday())
        return self.eventsFromDate(startOfWeek, days=7)

    def eventsOnDay(self, day):
        return self.eventsFromDate(day,days=1)

    def loadiCals(self, stream):
        loader = jppy.ical.icalLoader(self.env)
        return loader.parse(stream)

class installFile(Component):
    def install(self, filename):
        pass

