#
#  memodir_sync.py
#
#  Copyright 2002 Nick Piper <nick@nickpiper.co.uk>
#
#  This program is free software; you can redistribute it and/or modify
#  it under the terms of the GNU Library General Public License, version 2,
#  as published by the Free Software Foundation.
#
#  This program is distributed in the hope that it will be useful, but
#  WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
#  Library General Public License for more details.
#
#  You should have received a copy of the GNU Library General Public License
#  along with this program; if not, write the Free Software Foundation,
#  Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
#

import jppy
import difflib
import os
import tempfile
import types

class MemoDirSync(jppy.conduit):
    def __init__(self,directory):
        self.sync_completed = None
        self.memo_directory = directory
        self.originals_directory = os.path.join(jppy.jpilot_home,"memodb_mds")
        self.memolist = jppy.memoList()

    def memodir(self,memo,path=None):
        if path == None:
            path = self.memo_directory
        if type(memo) in (types.StringType, types.IntType):
            return os.path.join(path,str(memo))
        else:
            return os.path.join(path,str(memo.unique_id))

    def origdir(self,memo):
        return self.memodir(memo,self.originals_directory)

    def changed_from_original(self,memo,filename=None):
        if type(memo) in (types.StringType, types.IntType):
            uid = memo
            data = open(filename,"r").read()
        else:
            uid = memo.unique_id
            data = memo['text']
        if os.path.isfile(self.origdir(uid)):
            orig_data = open(self.origdir(uid),"r").read()
            return orig_data != data
        else:
            return 1

    def merge_changes(self,memo):
        jp_memo_fn = tempfile.mktemp("mds")
        open(jp_memo_fn,"w").write(memo['text'])
        os.system('merge -L "Jpilot" -L "Original" -L "MemoDir" %s %s %s' % (
            jp_memo_fn, self.origdir(memo), self.memodir(memo)))
        merged_text = open(jp_memo_fn,"r").read()
        os.unlink(jp_memo_fn)
        return merged_text

    def delete_memo_tree(self,rootdir):
        if not os.path.isdir(rootdir):
            os.makedirs(rootdir)
        else:
            for fn in os.listdir(rootdir):
                os.unlink(os.path.join(rootdir,fn))

    def write_memo_tree(self,rootdir):
        self.delete_memo_tree(rootdir)
        for memo in self.memolist:
            memo_file = open(os.path.join(rootdir,str(memo.unique_id)),"w")
            memo_file.write(memo.text)
            memo_file.close()

    def log(self,str,sd=None):
        jppy.log("memodir: %s" % str,sd=sd)

    def pre_sync(self):
        self.log("Syncronising with memo directory")
        if not os.path.isdir(self.memo_directory):
            os.makedirs(self.memo_directory)
        if not os.path.isdir(self.originals_directory):
            os.makedirs(self.originals_directory)
        
        for memo in self.memolist:
            if os.path.isfile(self.memodir(memo)):
                if self.changed_from_original(memo.unique_id,self.memodir(memo)):
                    if self.changed_from_original(memo):
                        self.log("Merging %s" % memo)
                        memo['text'] = self.merge_changes(memo)
                    else:
                        self.log("Updating %s" % memo)
                        memo['text'] = open(self.memodir(memo),"r").read()
                    self.memolist.save(memo)
            else:
                if os.path.isfile(self.origdir(memo)):
                    self.log("Deleting %s" % memo)
                    self.memolist.delete(memo)
        self.sync_completed = 1

    def sync(self,sd):
        if self.sync_completed:
            self.log("Synchronised during pre sync",sd=sd)
        else:
            self.log("Failed to pre sync",sd=sd)                        
                    
    def post_sync(self):
        # after the merge has all finished
        self.log("Writing memos to directory trees...")
        self.write_memo_tree(self.memo_directory)
        self.write_memo_tree(self.originals_directory)
        self.log("Done!")
