import MySQLdb
import jppy
import string
import types

from jppy.api import ITaskList
from jppy.conduit import Conduit
from jppy.config import ListOption, Option, BoolOption, ExtensionOption

class BugzillaSync(Conduit):
    """
    To use this conduit, write into your ~/.jppyrc:

    [conduit]
    enabled = jppy.conduits.bugzilla.BugzillaSync
    """    

    bug_status_in = ListOption("bugzilla","to_sync",
                               "UNCONFIRMED, NEW, ASSIGNED, REOPENED")
    url           = Option("bugzilla","url")
    db            = Option("bugzilla","db")
    host          = Option("bugzilla","host")
    user          = Option("bugzilla","user")
    passwd        = Option("bugzilla","passwd")
    email         = ListOption("bugzilla","email")
    category_name = Option("bugzilla","category_name")

    taskList    = ExtensionOption("jppy","taskList",ITaskList,
                                  "jppy.jpilot.legacy.taskList")

    def pre_sync(self):
        self.log.info("Connecting to bugzilla database")        
        try:
            self.sqlcon = MySQLdb.connect(db=self.db,host=self.host,
                                          user=self.user,passwd=self.passwd,
                                          connect_timeout=15)
        except MySQLdb.OperationalError, e:
            self.log.warn(e)
            return
        self.log.info("Connected")
        self.sqlcurs = self.sqlcon.cursor()

        self.sqlcurs.execute(
            "SELECT userid FROM profiles WHERE login_name in (%s)" %
            string.join(["'%s'" % address for address in self.email],", "))
        r = self.sqlcurs.fetchall()
        if len(r) < 1:
            self.log.warn("Unknown user(s) %s to bugzilla" % self.email)
            return
        self.userids = []
        for row in r:
            self.userids.append(str(row[0]))
        self.log.info("Found userids to be %s" % self.userids)

        try:
            self.cat = self.taskList.getCategories().index(self.category_name)
        except ValueError:
            self.log.warn("Unknown category name %s" % self.category_name)
            return
        self.log.info("Found Todo category %s: number %d" % (self.category_name,
                                                        self.cat))
 
        self.sqlcurs.execute("""
            SELECT bug_id, bug_severity, bug_status, short_Desc,
                   priority, product, login_name as reporter_name, component,
                   resolution
            FROM bugs, profiles
            WHERE
                reporter=userid
                AND
                assigned_to in (%s) and bug_status in %s""" % (
                string.join(self.userids,", "),
                self.bug_status_in))
        r = self.sqlcurs.fetchall()

        # we have the data for the new records at this point, so we can
        # delete the old ones safely.

        self.log.info("Deleting old records")
        for todo in self.taskList.records(categories=[self.cat]):
            self.log.info("Deleting %s" % todo)
            self.taskList.delete(todo)

        self.log.info("Writing new records")        
        for bug in r:
            bug_id, bug_severity, bug_status, short_Desc, priority, product, reporter_name, component, resolution = tuple(bug)
            todo = self.taskList.new()
            todo.category = self.cat
            todo['description'] = "%s-%s: %s" % (priority, component, short_Desc)
            todo['complete'] = bug_status == "RESOLVED"
            todo['note'] = """ID: %s
Reporter: %s
Severity: %s
Priority: %s
Product: %s
Component: %s
Resolution: %s
Status: %s
URL: %s

%s""" % (bug_id, reporter_name, priority, bug_severity, product, component, resolution,
         bug_status, "%s?id=%s" % (self.url,bug_id), short_Desc)
            self.taskList.save(todo)
            self.log.info("Created %s" % todo)

        self.log.info("Closing database connection")
        self.sqlcurs.close()
        self.sqlcon.close()
        self.sync_completed = 1        

    def sync(self,sd):
        if self.sync_completed:        
            self.log.info("Synchronised during pre sync")
        else:
            self.log.warn("Failed to pre sync")
        
