# trash_after_install.py: conduit for jppy to delete files that are
# installed during a sync
#
# Copyright (C) 2008 Jon Schewe
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  
# 02110-1301, USA.


import os.path
import os
import subprocess

from jppy.conduit import Conduit
from jppy.core import Component, implements
from jppy.config import Option

class Trash(Conduit):
    '''
    This conduit deletes files that are successfully installed once the
    sync completes.

    To use this conduit, write into your ~/.jppyrc:

    [conduit]
    enabled = jppy.conduits.trash_after_install.Trash

    If you would like to change the delete command from "rm" to something
    else, add the following:
    [trash]
    delete_command = command
    '''
    trash_command = Option("trash","delete_command","rm")

    def _install_queue_filename(self):
        possible_names = ("jpilot_to_install", "jpilot.install")
        for possible_name in possible_names:
            full_filename = os.path.join(os.environ['HOME'],".jpilot",possible_name)
            if os.path.exists(full_filename) and os.path.getsize(full_filename) > 0:
                return full_filename
        self.log.warn("Failed determine jpilot queue filename.")
        return None

    def pre_sync(self):
        self.install_queue_filename = self._install_queue_filename()
        if not self.install_queue_filename:
            return
        to_install = file(self.install_queue_filename)
        self.files_to_install = []
        for line in to_install:
            line = line.strip()
            self.log.info("To install %s" % line)
            self.files_to_install.append(line)

    def post_sync(self):
        # remove files that were installed
        if not self.install_queue_filename:
            return
        new_to_install = file(self.install_queue_filename)
        for line in new_to_install:
            line = line.strip()
            self.log.warn("File not installed %s" % line)
            self.files_to_install.remove(line)

        for installed_file in set(self.files_to_install):
            self.log.info("Installed file, removing with %s %s" % (self.trash_command,
                                                                   installed_file))
            try:
                retcode = subprocess.call(["%s \"%s\"" % (self.trash_command, installed_file)], shell=True)
                if retcode < 0:
                    self.log.warn("%s was terminated by signal: %d while trashing file: %s" % (self.trash_command, -retcode, installed_file))
            except OSError, e:
                self.log.warn("Execution failed on file %s with error %s" % (installed_file, e))
