from yaml import dump

from twisted.internet.defer import fail, succeed
from twisted.web.error import Error
from twisted.web.xmlrpc import Proxy

from juju.errors import EnvironmentNotFound
from juju.lib.testing import TestCase
from juju.providers.orchestra import MachineProvider
from juju.providers.orchestra.machine import OrchestraMachine

CONFIG = {"orchestra-server": "somewhe.re",
          "storage-url": "http://somewhe.re/webdav",
          "orchestra-user": "user",
          "orchestra-pass": "pass",
          "acquired-mgmt-class": "acquired",
          "available-mgmt-class": "available"}


class FindZookeepersTest(TestCase):

    def get_provider(self):
        return MachineProvider("tetrascape", CONFIG)

    def mock_load_state(self, result):
        getPage = self.mocker.replace("twisted.web.client.getPage")
        getPage("http://somewhe.re/webdav/provider-state")
        self.mocker.result(result)

    def assert_no_environment(self):
        provider = self.get_provider()
        d = provider.get_zookeeper_machines()
        self.assertFailure(d, EnvironmentNotFound)
        return d

    def verify_no_environment(self, load_result):
        self.mock_load_state(load_result)
        self.mocker.replay()
        return self.assert_no_environment()

    def test_no_state(self):
        self.verify_no_environment(fail(Error("404")))

    def test_empty_state(self):
        self.verify_no_environment(succeed(dump([])))

    def test_no_hosts(self):
        self.verify_no_environment(succeed(dump({"abc": 123})))

    def test_bad_instance(self):
        self.mock_load_state(succeed(dump({"zookeeper-instances": ["foo"]})))
        proxy_m = self.mocker.mock(Proxy)
        Proxy_m = self.mocker.replace(Proxy, spec=None)
        Proxy_m("http://somewhe.re/cobbler_api")
        self.mocker.result(proxy_m)
        proxy_m.callRemote("get_systems")
        self.mocker.result(succeed([]))
        self.mocker.replay()

        return self.assert_no_environment()

    def test_eventual_success(self):
        self.mock_load_state(succeed(dump({
            "zookeeper-instances": ["bad", "foo", "missing", "bar"]})))
        proxy_m = self.mocker.mock(Proxy)
        Proxy_m = self.mocker.replace(Proxy, spec=None)
        Proxy_m("http://somewhe.re/cobbler_api")
        self.mocker.result(proxy_m)
        for _ in range(4):
            proxy_m.callRemote("get_systems")
            self.mocker.result(succeed([
                {"uid": "bad", "mgmt_classes": ["whatever"]},
                {"uid": "foo", "mgmt_classes": ["acquired"], "name": "foo"},
                {"uid": "bar", "mgmt_classes": ["acquired"], "name": "bar"}]))
        self.mocker.replay()

        provider = self.get_provider()
        d = provider.get_zookeeper_machines()

        def verify_machine(machines):
            (foo, bar) = machines
            self.assertTrue(isinstance(foo, OrchestraMachine))
            self.assertEquals(foo.instance_id, "foo")
            self.assertTrue(isinstance(bar, OrchestraMachine))
            self.assertEquals(bar.instance_id, "bar")
        d.addCallback(verify_machine)
        return d
