"""Infrastructure shared by all tests for OpenStack provider"""

import json

from twisted.internet import (
    defer,
    )
from twisted.web import (
    http,
    http_headers,
    )

from juju import errors
from juju.lib import (
    mocker,
    )

from juju.providers.openstack import (
    client,
    files,
    machine,
    ports,
    provider,
    )

from juju.machine.constraints import ConstraintSet


class FakeResponse(object):

    def __init__(self, code):
        if code not in http.RESPONSES:
            raise ValueError("Unknown http code: %r" % (code,))
        self.code = code
        self.headers = http_headers.Headers(
            {"Content-Type": ["application/json"]})


class ConstraintSupportMixin(object):

    provider_type = "openstack"

    default_flavors = [
            {'id': 1, 'name': "standard.xsmall", 'vcpus': 1, 'ram': 1024},
            {'id': 2, 'name': "standard.medium", 'vcpus': 2, 'ram': 4096},
        ]

    def setup_constraints(self):
        self.constraint_set = ConstraintSet(self.provider_type)
        self.constraint_set.register_generics(
            [f['name'] for f in self.default_flavors])


class OpenStackTestMixin(ConstraintSupportMixin):
    """Helpers for test classes that want to exercise the OpenStack APIs

    This goes all the way down to the http layer, which is really a little too
    low for most of what the test cases want to assert. With some cleanups to
    the client code, tests could instead mock out client methods.
    """

    environment_name = "testing"
    api_url = "https://test.invalid/v2.0/"

    def setUp(self):
        self._mock_request = self.mocker.replace(
            "juju.providers.openstack.client.request",
            passthrough=False)
        self.mocker.order()
        self.nova_url = self.api_url + "nova"
        self.swift_url = self.api_url + "swift"
        self._mock_request("POST", self.api_url + "tokens", body=mocker.ANY,
            check_certs=True)
        self.mocker.result(defer.succeed((FakeResponse(200),
            json.dumps({"access": {
                "token": {"id": "tok", "expires": "2030-01-01T00:00:00"},
                "serviceCatalog": [
                    {
                        "type": "compute",
                        "endpoints": [{"publicURL": self.nova_url}]
                    },
                    {
                        "type": "object-store",
                        "endpoints": [{"publicURL": self.swift_url}]
                    }
                ]
            }}))))
        self.setup_constraints()
        # Clear the environment so provider won't pick up config from envvars
        self.change_environment()

    def get_config(self):
        return {
            "type": "openstack",
            "admin-secret": "password",
            "access-key": "90abcdef",
            "secret-key": "xxxxxxxx",
            "auth-mode": "keypair",
            "authorized-keys": "abc",
            "auth-url": self.api_url,
            "project-name": "test_project",
            "control-bucket": self.environment_name,
            "default-instance-type": "standard.xsmall",
            "default-image-id": 42,
            "use-floating-ip": True,
            "ssl-hostname-verification": True,
            }

    def get_provider(self):
        """Return the openstack machine provider.

        This should only be invoked after mocker is in replay mode.
        """
        return provider.MachineProvider(self.environment_name,
            self.get_config())

    def make_server(self, server_id, name=None, status="ACTIVE"):
        if name is None:
            # Would be machine id rather than server id really but will do.
            name = "juju testing instance " + str(server_id)
        return {
            "id": server_id,
            "name": name,
            "status": status,
            "addresses": {},
            }

    def assert_not_found(self, deferred, server_ids):
        self.assertFailure(deferred, errors.MachinesNotFound)
        return deferred.addCallback(self._check_not_found, server_ids)

    def _check_not_found(self, error, server_ids):
        self.assertEqual(error.instance_ids, server_ids)

    def _mock_nova(self, method, path, body):
        self._mock_request(method,
            "%s/%s" % (self.nova_url, path),
            [("X-Auth-Token", "tok")],
            body, True)

    def _mock_swift(self, method, path, body, extra_headers=None):
        headers = [("X-Auth-Token", "tok")]
        if extra_headers is not None:
            headers += extra_headers
        url = "%s/%s" % (self.swift_url, path)
        self._mock_request(method, url, headers, body, True)

    def expect_nova_get(self, path, code=200, response=""):
        self._mock_nova("GET", path, None)
        if not isinstance(response, str):
            response = json.dumps(response)
        self.mocker.result(defer.succeed((FakeResponse(code), response)))

    def expect_nova_post(self, path, body, code=200, response=""):
        self._mock_nova("POST", path, body)
        if not isinstance(response, str):
            response = json.dumps(response)
        self.mocker.result(defer.succeed((FakeResponse(code), response)))

    def expect_nova_delete(self, path, code=202, response=""):
        self._mock_nova("DELETE", path, None)
        self.mocker.result(defer.succeed((FakeResponse(code), response)))

    def expect_swift_get(self, path, code=200, response=""):
        self._mock_swift("GET", path, None)
        self.mocker.result(defer.succeed((FakeResponse(code), response)))

    def expect_swift_put(self, path, body, code=201, response=""):
        self._mock_swift("PUT", path, body)
        self.mocker.result(defer.succeed((FakeResponse(code), response)))

    def expect_swift_put_container(self, container, code=201, response=""):
        self._mock_swift("PUT", container, "", [('X-Container-Read', '.r:*')])
        self.mocker.result(defer.succeed((FakeResponse(code), response)))


class MockedProvider(ConstraintSupportMixin):

    provider_type = "openstack"
    environment_name = "testing"
    api_url = "https://testing.invalid/2.0/"

    default_config = {
        "type": "openstack",
        "admin-secret": "asecret",
        "authorized-keys": "abc",
        "username": "auser",
        "password": "xxxxxxxx",
        "auth-url": api_url,
        "project-name": "aproject",
        "control-bucket": environment_name,
        "default-instance-type": "standard.xsmall",
        "default-image-id": 42,
        "ssl-hostname-verification": True,
        }

    def __init__(self, mocker, config=None):
        if config is None:
            config = dict(self.default_config)
        self.config = config
        self._check_certs = config.get("ssl-hostname-verification", True)
        self.nova = mocker.proxy(client._NovaClient(None), passthrough=False)
        self.swift = mocker.proxy(client._SwiftClient(None), passthrough=False)
        self.port_manager = mocker.proxy(
            ports.NovaPortManager(self.nova, self.environment_name))
        self.provider_actions = mocker.mock()
        self.mocker = mocker
        self.setup_constraints()

    def get_file_storage(self):
        return files.FileStorage(self.swift, self.config['control-bucket'])

    def __getattr__(self, attr):
        return getattr(self.provider_actions, attr)

    def expect_swift_public_object_url(self, object_name):
        container_name = self.config['control-bucket']
        event = self.swift.public_object_url(container_name, object_name)
        self.mocker.result("%sswift/%s" % (self.api_url, object_name))
        return event

    def expect_swift_put(self, object_name, body, code=201, response=""):
        container_name = self.config['control-bucket']
        event = self.swift.put_object(container_name, object_name, body)
        self.mocker.result(defer.succeed((FakeResponse(code), response)))
        return event

    def expect_zookeeper_machines(self, iid, hostname="undefined.invalid"):
        event = self.provider_actions.get_zookeeper_machines()
        self.mocker.result(defer.succeed([
            machine.NovaProviderMachine(iid, hostname, hostname)
            ]))
        return event
