"""Tests for emulating port management with security groups"""

import logging

from juju import errors
from juju.lib.testing import TestCase
from juju.providers.openstack.machine import NovaProviderMachine
from juju.providers.openstack.ports import NovaPortManager
from juju.providers.openstack.tests import OpenStackTestMixin


class ProviderPortMgmtTests(OpenStackTestMixin, TestCase):
    """Tests for provider exposed port management methods"""

    def expect_create_rule(self, group_id, proto, port):
        self.expect_nova_post("os-security-group-rules",
            {'security_group_rule': {
                'parent_group_id': group_id,
                'ip_protocol': proto,
                'from_port': port,
                'to_port': port,
            }},
            response={'security_group_rule': {
                'id': 144, 'parent_group_id': group_id,
            }})

    def expect_existing_rule(self, rule_id, proto, port):
        self.expect_nova_get("servers/1000/os-security-groups",
            response={'security_groups': [
                    {'name': "juju-testing-1", 'id': 1, 'rules': [{
                        'id': rule_id,
                        'parent_group_id': 1,
                        'ip_protocol': proto,
                        'from_port': port,
                        'to_port': port,
                        }]
                    },
                ]})

    def test_open_port(self):
        """Opening a port adds the rule to the appropriate security group"""
        self.expect_nova_get("servers/1000/os-security-groups",
            response={'security_groups': [
                    {'name': "juju-testing-1", 'id': 1},
                ]})
        self.expect_create_rule(1, "tcp", 80)
        self.mocker.replay()

        log = self.capture_logging("juju.openstack", level=logging.DEBUG)
        machine = NovaProviderMachine('1000', "server1000.testing.invalid")
        deferred = self.get_provider().open_port(machine, "1", 80)

        def _check_log(_):
            self.assertIn("Opened 80/tcp on machine '1000'",
                log.getvalue())
        return deferred.addCallback(_check_log)

    def test_open_port_missing_group(self):
        """Missing security group raises an error on deleting port"""
        self.expect_nova_get("servers/1000/os-security-groups",
            response={'security_groups': []})
        self.mocker.replay()

        machine = NovaProviderMachine('1000', "server1000.testing.invalid")
        deferred = self.get_provider().open_port(machine, "1", 80)
        return self.assertFailure(deferred, errors.ProviderInteractionError)

    def test_close_port(self):
        """Closing a port removes the matching rule from the security group"""
        self.expect_existing_rule(12, "tcp", 80)
        self.expect_nova_delete("os-security-group-rules/12")
        self.mocker.replay()

        log = self.capture_logging("juju.openstack", level=logging.DEBUG)
        machine = NovaProviderMachine('1000', "server1000.testing.invalid")
        deferred = self.get_provider().close_port(machine, "1", 80)
        def _check_log(_):
            self.assertIn("Closed 80/tcp on machine '1000'",
                log.getvalue())
        return deferred.addCallback(_check_log)

    def test_close_port_missing_group(self):
        """Missing security group raises an error on closing port"""
        self.expect_nova_get("servers/1000/os-security-groups",
            response={'security_groups': []})
        self.mocker.replay()

        machine = NovaProviderMachine('1000', "server1000.testing.invalid")
        deferred = self.get_provider().close_port(machine, "1", 80)
        return self.assertFailure(deferred, errors.ProviderInteractionError)

    def test_close_port_missing_rule(self):
        """Missing security group rule raises an error on closing port"""
        self.expect_nova_get("servers/1000/os-security-groups",
            response={'security_groups': [{
                'name': "juju-testing-1", 'id': 1, "rules": [],
            }]})
        self.mocker.replay()

        machine = NovaProviderMachine('1000', "server1000.testing.invalid")
        deferred = self.get_provider().close_port(machine, "1", 80)
        return self.assertFailure(deferred, errors.ProviderInteractionError)

    def test_close_port_mismatching_rule(self):
        """Rule with different port raises an error on closing port"""
        self.expect_existing_rule(12, "tcp", 8080)
        self.mocker.replay()

        machine = NovaProviderMachine('1000', "server1000.testing.invalid")
        deferred = self.get_provider().close_port(machine, "1", 80)
        return self.assertFailure(deferred, errors.ProviderInteractionError)

    def test_get_opened_ports_none(self):
        """No opened ports are listed when there are no rules"""
        self.expect_nova_get("servers/1000/os-security-groups",
            response={'security_groups': [{
                'name': "juju-testing-1", 'id': 1, "rules": [],
            }]})
        self.mocker.replay()

        machine = NovaProviderMachine('1000', "server1000.testing.invalid")
        deferred = self.get_provider().get_opened_ports(machine, "1")
        return deferred.addCallback(self.assertEqual, set())

    def test_get_opened_ports_one(self):
        """Opened port is listed when there is a matching rule"""
        self.expect_existing_rule(12, "tcp", 80)
        self.mocker.replay()

        machine = NovaProviderMachine('1000', "server1000.testing.invalid")
        deferred = self.get_provider().get_opened_ports(machine, "1")
        return deferred.addCallback(self.assertEqual, set([(80, "tcp")]))

    def test_get_opened_ports_group_ignored(self):
        """Opened ports exclude rules delegating to other security groups"""
        self.expect_nova_get("servers/1000/os-security-groups",
            response={'security_groups': [{
                'name': "juju-testing-1", 'id': 1, "rules": [{
                        'id': 12,
                        'parent_group_id': 1,
                        'ip_protocol': None,
                        'from_port': None,
                        'to_port': None,
                        'group': {'name': "juju-testing"},
                        }],
            }]})
        self.mocker.replay()

        machine = NovaProviderMachine('1000', "server1000.testing.invalid")
        deferred = self.get_provider().get_opened_ports(machine, "1")
        return deferred.addCallback(self.assertEqual, set())

    def test_get_opened_ports_multiport_ignored(self):
        """Opened ports exclude rules spanning multiple ports"""
        self.expect_nova_get("servers/1000/os-security-groups",
            response={'security_groups': [{
                'name': "juju-testing-1", 'id': 1, "rules": [{
                        'id': 12,
                        'parent_group_id': 1,
                        'ip_protocol': "tcp",
                        'from_port': 8080,
                        'to_port': 8081,
                        }],
            }]})
        self.mocker.replay()

        machine = NovaProviderMachine('1000', "server1000.testing.invalid")
        deferred = self.get_provider().get_opened_ports(machine, "1")
        return deferred.addCallback(self.assertEqual, set())


class PortManagerTestMixin(OpenStackTestMixin):

    def get_port_manager(self):
        provider = self.get_provider()
        return NovaPortManager(provider.nova, provider.environment_name)


class EnsureGroupsTests(PortManagerTestMixin, TestCase):
    """Tests for ensure_groups method used when launching machines"""

    def expect_create_juju_group(self):
        self.expect_nova_post("os-security-groups",
            {'security_group': {
                'name': 'juju-testing',
                'description': 'juju group for testing',
            }},
            response={'security_group': {
                'id': 1,
            }})
        self.expect_nova_post("os-security-group-rules",
            {'security_group_rule': {
                'parent_group_id': 1,
                'ip_protocol': "tcp",
                'from_port': 22,
                'to_port': 22,
            }},
            response={'security_group_rule': {
                'id': 144, 'parent_group_id': 1,
            }})
        self.expect_nova_post("os-security-group-rules",
            {'security_group_rule': {
                'parent_group_id': 1,
                'group_id': 1,
                'ip_protocol': "tcp",
                'from_port': 1,
                'to_port': 65535,
            }},
            response={'security_group_rule': {
                'id': 145, 'parent_group_id': 1,
            }})
        self.expect_nova_post("os-security-group-rules",
            {'security_group_rule': {
                'parent_group_id': 1,
                'group_id': 1,
                'ip_protocol': "udp",
                'from_port': 1,
                'to_port': 65535,
            }},
            response={'security_group_rule': {
                'id': 146, 'parent_group_id': 1,
            }})

    def expect_create_machine_group(self, machine_id):
        machine = str(machine_id)
        self.expect_nova_post("os-security-groups",
            {'security_group': {
                'name': 'juju-testing-' + machine,
                'description': 'juju group for testing machine ' + machine,
            }},
            response={'security_group': {
                'id': 2,
            }})

    def check_group_names(self, result, machine_id):
        self.assertEqual(["juju-testing", "juju-testing-" + str(machine_id)],
            result)

    def test_none_existing(self):
        """When no groups exist juju and machine security groups are created"""
        self.expect_nova_get("os-security-groups",
            response={'security_groups': []})
        self.expect_create_juju_group()
        self.expect_create_machine_group(0)
        self.mocker.replay()
        deferred = self.get_port_manager().ensure_groups(0)
        return deferred.addCallback(self.check_group_names, 0)

    def test_other_existing(self):
        """Existing groups in a different environment are not affected"""
        self.expect_nova_get("os-security-groups",
            response={'security_groups': [
                    {'name': "juju-testingish", 'id': 7},
                    {'name': "juju-testingish-0", 'id': 8},
                ]})
        self.expect_create_juju_group()
        self.expect_create_machine_group(0)
        self.mocker.replay()
        deferred = self.get_port_manager().ensure_groups(0)
        return deferred.addCallback(self.check_group_names, 0)

    def test_existing_juju_group(self):
        """An exisiting juju security group is reused"""
        self.expect_nova_get("os-security-groups",
            response={'security_groups': [
                    {'name': "juju-testing", 'id': 1},
                ]})
        self.expect_create_machine_group(0)
        self.mocker.replay()
        deferred = self.get_port_manager().ensure_groups(0)
        return deferred.addCallback(self.check_group_names, 0)

    def test_existing_machine_group(self):
        """An existing machine security group is deleted and remade"""
        self.expect_nova_get("os-security-groups",
            response={'security_groups': [
                    {'name': "juju-testing-6", 'id': 3},
                ]})
        self.expect_create_juju_group()
        self.expect_nova_delete("os-security-groups/3")
        self.expect_create_machine_group(6)
        self.mocker.replay()
        deferred = self.get_port_manager().ensure_groups(6)
        return deferred.addCallback(self.check_group_names, 6)


class GetMachineGroupsTests(PortManagerTestMixin, TestCase):
    """Tests for get_machine_groups method needed for machine shutdown"""

    def test_normal(self):
        """A standard juju machine returns the machine group name and id"""
        self.expect_nova_get("servers/1000/os-security-groups",
            response={'security_groups': [
                    {'id': 7, 'name': "juju-testing"},
                    {'id': 8, 'name': "juju-testing-0"},
                ]})
        self.mocker.replay()
        machine = NovaProviderMachine(1000)
        deferred = self.get_port_manager().get_machine_groups(machine)
        return deferred.addCallback(self.assertEqual, {"juju-testing-0": 8})

    def test_normal_include_juju(self):
        """If param with_juju_group=True the juju group is also returned"""
        self.expect_nova_get("servers/1000/os-security-groups",
            response={'security_groups': [
                    {'id': 7, 'name': "juju-testing"},
                    {'id': 8, 'name': "juju-testing-0"},
                ]})
        self.mocker.replay()
        machine = NovaProviderMachine(1000)
        deferred = self.get_port_manager().get_machine_groups(machine, True)
        return deferred.addCallback(self.assertEqual,
            {"juju-testing": 7, "juju-testing-0": 8})

    def test_extra_group(self):
        """Additional groups not in the juju namespace are ignored"""
        self.expect_nova_get("servers/1000/os-security-groups",
            response={'security_groups': [
                    {'id': 1, 'name': "default"},
                    {'id': 7, 'name': "juju-testing"},
                    {'id': 8, 'name': "juju-testing-0"},
                ]})
        self.mocker.replay()
        machine = NovaProviderMachine(1000)
        deferred = self.get_port_manager().get_machine_groups(machine)
        return deferred.addCallback(self.assertEqual, {"juju-testing-0": 8})

    def test_other_group(self):
        """A server not managed by juju returns nothing"""
        self.expect_nova_get("servers/1000/os-security-groups",
            response={'security_groups': [
                    {'id': 1, 'name': "default"},
                ]})
        self.mocker.replay()
        machine = NovaProviderMachine(1000)
        deferred = self.get_port_manager().get_machine_groups(machine)
        return deferred.addCallback(self.assertEqual, None)

    def test_missing_groups(self):
        """A server with no groups returns nothing"""
        self.expect_nova_get("servers/1000/os-security-groups",
            response={'security_groups': []})
        self.mocker.replay()
        machine = NovaProviderMachine(1000)
        deferred = self.get_port_manager().get_machine_groups(machine)
        return deferred.addCallback(self.assertEqual, None)

    def test_error_missing_server(self):
        """A server that doesn't exist or has been deleted returns nothing"""
        self.expect_nova_get("servers/1000/os-security-groups",
            code=404, response={"itemNotFound": {
                "message": "Instance 1000 could not be found.",
                "code": 404,
            }})
        self.mocker.replay()
        machine = NovaProviderMachine(1000)
        deferred = self.get_port_manager().get_machine_groups(machine)
        return deferred.addCallback(self.assertEqual, None)
    # XXX: Broken by workaround for HP not supporting this api
    test_error_missing_server.skip = True

    def test_error_missing_page(self):
        """Unexpected errors from the client are propogated"""
        self.expect_nova_get("servers/1000/os-security-groups",
            code=404, response="404 Not Found\n\n"
                "The resource could not be found.\n\n   ")
        self.mocker.replay()
        machine = NovaProviderMachine(1000)
        deferred = self.get_port_manager().get_machine_groups(machine)
        return self.assertFailure(deferred, errors.ProviderInteractionError)
    # XXX: Need implemention of fancy error to exception mapping
    test_error_missing_page.skip = True

    def test_error_missing_server_fault(self):
        "A bogus compute fault due to lp:1010486 returns nothing"""
        self.expect_nova_get("servers/1000/os-security-groups",
            code=500, response={"computeFault": {
                "message": "The server has either erred or is incapable of"
                    " performing the requested operation.",
                "code": 500,
            }})
        self.expect_nova_get("servers/1000",
            code=404, response={"itemNotFound": {
                "message": "The resource could not be found.",
                "code": 404,
            }})
        self.mocker.replay()
        machine = NovaProviderMachine(1000)
        deferred = self.get_port_manager().get_machine_groups(machine)
        return deferred.addCallback(self.assertEqual, None)
    # XXX: Need implemention of fancy error to exception mapping
    test_error_missing_server_fault.skip = True

    def test_error_really_fault(self):
        """A real compute fault is propogated"""
        self.expect_nova_get("servers/1000/os-security-groups",
            code=500, response={"computeFault": {
                "message": "The server has either erred or is incapable of"
                    " performing the requested operation.",
                "code": 500,
            }})
        self.expect_nova_get("servers/1000", response={"server": {"id": 1000}})
        self.mocker.replay()
        machine = NovaProviderMachine(1000)
        deferred = self.get_port_manager().get_machine_groups(machine)
        return self.assertFailure(deferred, errors.ProviderInteractionError)
    # XXX: Need implemention of fancy error to exception mapping
    test_error_really_fault.skip = True
