// K-3D
// Copyright (c) 1995-2004, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\brief Implements functions and classes for simplifying interactive 3D manipulator implementations
		\author Tim Shead (tshead@k-3d.com)
*/

#include "algebra.h"
#include "basic_math.h"
#include "manipulators.h"
#include "transform.h"

namespace k3d
{

const vector2 transform_mouse_coordinates(k3dICamera& Camera, const vector2 MouseCoords, double Left, double Right, double Top, double Bottom)
{
	double width = Camera.Width();
	double height = Camera.Height();

	const double mousex = Clamp<double>(0, width, MouseCoords[0]);
	const double mousey = Clamp<double>(0, height, MouseCoords[1]);

	if(0 == width)
		width = 1;

	if(0 == height)
		height = 1;

	return vector2(Lerp(Left, Right, mousex / width), Lerp(Top, Bottom, mousey / height));
}

const vector2 transform_mouse_coordinates(k3dICamera& Camera, const vector2 MouseCoords, double Left, double Right, double Top, double Bottom, double scale)
{
	double width = Camera.Width();
	double height = Camera.Height();

	// Translate the window inside a bigger one (TODO : add "scale < 1" support)
	const double newx = MouseCoords[0] + width / scale;
	const double newy = MouseCoords[1] + height / scale;
	width *= scale;
	height *= scale;

	if(0 == width)
		width = 1;

	if(0 == height)
		height = 1;

	double mousex = Clamp<double>(0, width, newx);
	double mousey = Clamp<double>(0, height, newy);

	return vector2(Lerp(Left, Right, mousex / width), Lerp(Top, Bottom, mousey / height));
}

void mouse_to_camera_ray(k3dICamera& Camera, const vector2 MouseCoords, vector3& RayDirection)
{
	// Get the camera frustum ...
//	const bool orthographic = Camera.ViewVolume()->Orthographic();
	const double nearplane = Camera.ViewVolume()->NearPlane();
//	const double farplane = Camera.ViewVolume()->FarPlane();
	const double left = Camera.ViewVolume()->Left();
	const double right = Camera.ViewVolume()->Right();
	const double top = Camera.ViewVolume()->Top();
	const double bottom = Camera.ViewVolume()->Bottom();

	// Convert mouse coordinates (pixels) to screen coordinates ...
	const vector2 screencoords = transform_mouse_coordinates(Camera, MouseCoords, left * nearplane, right * nearplane, top * nearplane, bottom * nearplane);

	RayDirection = vector3(screencoords, nearplane);
	RayDirection.Normalize();
}

void mouse_to_world_ray(k3dICamera& Camera, const vector2 MouseCoords, vector3& RayOrigin, vector3& RayDirection)
{
	// Get the mouse ray in camera coordinates ...
	vector3 ray;
	mouse_to_camera_ray(Camera, MouseCoords, ray);

	// Transform it to the world frame ...
	const matrix4 cameratoworld = object_to_world_matrix(Camera);
	
	RayOrigin = cameratoworld * vector3(0, 0, 0);
	RayDirection = (cameratoworld * ray) - RayOrigin;
}

} // namespace k3d

