// K-3D
// Copyright (c) 1995-2004, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\author Andy Gill (buzz@ucky.com)
		\author Tim Shead (tshead@k-3d.com)
		\author Romain Behar (romainbehar@yahoo.com)
*/

#include <k3dsdk/axis.h>
#include <k3dsdk/bounding_box.h>
#include <k3dsdk/imaterial.h>
#include <k3dsdk/object.h>
#include <k3dsdk/persistence.h>
#include <k3dsdk/measurement.h>
#include <k3dsdk/mesh_filter.h>
#include <k3dsdk/module.h>
#include <k3dsdk/mouse_event_observer.h>
#include <k3dsdk/plugins.h>
#include <k3dsdk/selection.h>
#include <k3dsdk/transform.h>

#include <iterator>

namespace libk3ddeformation
{

namespace detail
{

class bender
{
public:
	bender(const k3d::vector3& Origin, const k3d::vector3& Min, const k3d::vector3& Max, const k3d::axis Along, const k3d::axis Around, const double BendAngle, const double Tightness, const double JointPosition) :
		m_Origin(Origin),
		m_Min(Min),
		m_Max(Max),
		m_Along(Along),
		m_Around(Around),
		m_BendAngle(BendAngle),
		m_Tightness(Tightness),
		m_JointPosition(JointPosition),
		m_Size(Max - Min),
		m_Joint(k3d::mix(Min, Max, JointPosition)),
		m_Radius(1.0 - Tightness),
		m_JLower(k3d::mix(Min, Max, JointPosition - m_Radius))
	{
	}

	const k3d::vector3 bend(k3d::vector3 coords)
	{
		if(m_Along == m_Around || m_BendAngle == 0)
			return coords;

		double delta = 0;
		double length = 0;
		k3d::vector3 axis;
		k3d::vector3 offset;
		k3d::vector3 pivot = m_Origin;

		switch(m_Along)
			{
				case k3d::X :
					if(m_Size[0] != 0)
						delta = (coords[0] - m_Min[0]) / m_Size[0];

					length = m_Size[0];

					if(delta > m_JointPosition - m_Radius && delta < m_JointPosition + m_Radius && m_BendAngle != 0)
						{
							coords[0] = m_JLower[0];
							pivot[0] = m_JLower[0];
						}
					else
						m_Origin[0] = m_Joint[0];

					switch(m_Around)
						{
							case k3d::X :
								break;
							case k3d::Y :
								axis = k3d::vector3(0, 1, 0);
								offset = k3d::vector3(0, 0, -1);
								break;
							case k3d::Z:
								axis = k3d::vector3(0, 0, 1);
								offset = k3d::vector3(0, 1, 0);
								break;
						}

					break;

				case k3d::Y :
					if(m_Size[1])
						delta = (coords[1] - m_Min[1]) / m_Size[1];

					length = m_Size[1];

					if(delta > m_JointPosition - m_Radius && delta < m_JointPosition + m_Radius && m_BendAngle != 0)
					{
						coords[1] = m_JLower[1];
						pivot[1] = m_JLower[1];
					}
					else
						m_Origin[1] = m_Joint[1];

					switch(m_Around)
						{
							case k3d::X :
								axis = k3d::vector3(1, 0, 0);
								offset = k3d::vector3(0, 0, 1);
								break;
							case k3d::Y :
								break;
							case k3d::Z :
								axis = k3d::vector3(0, 0, 1);
								offset = k3d::vector3(-1, 0, 0);
								break;
						}
					break;
				case k3d::Z:
					if(m_Size[2])
						delta = (coords[2] - m_Min[2]) / m_Size[2];

					length = m_Size[2];

					if(delta > m_JointPosition - m_Radius && delta < m_JointPosition + m_Radius && m_BendAngle != 0)
					{
						coords[2] = m_JLower[2];
						pivot[2] = m_JLower[2];
					}
					else
						m_Origin[2] = m_Joint[2];

					switch(m_Around)
						{
							case k3d::X :
								axis = k3d::vector3(1, 0, 0);
								offset = k3d::vector3(0, -1, 0);
								break;
							case k3d::Y :
								axis = k3d::vector3(0, 1, 0);
								offset = k3d::vector3(1, 0, 0);
								break;
							case k3d::Z :
								break;
						}
					break;
			}

		if(delta > m_JointPosition - m_Radius && delta < m_JointPosition + m_Radius && m_BendAngle != 0)
			{
				double d = (delta - (m_JointPosition - m_Radius)) / (m_Radius * 2.0);
				double factor = m_Radius * length / tan(m_BendAngle * 0.5);
				k3d::matrix4 rotation = k3d::rotation3D(d * m_BendAngle, axis);
				offset = offset * factor;
				coords -= pivot + offset;
				coords = rotation * coords;
				coords += pivot + offset;
			}
		else if(delta >= m_JointPosition + m_Radius)
			{
				k3d::matrix4 rotation = k3d::rotation3D(m_BendAngle, axis);
				coords -= m_Origin;
				coords = rotation * coords;
				coords += m_Origin;
			}

		return coords;
	}

private:
	k3d::vector3 m_Origin;
	const k3d::vector3 m_Min;
	const k3d::vector3 m_Max;
	const k3d::axis m_Along;
	const k3d::axis m_Around;
	const double m_BendAngle;
	const double m_Tightness;
	const double m_JointPosition;
	const k3d::vector3 m_Size;
	const k3d::vector3 m_Joint;
	const double m_Radius;
	const k3d::vector3 m_JLower;
};

} // namespace detail

/////////////////////////////////////////////////////////////////////////////
// bend_points_implementation

class bend_points_implementation :
	public k3d::mesh_filter<k3d::persistent<k3d::object> >,
	public k3d::mouse_event_observer
{
	typedef k3d::mesh_filter<k3d::persistent<k3d::object> > base;

public:
	bend_points_implementation(k3d::idocument& Document) :
		base(Document),
		k3d::mouse_event_observer("LMB drag to bend points"),
		m_angle(k3d::init_name("angle") + k3d::init_description("Bend Angle [number]") + k3d::init_value(0.0) + k3d::init_document(Document) + k3d::init_precision(2) + k3d::init_step_increment(k3d::radians(1.0)) + k3d::init_units(typeid(k3d::measurement::angle))),
		m_tightness(k3d::init_name("tightness") + k3d::init_description("Bend Tightness [number]") + k3d::init_value(0.6) + k3d::init_constraint(k3d::constraint::minimum(0.0, k3d::constraint::maximum(1.0))) + k3d::init_document(Document) + k3d::init_precision(2) + k3d::init_step_increment(0.01) + k3d::init_units(typeid(k3d::measurement::scalar))),
		m_position(k3d::init_name("position") + k3d::init_description("Bend position [number]") + k3d::init_value(0.5) + k3d::init_constraint(k3d::constraint::minimum(0.0, k3d::constraint::maximum(1.0))) + k3d::init_document(Document) + k3d::init_precision(2) + k3d::init_step_increment(0.01) + k3d::init_units(typeid(k3d::measurement::scalar))),
		m_along(k3d::init_name("along") + k3d::init_description("Axis to bend along [enumeration]") + k3d::init_value(k3d::Y) + k3d::init_document(Document) + k3d::init_enumeration(k3d::axis_values())),
		m_around(k3d::init_name("around") + k3d::init_description("Axis to bend around [enumeration]") + k3d::init_value(k3d::X) + k3d::init_document(Document) + k3d::init_enumeration(k3d::axis_values()))
	{
		enable_serialization(k3d::persistence::proxy(m_angle));
		enable_serialization(k3d::persistence::proxy(m_tightness));
		enable_serialization(k3d::persistence::proxy(m_position));
		enable_serialization(k3d::persistence::proxy(m_along));
		enable_serialization(k3d::persistence::proxy(m_around));

		register_property(m_angle);
		register_property(m_tightness);
		register_property(m_position);
		register_property(m_along);
		register_property(m_around);

		m_input_mesh.changed_signal().connect(SigC::slot(*this, &bend_points_implementation::on_reset_geometry));

		m_angle.changed_signal().connect(SigC::slot(*this, &bend_points_implementation::on_reshape_geometry));
		m_tightness.changed_signal().connect(SigC::slot(*this, &bend_points_implementation::on_reshape_geometry));
		m_position.changed_signal().connect(SigC::slot(*this, &bend_points_implementation::on_reshape_geometry));
		m_along.changed_signal().connect(SigC::slot(*this, &bend_points_implementation::on_reshape_geometry));
		m_around.changed_signal().connect(SigC::slot(*this, &bend_points_implementation::on_reshape_geometry));

		m_output_mesh.need_data_signal().connect(SigC::slot(*this, &bend_points_implementation::on_create_geometry));
	}

	bool OnLButtonDrag(const k3d::imouse_event_observer::event_state& State, const k3d::vector2& Current, const k3d::vector2& Last, const k3d::vector2& Start, const drag_type_t DragType)
	{
		k3d::vector2 currentmouse(Current[0] - 0.5, 0.5 - Current[1]);
		k3d::vector2 lastmouse(Last[0] - 0.5, 0.5 - Last[1]);

		const double delta_theta = currentmouse.Angle() - lastmouse.Angle();
		m_angle.set_value(m_angle.value() + delta_theta);

		return true;
	}

	void on_reset_geometry()
	{
		m_output_mesh.reset();
	}

	void on_reshape_geometry()
	{
		if(m_output_mesh.empty())
			return;

		const k3d::mesh* const input = m_input_mesh.property_value();
		if(!input)
			return;

		k3d::mesh* const output = const_cast<k3d::mesh*>(m_output_mesh.value());
		if(!output)
			return;

		reshape_geometry(*input, *output);

		m_output_mesh.changed_signal().emit();
	}

	k3d::mesh* on_create_geometry()
	{
		// If we don't have any input mesh, we're done ...
		const k3d::mesh* const input = m_input_mesh.property_value();
		if(!input)
			return 0;

		// Otherwise, we make a copy of the input mesh and modify the copy ...
		k3d::mesh* const output = new k3d::mesh();
		k3d::deep_copy(*input, *output);
		reshape_geometry(*input, *output);

		return output;
	}

	void reshape_geometry(const k3d::mesh& Source, k3d::mesh& Target)
	{
		return_if_fail(Source.points.size() == Target.points.size());

		const bool ignore_selection = !k3d::contains_selection(Source);
		const k3d::bounding_box bounds = k3d::bounds(Source);

		const double angle = m_angle.property_value();
		const double tightness = m_tightness.property_value();
		const double position = m_position.property_value();
		const k3d::axis along = m_along.property_value();
		const k3d::axis around = m_around.property_value();

		detail::bender bender(k3d::vector3(0, 0, 0), k3d::vector3(bounds.nx, bounds.ny, bounds.nz), k3d::vector3(bounds.px, bounds.py, bounds.pz), along, around, angle, tightness, position);

		for(unsigned long i = 0; i != Target.points.size(); ++i)
			{
				if(ignore_selection || Source.points[i]->selected)
					{
						Target.points[i]->position = bender.bend(Source.points[i]->position);
					}
				else
					{
						Target.points[i]->position = Source.points[i]->position;
					}
			}
	}

	k3d::iplugin_factory& factory()
	{
		return get_factory();
	}

	static k3d::iplugin_factory& get_factory()
	{
		static k3d::plugin_factory<k3d::document_plugin<bend_points_implementation>,
			k3d::interface_list<k3d::imesh_source,
			k3d::interface_list<k3d::imesh_sink > > > factory(
				k3d::uuid(0xfd599b45, 0x20384cd3, 0xb0af634b, 0x9372888a),
				"BendPoints",
				"Bends mesh points around an axis",
				"Objects",
				k3d::iplugin_factory::STABLE);

		return factory;
	}

private:
	k3d_measurement_property(double, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::no_constraint) m_angle;
	k3d_measurement_property(double, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::with_constraint) m_tightness;
	k3d_measurement_property(double, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::with_constraint) m_position;
	k3d_enumeration_property(k3d::axis, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::no_constraint) m_along;
	k3d_enumeration_property(k3d::axis, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::no_constraint) m_around;
};

/////////////////////////////////////////////////////////////////////////////
// bend_points_factory

k3d::iplugin_factory& bend_points_factory()
{
	return bend_points_implementation::get_factory();
}

} // namespace libk3ddeformation


