// K-3D
// Copyright (c) 1995-2004, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\author Timothy M. Shead (tshead@k-3d.com)
		\author Romain Behar (romainbehar@yahoo.com)
*/

#include <k3dsdk/axis.h>
#include <k3dsdk/bounding_box.h>
#include <k3dsdk/imaterial.h>
#include <k3dsdk/object.h>
#include <k3dsdk/persistence.h>
#include <k3dsdk/measurement.h>
#include <k3dsdk/mesh_filter.h>
#include <k3dsdk/module.h>
#include <k3dsdk/mouse_event_observer.h>
#include <k3dsdk/plugins.h>
#include <k3dsdk/selection.h>
#include <k3dsdk/transform.h>

#include <iterator>

namespace libk3ddeformation
{

/////////////////////////////////////////////////////////////////////////////
// taper_points_implementation

class taper_points_implementation :
	public k3d::mesh_filter<k3d::persistent<k3d::object> >,
	public k3d::mouse_event_observer
{
	typedef k3d::mesh_filter<k3d::persistent<k3d::object> > base;

public:
	taper_points_implementation(k3d::idocument& Document) :
		base(Document),
		k3d::mouse_event_observer("LMB drag to taper points"),
		m_axis(k3d::init_name("axis") + k3d::init_description("Axis [enumeration]") + k3d::init_document(Document) + k3d::init_value(k3d::Y) + k3d::init_enumeration(k3d::axis_values())),
		m_taper_factor(k3d::init_name("taper_fator") + k3d::init_description("Taper Factor [number]") + k3d::init_document(Document) + k3d::init_value(0.0) + k3d::init_precision(2) + k3d::init_step_increment(k3d::radians(1.0)) + k3d::init_units(typeid(k3d::measurement::scalar))),
		m_displace_x(k3d::init_name("displace_x") + k3d::init_description("Displace X [boolean]") + k3d::init_value(true) + k3d::init_document(Document)),
		m_displace_y(k3d::init_name("displace_y") + k3d::init_description("Displace Y [boolean]") + k3d::init_value(false) + k3d::init_document(Document)),
		m_displace_z(k3d::init_name("displace_z") + k3d::init_description("Displace Z [boolean]") + k3d::init_value(true) + k3d::init_document(Document))
	{
		enable_serialization(k3d::persistence::proxy(m_axis));
		enable_serialization(k3d::persistence::proxy(m_taper_factor));
		enable_serialization(k3d::persistence::proxy(m_displace_x));
		enable_serialization(k3d::persistence::proxy(m_displace_y));
		enable_serialization(k3d::persistence::proxy(m_displace_z));

		register_property(m_axis);
		register_property(m_taper_factor);
		register_property(m_displace_x);
		register_property(m_displace_y);
		register_property(m_displace_z);

		m_input_mesh.changed_signal().connect(SigC::slot(*this, &taper_points_implementation::on_reset_geometry));

		m_axis.changed_signal().connect(SigC::slot(*this, &taper_points_implementation::on_reshape_geometry));
		m_taper_factor.changed_signal().connect(SigC::slot(*this, &taper_points_implementation::on_reshape_geometry));
		m_displace_x.changed_signal().connect(SigC::slot(*this, &taper_points_implementation::on_reshape_geometry));
		m_displace_y.changed_signal().connect(SigC::slot(*this, &taper_points_implementation::on_reshape_geometry));
		m_displace_z.changed_signal().connect(SigC::slot(*this, &taper_points_implementation::on_reshape_geometry));

		m_output_mesh.need_data_signal().connect(SigC::slot(*this, &taper_points_implementation::on_create_geometry));
	}

	bool OnLButtonDrag(const k3d::imouse_event_observer::event_state& State, const k3d::vector2& Current, const k3d::vector2& Last, const k3d::vector2& Start, const drag_type_t DragType)
	{
		const double xfactor = (Current[0] - Last[0]);
		const double yfactor = (Current[1] - Last[1]);

		m_taper_factor.set_value(m_taper_factor.value() + (std::fabs(xfactor) >= std::fabs(yfactor) ? xfactor : yfactor));

		return true;
	}

	void on_reset_geometry()
	{
		m_output_mesh.reset();
	}

	void on_reshape_geometry()
	{
		if(m_output_mesh.empty())
			return;

		const k3d::mesh* const input = m_input_mesh.property_value();
		if(!input)
			return;

		k3d::mesh* const output = const_cast<k3d::mesh*>(m_output_mesh.value());
		if(!output)
			return;

		reshape_geometry(*input, *output);

		m_output_mesh.changed_signal().emit();
	}

	k3d::mesh* on_create_geometry()
	{
		// If we don't have any input mesh, we're done ...
		const k3d::mesh* const input = m_input_mesh.property_value();
		if(!input)
			return 0;

		// Otherwise, we make a copy of the input mesh and modify the copy ...
		k3d::mesh* const output = new k3d::mesh();
		k3d::deep_copy(*input, *output);
		reshape_geometry(*input, *output);

		return output;
	}

	void reshape_geometry(const k3d::mesh& Source, k3d::mesh& Target)
	{
		// Sanity checks ...
		assert(Source.points.size() == Target.points.size());

		const bool ignore_selection = !k3d::contains_selection(Source);
		const k3d::bounding_box bounds = k3d::bounds(Source);

		const k3d::axis axis = m_axis.property_value();
		const double taper_factor = m_taper_factor.property_value();
		const bool displace_x = m_displace_x.property_value();
		const bool displace_y = m_displace_y.property_value();
		const bool displace_z = m_displace_z.property_value();

		double size = 0.0;
		switch(axis)
			{
				case k3d::X:
					size = std::max(std::abs(bounds.px), std::abs(bounds.nx));
					break;
				case k3d::Y:
					size = std::max(std::abs(bounds.py), std::abs(bounds.ny));
					break;
				case k3d::Z:
					size = std::max(std::abs(bounds.pz), std::abs(bounds.nz));
					break;
				default:
					assert_not_reached();
			}
		if(0.0 == size)
			return;

		for(unsigned long i = 0; i != Target.points.size(); ++i)
			{
				if(ignore_selection || Source.points[i]->selected)
					{
						const double scale = k3d::mix(1.0, 1.0 - taper_factor, std::abs(Source.points[i]->position[axis] / size));

						k3d::vector3 position(Source.points[i]->position);
						if(displace_x)
							position[0] *= scale;
						if(displace_y)
							position[1] *= scale;
						if(displace_z)
							position[2] *= scale;

						Target.points[i]->position = position;
					}
				else
					{
						Target.points[i]->position = Source.points[i]->position;
					}
			}
	}

	k3d::iplugin_factory& factory()
	{
		return get_factory();
	}

	static k3d::iplugin_factory& get_factory()
	{
		static k3d::plugin_factory<
			k3d::document_plugin<taper_points_implementation>,
			k3d::interface_list<k3d::imesh_source,
			k3d::interface_list<k3d::imesh_sink > > > factory(
				k3d::uuid(0x4d05f002, 0x27764b67, 0xa1a12e21, 0x436c3d06),
				"TaperPoints",
				"Tapers mesh points along an axis",
				"Objects",
				k3d::iplugin_factory::STABLE);

		return factory;
	}

private:
	k3d_enumeration_property(k3d::axis, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::no_constraint) m_axis;
	k3d_measurement_property(double, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::no_constraint) m_taper_factor;
	k3d_data_property(bool, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::no_constraint) m_displace_x;
	k3d_data_property(bool, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::no_constraint) m_displace_y;
	k3d_data_property(bool, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::no_constraint) m_displace_z;
};

/////////////////////////////////////////////////////////////////////////////
// taper_points_factory

k3d::iplugin_factory& taper_points_factory()
{
	return taper_points_implementation::get_factory();
}

} // namespace libk3ddeformation


