// K-3D
// Copyright (c) 1995-2004, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\author Tim Shead (tshead@k-3d.com)
*/

#include <k3dsdk/geometry.h>
#include <k3dsdk/imaterial.h>
#include <k3dsdk/object.h>
#include <k3dsdk/persistence.h>
#include <k3dsdk/material.h>
#include <k3dsdk/material_collection.h>
#include <k3dsdk/measurement.h>
#include <k3dsdk/mesh_source.h>
#include <k3dsdk/module.h>
#include <k3dsdk/plugins.h>
#include <k3dsdk/transform.h>

namespace libk3dnurbs
{

/////////////////////////////////////////////////////////////////////////////
// nurbs_torus_implementation

class nurbs_torus_implementation :
	public k3d::material_collection<k3d::mesh_source<k3d::persistent<k3d::object> > >
{
	typedef k3d::material_collection<k3d::mesh_source<k3d::persistent<k3d::object> > > base;

public:
	nurbs_torus_implementation(k3d::idocument& Document) :
		base(Document),
		m_majorradius(k3d::init_name("majorradius") + k3d::init_description("Major Radius [distance]") + k3d::init_value(5.0) + k3d::init_document(Document) + k3d::init_precision( 2) + k3d::init_step_increment(0.1) + k3d::init_units(typeid(k3d::measurement::distance))),
		m_minorradius(k3d::init_name("minorradius") + k3d::init_description("Minor Radius [distance]") + k3d::init_value(2.0) + k3d::init_document(Document) + k3d::init_precision( 2) + k3d::init_step_increment(0.1) + k3d::init_units(typeid(k3d::measurement::distance))),
		m_phimin(k3d::init_name("phimin") + k3d::init_description("Phi Minimum [angle]") + k3d::init_value(0.0) + k3d::init_document(Document) + k3d::init_precision( 2) + k3d::init_step_increment(k3d::radians(1.0)) + k3d::init_units(typeid(k3d::measurement::angle))),
		m_phimax(k3d::init_name("phimax") + k3d::init_description("Phi Maximum [angle]") + k3d::init_value(k3d::radians(360.0)) + k3d::init_document(Document) + k3d::init_precision( 2) + k3d::init_step_increment(k3d::radians(1.0)) + k3d::init_units(typeid(k3d::measurement::angle))),
		m_thetamax(k3d::init_name("thetamax") + k3d::init_description("Theta Maximum [angle]") + k3d::init_value(k3d::radians(360.0)) + k3d::init_document(Document) + k3d::init_precision( 2) + k3d::init_step_increment(k3d::radians(1.0)) + k3d::init_units(typeid(k3d::measurement::angle))),
		m_u_segments(k3d::init_name("u_segments") + k3d::init_description("U Segments [integer]") + k3d::init_value(4) + k3d::init_document(Document) + k3d::init_constraint(k3d::constraint::minimum(1UL)) + k3d::init_precision(0) + k3d::init_step_increment(1) + k3d::init_units(typeid(k3d::measurement::scalar))),
		m_v_segments(k3d::init_name("v_segments") + k3d::init_description("V Segments [integer]") + k3d::init_value(4) + k3d::init_document(Document) + k3d::init_constraint(k3d::constraint::minimum(1UL)) + k3d::init_precision(0) + k3d::init_step_increment(1) + k3d::init_units(typeid(k3d::measurement::scalar)))
	{
		enable_serialization(k3d::persistence::proxy(m_majorradius));
		enable_serialization(k3d::persistence::proxy(m_minorradius));
		enable_serialization(k3d::persistence::proxy(m_phimin));
		enable_serialization(k3d::persistence::proxy(m_phimax));
		enable_serialization(k3d::persistence::proxy(m_thetamax));
		enable_serialization(k3d::persistence::proxy(m_u_segments));
		enable_serialization(k3d::persistence::proxy(m_v_segments));

		register_property(m_majorradius);
		register_property(m_minorradius);
		register_property(m_phimin);
		register_property(m_phimax);
		register_property(m_thetamax);
		register_property(m_u_segments);
		register_property(m_v_segments);

		m_material.changed_signal().connect(SigC::slot(*this, &nurbs_torus_implementation::on_reset_geometry));
		m_majorradius.changed_signal().connect(SigC::slot(*this, &nurbs_torus_implementation::on_reset_geometry));
		m_minorradius.changed_signal().connect(SigC::slot(*this, &nurbs_torus_implementation::on_reset_geometry));
		m_phimin.changed_signal().connect(SigC::slot(*this, &nurbs_torus_implementation::on_reset_geometry));
		m_phimax.changed_signal().connect(SigC::slot(*this, &nurbs_torus_implementation::on_reset_geometry));
		m_thetamax.changed_signal().connect(SigC::slot(*this, &nurbs_torus_implementation::on_reset_geometry));
		m_u_segments.changed_signal().connect(SigC::slot(*this, &nurbs_torus_implementation::on_reset_geometry));
		m_v_segments.changed_signal().connect(SigC::slot(*this, &nurbs_torus_implementation::on_reset_geometry));

		m_output_mesh.need_data_signal().connect(SigC::slot(*this, &nurbs_torus_implementation::on_create_geometry));
	}

	void on_reset_geometry()
	{
		m_output_mesh.reset();
	}

	k3d::mesh* on_create_geometry()
	{
		const double majorradius = m_majorradius.property_value();
		const double minorradius = m_minorradius.property_value();
		const double phimin = m_phimin.property_value();
		const double phimax = m_phimax.property_value();
		const double thetamax = m_thetamax.property_value();
		const unsigned long u_segments = m_u_segments.property_value();
		const unsigned long v_segments = m_v_segments.property_value();

		k3d::mesh* const mesh = new k3d::mesh();
		k3d::nupatch* const nupatch = new k3d::nupatch();
		mesh->nupatches.push_back(nupatch);

		nupatch->material = m_material.interface();
		nupatch->u_order = 3;
		nupatch->v_order = 3;

		std::vector<double> v_weights;
		std::vector<k3d::vector3> v_arc_points;
		k3d::nurbs_arc(k3d::vector3(0, 1, 0), k3d::vector3(0, 0, 1), phimin, phimax, v_segments, nupatch->v_knots, v_weights, v_arc_points);

		std::vector<double> u_weights;
		std::vector<k3d::vector3> u_arc_points;
		k3d::nurbs_arc(k3d::vector3(1, 0, 0), k3d::vector3(0, 1, 0), 0, thetamax, u_segments, nupatch->u_knots, u_weights, u_arc_points);

		for(unsigned long v = 0; v != v_arc_points.size(); ++v)
			{
				const k3d::vector3 offset = minorradius * v_arc_points[v][2] * k3d::vector3(0, 0, 1);
				const double radius2 = majorradius + (minorradius * v_arc_points[v][1]);
				const double v_weight = v_weights[v];

				for(unsigned long u = 0; u != u_arc_points.size(); ++u)
					{
						mesh->points.push_back(new k3d::point(radius2 * u_arc_points[u] + offset));
						nupatch->control_points.push_back(k3d::nupatch::control_point(mesh->points.back(), v_weight * u_weights[u]));
					}
			}

		assert_warning(is_valid(*nupatch));
		return mesh;
	}

	k3d::iplugin_factory& factory()
	{
		return get_factory();
	}

	static k3d::iplugin_factory& get_factory()
	{
		static k3d::plugin_factory<k3d::document_plugin<nurbs_torus_implementation>, k3d::interface_list<k3d::imesh_source > > factory(
		k3d::uuid(0xc6cb880c, 0x4e4d4028, 0x8c77305f, 0x4b8f05a2),
			"NurbsTorus",
			"Generates a NURBS torus",
			"Objects",
			k3d::iplugin_factory::STABLE);

		return factory;
	}

private:
	k3d_measurement_property(double, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::no_constraint) m_majorradius;
	k3d_measurement_property(double, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::no_constraint) m_minorradius;
	k3d_measurement_property(double, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::no_constraint) m_phimin;
	k3d_measurement_property(double, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::no_constraint) m_phimax;
	k3d_measurement_property(double, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::no_constraint) m_thetamax;
	k3d_measurement_property(unsigned long, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::with_constraint) m_u_segments;
	k3d_measurement_property(unsigned long, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::with_constraint) m_v_segments;
};

/////////////////////////////////////////////////////////////////////////////
// nurbs_torus_factory

k3d::iplugin_factory& nurbs_torus_factory()
{
	return nurbs_torus_implementation::get_factory();
}

} // namespace libk3dnurbs


