// K-3D
// Copyright (c) 1995-2004, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\brief Implements the k3d::window_geometry_store and derived classes, which encapsulate persisting window geometry (position & size) to various storage types
		\author Timothy M. Shead (tshead@k-3d.com)
*/

#include "window_geometry.h"

#include <k3dsdk/application.h>
#include <k3dsdk/ioptions.h>
#include <k3dsdk/result.h>
#include <k3dsdk/string_modifiers.h>
#include <k3dsdk/vectors.h>
#include <k3dsdk/xml_utility.h>

#include <sdpxml/sdpxml.h>

using k3d::xml::safe_element;

namespace k3d
{

/////////////////////////////////////////////////////////////////////////////
// options_window_geometry_store

void options_window_geometry_store::set_window_geometry(const std::string& WindowName, const int Left, const int Top, const unsigned int Width, const unsigned int Height)
{
	// Sanity checks ...
	assert_warning(WindowName.size());

	// If the user has disabled restoring window size & position, we're done ...
	if(!application().options().restore_window_geometry())
		return;

	// We never store minimized dimensions ...
	if(!Width || !Height)
		return;

	application().options().set_window_geometry(WindowName, k3d::vector4(Left, Top, Width, Height));
}

bool options_window_geometry_store::get_window_geometry(const std::string& WindowName, int& Left, int& Top, unsigned int& Width, unsigned int& Height)
{
	// Sanity checks ...
	assert_warning(WindowName.size());

	// If the user has disabled restoring window size & position, we're done ...
	if(!k3d::application().options().restore_window_geometry())
		return false;

	const k3d::vector4 geometry = application().options().window_geometry(WindowName);

	const int left = static_cast<int>(geometry[0]);
	const int top = static_cast<int>(geometry[1]);
	const unsigned int width = static_cast<unsigned int>(geometry[2]);
	const unsigned int height = static_cast<unsigned int>(geometry[3]);

	if((0 == width) || (0 == height))
		return false;

	Left = left;
	Top = top;
	Width = width;
	Height = height;

	return true;
}

} // namespace k3d


