// K-3D
// Copyright (c) 1995-2004, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\author Anders Dahnielson (anders@dahnielson.com)
*/

#include <k3dsdk/auto_ptr.h>
#include <k3dsdk/bitmap_source.h>
#include <k3dsdk/frames.h>
#include <k3dsdk/file_filter.h>
#include <k3dsdk/ibitmap_read_format.h>
#include <k3dsdk/itime_sink.h>
#include <k3dsdk/measurement.h>
#include <k3dsdk/module.h>
#include <k3dsdk/object.h>
#include <k3dsdk/path_data.h>
#include <k3dsdk/persistence.h>
#include <k3dsdk/plugins.h>
#include <k3dsdk/property.h>
#include <k3dsdk/time_source.h>

#include <boost/filesystem/fstream.hpp>
#include <boost/filesystem/operations.hpp>
#include <boost/filesystem/path.hpp>

#include <fstream>
#include <iomanip>
#include <iterator>

namespace libk3dbitmap
{

/////////////////////////////////////////////////////////////////////////////
// bitmap_sequence_in_implementation

class bitmap_sequence_in_implementation :
	public k3d::bitmap_source<k3d::persistent<k3d::object> >,
	public k3d::itime_sink
{
	typedef k3d::bitmap_source<k3d::persistent<k3d::object> > base;

public:
	bitmap_sequence_in_implementation(k3d::idocument& Document) :
		base(Document),
		m_file(k3d::init_name("file") + k3d::init_description("File [string]") + k3d::init_document(Document) + k3d::init_value(boost::filesystem::path())),
		m_time(k3d::init_name("time") + k3d::init_description("Input time [double]") + k3d::init_value(0.0) + k3d::init_document(Document))
	{
		enable_serialization(k3d::persistence::proxy(m_file));
		enable_serialization(k3d::persistence::proxy(m_time));
		
		register_property(m_file);
		register_property(m_time);

		m_file.changed_signal().connect(SigC::slot(*this, &bitmap_sequence_in_implementation::on_reset_bitmap));
		m_time.changed_signal().connect(SigC::slot(*this, &bitmap_sequence_in_implementation::on_reset_bitmap));
		m_output.need_data_signal().connect(SigC::slot(*this, &bitmap_sequence_in_implementation::on_create_bitmap));
	}

	k3d::iproperty& time_sink_input()
	{
		return m_time;
	}

	void on_reset_bitmap()
	{
		m_output.reset();
	}

	k3d::bitmap* on_create_bitmap()
	{
		const boost::filesystem::path file = m_file.property_value();

		// Check that the document has animation capabilities ...
		k3d::iproperty* const frame_rate_property = k3d::get_frame_rate(document());
		if (frame_rate_property)
		{
			// Solve frame number and filename ...
			const double frame_rate = boost::any_cast<double>(k3d::get_property_value(document().dag(), *frame_rate_property));
			const long view_frame = static_cast<long>(k3d::round(frame_rate * m_time.property_value()));
			
			k3d::frames frames(file);
			boost::filesystem::path destination;
			frames.frame(view_frame, destination);

			// Read file ...
			if (!boost::filesystem::exists(destination))
				return 0;

			std::auto_ptr<k3d::bitmap> bitmap(new k3d::bitmap());
			k3d::auto_ptr<k3d::ibitmap_read_format> filter(k3d::auto_file_filter<k3d::ibitmap_read_format>(destination));

			return_val_if_fail(filter.get(), 0);
			return_val_if_fail(filter->read_file(destination, *bitmap), 0);

			return bitmap.release();
		}
		else
		{
			// Read file ...
			if (!boost::filesystem::exists(file))
				return 0;

			std::auto_ptr<k3d::bitmap> bitmap(new k3d::bitmap());
			k3d::auto_ptr<k3d::ibitmap_read_format> filter(k3d::auto_file_filter<k3d::ibitmap_read_format>(file));

			return_val_if_fail(filter.get(), 0);
			return_val_if_fail(filter->read_file(file, *bitmap), 0);

			return bitmap.release();
		}
	}

	k3d::iplugin_factory& factory()
	{
		return get_factory();
	}

	static k3d::iplugin_factory& get_factory()
	{
		static k3d::plugin_factory<
			k3d::document_plugin<bitmap_sequence_in_implementation>,
			k3d::interface_list<k3d::ibitmap_source,
			k3d::interface_list<k3d::itime_sink> > > factory(
				k3d::uuid(0x23ec2e00, 0x53064bed, 0xa5bf23d9, 0x464a8db4),
				"BitmapSequenceIn",
				"Load sequence of bitmaps from the filesystem",
				"Objects",
				k3d::iplugin_factory::STABLE);

		return factory;
	}

private:
	k3d_data_property(boost::filesystem::path, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::no_constraint) m_file;
	k3d_data_property(double, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::no_constraint) m_time;
};

/////////////////////////////////////////////////////////////////////////////
// bitmap_sequence_in_factory

k3d::iplugin_factory& bitmap_sequence_in_factory()
{
	return bitmap_sequence_in_implementation::get_factory();
}

} // namespace libk3dbitmap

