// K-3D
// Copyright (c) 1995-2004, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\author Timothy M. Shead (tshead@k-3d.com)
*/

#include <k3dsdk/axis.h>
#include <k3dsdk/imaterial.h>
#include <k3dsdk/object.h>
#include <k3dsdk/persistence.h>
#include <k3dsdk/measurement.h>
#include <k3dsdk/mesh_filter.h>
#include <k3dsdk/module.h>
#include <k3dsdk/mouse_event_observer.h>
#include <k3dsdk/plugins.h>
#include <k3dsdk/selection.h>
#include <k3dsdk/transform.h>

#include <iterator>

namespace libk3ddeformation
{

/////////////////////////////////////////////////////////////////////////////
// translate_points_implementation

class translate_points_implementation :
	public k3d::mesh_filter<k3d::persistent<k3d::object> >,
	public k3d::mouse_event_observer
{
	typedef k3d::mesh_filter<k3d::persistent<k3d::object> > base;

public:
	translate_points_implementation(k3d::idocument& Document) :
		base(Document),
		k3d::mouse_event_observer("LMB drag to translate points"),
		m_x(k3d::init_name("x") + k3d::init_description("X translate [number]") + k3d::init_document(Document) + k3d::init_value(0.0) + k3d::init_precision(2) + k3d::init_step_increment(0.1) + k3d::init_units(typeid(k3d::measurement::distance))),
		m_y(k3d::init_name("y") + k3d::init_description("Y translate [number]") + k3d::init_document(Document) + k3d::init_value(0.0) + k3d::init_precision(2) + k3d::init_step_increment(0.1) + k3d::init_units(typeid(k3d::measurement::distance))),
		m_z(k3d::init_name("z") + k3d::init_description("Z translate [number]") + k3d::init_document(Document) + k3d::init_value(0.0) + k3d::init_precision(2) + k3d::init_step_increment(0.1) + k3d::init_units(typeid(k3d::measurement::distance)))
	{
		enable_serialization(k3d::persistence::proxy(m_x));
		enable_serialization(k3d::persistence::proxy(m_y));
		enable_serialization(k3d::persistence::proxy(m_z));

		register_property(m_x);
		register_property(m_y);
		register_property(m_z);

		m_input_mesh.changed_signal().connect(SigC::slot(*this, &translate_points_implementation::on_reset_geometry));

		m_x.changed_signal().connect(SigC::slot(*this, &translate_points_implementation::on_reshape_geometry));
		m_y.changed_signal().connect(SigC::slot(*this, &translate_points_implementation::on_reshape_geometry));
		m_z.changed_signal().connect(SigC::slot(*this, &translate_points_implementation::on_reshape_geometry));

		m_output_mesh.need_data_signal().connect(SigC::slot(*this, &translate_points_implementation::on_create_geometry));
	}

	bool OnLButtonDrag(const k3d::imouse_event_observer::event_state& State, const k3d::vector2& Current, const k3d::vector2& Last, const k3d::vector2& Start, const drag_type_t DragType)
	{
		// Compute changes along each axis ...
		double dx = 0.0;
		double dy = 0.0;
		double dz = 0.0;

		// Some useful quantities ...
		const double mousex = (Current[0] - Last[0]);
		const double mousey = (Current[1] - Last[1]) * -1.0;
		const double mouse = fabs(mousex) > fabs(mousey) ? mousex : mousey;

		// If the user holds down either shift key, constrain translate factors to be equal ...
		if(State.modifiers.shift())
			{
				dx = dy = dz = mouse;
			}
		// If the user holds down either control key, only translate along the active axis ...
		else if(State.modifiers.control())
			{
				switch(State.active_axis)
					{
						case k3d::X:
							dx = mouse;
							break;
						case k3d::Y:
							dy = mouse;
							break;
						case k3d::Z:
							dz = mouse;
							break;
						default:
							assert_warning(0);
					}
			}
		// Otherwise, translate along the two non-active axes ...
		else
			{
				switch(State.active_axis)
					{
						case k3d::X:
							dz = mousex;
							dy = mousey;
							break;
						case k3d::Y:
							dx = mousex;
							dz = mousey;
							break;
						case k3d::Z:
							dx = mousex;
							dy = mousey;
							break;
						default:
							assert_warning(0);
					}
			}

		m_x.set_value(m_x.value() + dx);
		m_y.set_value(m_y.value() + dy);
		m_z.set_value(m_z.value() + dz);

		return true;
	}

	void on_reset_geometry()
	{
		m_output_mesh.reset();
	}

	void on_reshape_geometry()
	{
		if(m_output_mesh.empty())
			return;

		const k3d::mesh* const input = m_input_mesh.property_value();
		if(!input)
			return;

		k3d::mesh* const output = const_cast<k3d::mesh*>(m_output_mesh.value());
		if(!output)
			return;

		reshape_geometry(*input, *output);

		m_output_mesh.changed_signal().emit();
	}

	k3d::mesh* on_create_geometry()
	{
		// If we don't have any input mesh, we're done ...
		const k3d::mesh* const input = m_input_mesh.property_value();
		if(!input)
			return 0;

		// Otherwise, we make a copy of the input mesh and modify the copy ...
		k3d::mesh* const output = new k3d::mesh();
		k3d::deep_copy(*input, *output);
		reshape_geometry(*input, *output);

		return output;
	}

	void reshape_geometry(const k3d::mesh& Source, k3d::mesh& Target)
	{
		// Sanity checks ...
		assert(Source.points.size() == Target.points.size());

		// Scale points ...
		const bool ignore_selection = !k3d::contains_selection(Source);
		const k3d::matrix4 matrix = k3d::translation3D(k3d::vector3(m_x.property_value(), m_y.property_value(), m_z.property_value()));

		for(unsigned long i = 0; i != Target.points.size(); ++i)
			{
				if(ignore_selection || Source.points[i]->selected)
					{
						Target.points[i]->position = matrix * Source.points[i]->position;
					}
				else
					{
						Target.points[i]->position = Source.points[i]->position;
					}
			}
	}

	k3d::iplugin_factory& factory()
	{
		return get_factory();
	}

	static k3d::iplugin_factory& get_factory()
	{
		static k3d::plugin_factory<
			k3d::document_plugin<translate_points_implementation>,
			k3d::interface_list<k3d::imesh_source,
			k3d::interface_list<k3d::imesh_sink > > > factory(
				k3d::uuid(0x460b8dac, 0x780240a3, 0xa2f87b5b, 0xba0272cd),
				"TranslatePoints",
				"Translates mesh points",
				"Objects",
				k3d::iplugin_factory::STABLE);

		return factory;
	}

private:
	k3d_measurement_property(double, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::no_constraint) m_x;
	k3d_measurement_property(double, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::no_constraint) m_y;
	k3d_measurement_property(double, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::no_constraint) m_z;
};

/////////////////////////////////////////////////////////////////////////////
// translate_points_factory

k3d::iplugin_factory& translate_points_factory()
{
	return translate_points_implementation::get_factory();
}

} // namespace libk3ddeformation


