// K-3D
// Copyright (c) 1995-2004, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\author Romain Behar (romainbehar@yahoo.com)
*/

#include <k3dsdk/basic_math.h>
#include <k3dsdk/imaterial.h>
#include <k3dsdk/object.h>
#include <k3dsdk/persistence.h>
#include <k3dsdk/material.h>
#include <k3dsdk/material_collection.h>
#include <k3dsdk/measurement.h>
#include <k3dsdk/mesh_source.h>
#include <k3dsdk/module.h>
#include <k3dsdk/plugins.h>
#include <k3dsdk/vectors.h>

namespace libk3dmesh
{

/////////////////////////////////////////////////////////////////////////////
// knot_curve_implementation

class knot_curve_implementation :
	public k3d::material_collection<k3d::mesh_source<k3d::persistent<k3d::object> > >
{
	typedef k3d::material_collection<k3d::mesh_source<k3d::persistent<k3d::object> > > base;

public:
	knot_curve_implementation(k3d::idocument& Document) :
		base(Document),
		m_edge_count(k3d::init_name("edgecount") + k3d::init_description("Edge count [integer]") + k3d::init_value(100) + k3d::init_constraint(k3d::constraint::minimum(3UL)) + k3d::init_precision(0) + k3d::init_step_increment(1) + k3d::init_units(typeid(k3d::measurement::scalar)) + k3d::init_document(Document)),
		m_meridian_wraps(k3d::init_name("meridianwraps") + k3d::init_description("Meridian wraps [integer]") + k3d::init_value(3) + k3d::init_constraint(k3d::constraint::minimum(1UL)) + k3d::init_precision(0) + k3d::init_step_increment(1) + k3d::init_units(typeid(k3d::measurement::scalar)) + k3d::init_document(Document)),
		m_longitudinal_wraps(k3d::init_name("longitudinalwraps") + k3d::init_description("Longitudinal wraps [integer]") + k3d::init_value(4) + k3d::init_constraint(k3d::constraint::minimum(1UL)) + k3d::init_precision(0) + k3d::init_step_increment(1) + k3d::init_units(typeid(k3d::measurement::scalar)) + k3d::init_document(Document)),
		m_scale(k3d::init_name("scale") + k3d::init_description("Scale [number]") + k3d::init_value(5.0) + k3d::init_precision(2) + k3d::init_step_increment(0.1) + k3d::init_units(typeid(k3d::measurement::distance)) + k3d::init_document(Document)),
		m_width(k3d::init_name("width") + k3d::init_description("Width [number]") + k3d::init_value(0.1) + k3d::init_precision(2) + k3d::init_step_increment(0.1) + k3d::init_units(typeid(k3d::measurement::distance)) + k3d::init_document(Document)),
		m_wrap(k3d::init_name("closed") + k3d::init_description("Closed Curve [boolean]") + k3d::init_value(true) + k3d::init_document(Document))
	{
		enable_serialization(k3d::persistence::proxy(m_edge_count));
		enable_serialization(k3d::persistence::proxy(m_meridian_wraps));
		enable_serialization(k3d::persistence::proxy(m_longitudinal_wraps));
		enable_serialization(k3d::persistence::proxy(m_scale));
		enable_serialization(k3d::persistence::proxy(m_width));
		enable_serialization(k3d::persistence::proxy(m_wrap));

		register_property(m_edge_count);
		register_property(m_meridian_wraps);
		register_property(m_longitudinal_wraps);
		register_property(m_scale);
		register_property(m_width);
		register_property(m_wrap);

		m_material.changed_signal().connect(SigC::slot(*this, &knot_curve_implementation::on_reset_geometry));

		m_edge_count.changed_signal().connect(SigC::slot(*this, &knot_curve_implementation::on_reset_geometry));
		m_meridian_wraps.changed_signal().connect(SigC::slot(*this, &knot_curve_implementation::on_reset_geometry));
		m_longitudinal_wraps.changed_signal().connect(SigC::slot(*this, &knot_curve_implementation::on_reset_geometry));
		m_scale.changed_signal().connect(SigC::slot(*this, &knot_curve_implementation::on_reset_geometry));
		m_width.changed_signal().connect(SigC::slot(*this, &knot_curve_implementation::on_reset_geometry));
		m_wrap.changed_signal().connect(SigC::slot(*this, &knot_curve_implementation::on_reset_geometry));

		m_output_mesh.need_data_signal().connect(SigC::slot(*this, &knot_curve_implementation::on_create_geometry));
	}

	void on_reset_geometry()
	{
		m_output_mesh.reset();
	}

	k3d::mesh* on_create_geometry()
	{
		const unsigned long edges = m_edge_count.property_value();
		const double mwraps = m_meridian_wraps.property_value();
		const double lwraps = m_longitudinal_wraps.property_value();
		const double scale = m_scale.property_value();
		const double width = m_width.property_value();
		const double wrap = m_wrap.property_value();

		// Create a linear curve k3d::mesh ...
		std::auto_ptr<k3d::mesh> mesh(new k3d::mesh());

		k3d::linear_curve_group* const group = new k3d::linear_curve_group();
		group->material = m_material.interface();
		group->wrap = wrap;
		group->constant_data["width"] = width;

		k3d::linear_curve* const curve = new k3d::linear_curve();
		group->curves.push_back(curve);

		mesh->linear_curve_groups.push_back(group);

		// Create the curve ...
		for(unsigned long n = 0; n != edges; n++)
		{
			const double percent = static_cast<double>(n) / static_cast<double>(edges);
			const double radians = percent * k3d::pi_times_2();
			const double mu = radians * mwraps;

			k3d::vector3 point;

			point[0] = cos(mu) * (1 + cos(lwraps * mu / mwraps) / 2.0);
			point[1] = sin(mu) * (1 + cos(lwraps * mu / mwraps) / 2.0);
			point[2] = sin(lwraps * mu / mwraps) / 2.0;

			k3d::point* const p = new k3d::point(point * scale);

			mesh->points.push_back(p);
			curve->control_points.push_back(p);
		}

		return mesh.release();
	}

	k3d::iplugin_factory& factory()
	{
		return get_factory();
	}

	static k3d::iplugin_factory& get_factory()
	{
		static k3d::plugin_factory<k3d::document_plugin<knot_curve_implementation> > factory(
		k3d::uuid(0xb39c53a3, 0xe2fe0e32, 0x21d09384, 0xc4ed13fe),
			"KnotCurve",
			"Generates a PQ-knot curve",
			"Objects",
			k3d::iplugin_factory::STABLE);

		return factory;
	}

private:
	k3d_measurement_property(unsigned long, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::with_constraint) m_edge_count;
	k3d_measurement_property(unsigned long, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::with_constraint) m_meridian_wraps;
	k3d_measurement_property(unsigned long, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::with_constraint) m_longitudinal_wraps;
	k3d_measurement_property(double, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::no_constraint) m_scale;
	k3d_measurement_property(double, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::no_constraint) m_width;
	k3d_data_property(bool, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::no_constraint) m_wrap;
};

k3d::iplugin_factory& knot_curve_factory()
{
	return knot_curve_implementation::get_factory();
}

} // namespace libk3dmesh


