// K-3D
// Copyright (c) 1995-2004, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\author Bart Janssens <bart.janssens@lid.kviv.be>
*/

#include "subdiv_algorithms.h"

#include <k3dsdk/basic_math.h>
#include <k3dsdk/object.h>
#include <k3dsdk/persistence.h>
#include <k3dsdk/mesh_filter.h>
#include <k3dsdk/module.h>
#include <k3dsdk/plugins.h>
#include <k3dsdk/utility.h>

namespace libk3dmesh
{

/////////////////////////////////////////////////////////////////////////////
// link_points_implementation

class link_points_implementation :
	public k3d::mesh_filter<k3d::persistent<k3d::object> >
{
	typedef k3d::mesh_filter<k3d::persistent<k3d::object> > base;

public:
	link_points_implementation(k3d::idocument& Document) :
		base(Document)
	{
		m_input_mesh.changed_signal().connect(SigC::slot(*this, &link_points_implementation::on_reset_geometry));
		m_output_mesh.need_data_signal().connect(SigC::slot(*this, &link_points_implementation::on_create_geometry));
	}

	void on_reset_geometry()
	{
		m_output_mesh.reset();
	}

	/// Returns true iff the given object is selected
	struct is_selected
	{
		bool operator()(k3d::split_edge* const Edge)
		{
			return Edge->selected;
		}
	};

	k3d::mesh* on_create_geometry()
	{
		// Get the input geometry ...
		k3d::mesh* const input = m_input_mesh.property_value();
		if(!input)
			return 0;

		// Create output geometry ...
		std::auto_ptr<k3d::mesh> output(new k3d::mesh());
		k3d::deep_copy(*input, *output);
		for(k3d::mesh::polyhedra_t::iterator it = output->polyhedra.begin(); it != output->polyhedra.end(); ++it)
			{
				subdiv::splitter s(*(*it), output->points, true);
				s.link_points();
			}

		return output.release();
	}

	k3d::iplugin_factory& factory()
	{
		return get_factory();
	}

	static k3d::iplugin_factory& get_factory()
	{
		static k3d::plugin_factory<
			k3d::document_plugin<link_points_implementation>,
				k3d::interface_list<k3d::imesh_source,
				k3d::interface_list<k3d::imesh_sink > > > factory(
					k3d::uuid(0x2e675395, 0x247b42c0, 0x9d663a38, 0x9041b33b),
				"LinkPoints",
				"Links two selected points that belong to the same face",
				"Objects",
				k3d::iplugin_factory::EXPERIMENTAL);

		return factory;
	}
};

/////////////////////////////////////////////////////////////////////////////
// link_points_factory

k3d::iplugin_factory& link_points_factory()
{
	return link_points_implementation::get_factory();
}

} // namespace libk3dmesh


