// K-3D
// Copyright (c) 1995-2004, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\author Tim Shead (tshead@k-3d.com)
*/

#include <k3dsdk/geometry.h>
#include <k3dsdk/imaterial.h>
#include <k3dsdk/object.h>
#include <k3dsdk/persistence.h>
#include <k3dsdk/material.h>
#include <k3dsdk/material_collection.h>
#include <k3dsdk/measurement.h>
#include <k3dsdk/mesh_source.h>
#include <k3dsdk/module.h>
#include <k3dsdk/plugins.h>
#include <k3dsdk/transform.h>

namespace libk3dnurbs
{

/////////////////////////////////////////////////////////////////////////////
// nurbs_sphere_implementation

class nurbs_sphere_implementation :
	public k3d::material_collection<k3d::mesh_source<k3d::persistent<k3d::object> > >
{
	typedef k3d::material_collection<k3d::mesh_source<k3d::persistent<k3d::object> > > base;

public:
	nurbs_sphere_implementation(k3d::idocument& Document) :
		base(Document),
		m_radius(k3d::init_name("radius") + k3d::init_description("Radius [number]") + k3d::init_value(5.0) + k3d::init_precision(2) + k3d::init_step_increment(0.1) + k3d::init_units(typeid(k3d::measurement::distance)) + k3d::init_document(Document)),
		m_zmax(k3d::init_name("zmax") + k3d::init_description("ZMax [number]") + k3d::init_value(1.0) + k3d::init_precision(2) + k3d::init_step_increment(0.1) + k3d::init_units(typeid(k3d::measurement::distance)) + k3d::init_document(Document)),
		m_zmin(k3d::init_name("zmin") + k3d::init_description("ZMin [number]") + k3d::init_value(-1.0) + k3d::init_precision(2) + k3d::init_step_increment(0.1) + k3d::init_units(typeid(k3d::measurement::distance)) + k3d::init_document(Document)),
		m_thetamax(k3d::init_name("thetamax") + k3d::init_description("thetamax [number]") + k3d::init_value(k3d::radians(360.0)) + k3d::init_document(Document) + k3d::init_precision(2) + k3d::init_step_increment(k3d::radians(1.0)) + k3d::init_units(typeid(k3d::measurement::angle))),
		m_u_segments(k3d::init_name("u_segments") + k3d::init_description("Radial Segments [integer]") + k3d::init_value(4) + k3d::init_document(Document) + k3d::init_constraint(k3d::constraint::minimum(1UL)) + k3d::init_precision(0) + k3d::init_step_increment(1) + k3d::init_units(typeid(k3d::measurement::scalar))),
		m_v_segments(k3d::init_name("v_segments") + k3d::init_description("Radial Segments [integer]") + k3d::init_value(2) + k3d::init_document(Document) + k3d::init_constraint(k3d::constraint::minimum(1UL)) + k3d::init_precision(0) + k3d::init_step_increment(1) + k3d::init_units(typeid(k3d::measurement::scalar)))
	{
		enable_serialization(k3d::persistence::proxy(m_radius));
		enable_serialization(k3d::persistence::proxy(m_zmax));
		enable_serialization(k3d::persistence::proxy(m_zmin));
		enable_serialization(k3d::persistence::proxy(m_thetamax));
		enable_serialization(k3d::persistence::proxy(m_u_segments));
		enable_serialization(k3d::persistence::proxy(m_v_segments));

		register_property(m_radius);
		register_property(m_zmax);
		register_property(m_zmin);
		register_property(m_thetamax);
		register_property(m_u_segments);
		register_property(m_v_segments);

		m_material.changed_signal().connect(SigC::slot(*this, &nurbs_sphere_implementation::on_reset_geometry));
		m_radius.changed_signal().connect(SigC::slot(*this, &nurbs_sphere_implementation::on_reset_geometry));
		m_zmax.changed_signal().connect(SigC::slot(*this, &nurbs_sphere_implementation::on_reset_geometry));
		m_zmin.changed_signal().connect(SigC::slot(*this, &nurbs_sphere_implementation::on_reset_geometry));
		m_thetamax.changed_signal().connect(SigC::slot(*this, &nurbs_sphere_implementation::on_reset_geometry));
		m_u_segments.changed_signal().connect(SigC::slot(*this, &nurbs_sphere_implementation::on_reset_geometry));
		m_v_segments.changed_signal().connect(SigC::slot(*this, &nurbs_sphere_implementation::on_reset_geometry));

		m_output_mesh.need_data_signal().connect(SigC::slot(*this, &nurbs_sphere_implementation::on_create_geometry));
	}

	void on_reset_geometry()
	{
		m_output_mesh.reset();
	}

	k3d::mesh* on_create_geometry()
	{
		const double radius = m_radius.property_value();
		const double zmax = m_zmax.property_value();
		const double zmin = m_zmin.property_value();
		const double thetamax = m_thetamax.property_value();
		const unsigned long u_segments = m_u_segments.property_value();
		const unsigned long v_segments = m_v_segments.property_value();

		const double phimin = (zmin > -1) ? asin(zmin) : -k3d::pi_over_2();
		const double phimax = (zmax < 1) ? asin(zmax) : k3d::pi_over_2();

		k3d::mesh* const mesh = new k3d::mesh();
		k3d::nupatch* const nupatch = new k3d::nupatch();
		mesh->nupatches.push_back(nupatch);

		nupatch->material = m_material.interface();
		nupatch->u_order = 3;
		nupatch->v_order = 3;

		std::vector<double> v_weights;
		std::vector<k3d::vector3> v_arc_points;
		k3d::nurbs_arc(k3d::vector3(0, 1, 0), k3d::vector3(0, 0, 1), phimin, phimax, v_segments, nupatch->v_knots, v_weights, v_arc_points);

		std::vector<double> u_weights;
		std::vector<k3d::vector3> u_arc_points;
		k3d::nurbs_arc(k3d::vector3(1, 0, 0), k3d::vector3(0, 1, 0), 0, thetamax, u_segments, nupatch->u_knots, u_weights, u_arc_points);

		for(unsigned long v = 0; v != v_arc_points.size(); ++v)
			{
				const k3d::vector3 offset = radius * v_arc_points[v][2] * k3d::vector3(0, 0, 1);
				const double radius2 = radius * v_arc_points[v][1];
				const double v_weight = v_weights[v];

				for(unsigned long u = 0; u != u_arc_points.size(); ++u)
					{
						mesh->points.push_back(new k3d::point(radius2 * u_arc_points[u] + offset));
						nupatch->control_points.push_back(k3d::nupatch::control_point(mesh->points.back(), v_weight * u_weights[u]));
					}
			}

		assert_warning(is_valid(*nupatch));
		return mesh;
	}

	k3d::iplugin_factory& factory()
	{
		return get_factory();
	}

	static k3d::iplugin_factory& get_factory()
	{
		static k3d::plugin_factory<k3d::document_plugin<nurbs_sphere_implementation>, k3d::interface_list<k3d::imesh_source > > factory(
		k3d::uuid(0x082eeef0, 0x837f4277, 0xa7c21ff1, 0x839b286e),
			"NurbsSphere",
			"Generates a NURBS sphere",
			"Objects",
			k3d::iplugin_factory::STABLE);

		return factory;
	}

private:
	k3d_measurement_property(double, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::no_constraint) m_radius;
	k3d_measurement_property(double, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::no_constraint) m_zmax;
	k3d_measurement_property(double, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::no_constraint) m_zmin;
	k3d_measurement_property(double, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::no_constraint) m_thetamax;
	k3d_measurement_property(unsigned long, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::with_constraint) m_u_segments;
	k3d_measurement_property(unsigned long, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::with_constraint) m_v_segments;
};

/////////////////////////////////////////////////////////////////////////////
// nurbs_sphere_factory

k3d::iplugin_factory& nurbs_sphere_factory()
{
	return nurbs_sphere_implementation::get_factory();
}

} // namespace libk3dnurbs


