// K-3D
// Copyright (c) 1995-2004, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\brief Implements the csg_solid K-3D object, which allows geometry to be rendered with boolean effects
		\author Tim Shead (tshead@k-3d.com)
*/

#include <k3dsdk/object.h>
#include <k3dsdk/persistence.h>
#include <k3dsdk/module.h>
#include <k3dsdk/renderman.h>

namespace libk3drenderman
{

/////////////////////////////////////////////////////////////////////////////////////////////////
// csg_solid_implementation

class csg_solid_implementation :
	public k3d::persistent<k3d::object> ,
	public k3d::ri::irenderable
{
	typedef k3d::persistent<k3d::object>  base;
	
public:
	csg_solid_implementation(k3d::idocument& Document) :
		base(Document),
		m_instance(k3d::init_name("instance") + k3d::init_description("Instance [object]") + k3d::init_object_value(0) + k3d::init_document(Document))
	{
		enable_serialization(k3d::persistence::object_proxy(m_instance));
		register_property(m_instance);
	}

	void renderman_pre_render(const k3d::ri::render_state&)
	{
	}

	void renderman_render(const k3d::ri::render_state& State)
	{
		// We only generate RIB during CSG rendering ...
		if(State.render_context != k3d::ri::CSG_SOLID)
			return;
			
		// We only generate RIB on the last sample ...
		if(!k3d::ri::last_sample(State))
			return;

		k3d::ri::irenderable* const renderable = m_instance.interface();
		if(!renderable)
			return;

		// Make sure we don't enter an infinite loop trying to render ourself ...
		if(renderable == this)
			{
				std::cerr << error << factory().name() << " [" << name() << "] cannot instance itself" << std::endl;
				return;
			}

		State.engine.RiSolidBegin(k3d::ri::RI_PRIMITIVE());
			k3d::ri::render_state state(State);
			state.render_context = k3d::ri::CSG_SOLID;
			renderable->renderman_render(state);
		State.engine.RiSolidEnd();
	}

	k3d::iplugin_factory& factory()
	{
		return get_factory();
	}
	
	static k3d::iplugin_factory& get_factory()
	{
		static k3d::plugin_factory<k3d::document_plugin<csg_solid_implementation> >factory(
			k3d::uuid(0x00000001, 0x00000000, 0x00000000, 0x0000007b),
			"RenderManCSGSolid",
			"",
			"Objects",
			k3d::iplugin_factory::STABLE);

		return factory;	
	}

private:
	k3d_object_property(k3d::ri::irenderable, k3d::immutable_name, k3d::with_undo, k3d::local_storage) m_instance;
};

k3d::iplugin_factory& csg_solid_factory()
{
	return csg_solid_implementation::get_factory();
}

} // namespace libk3drenderman

