// K-3D
// Copyright (c) 1995-2004, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\brief Uses code from Aqsis, http://www.aqsis.com to implement Ken Perlin's noise functions
		\author Tim Shead (tshead@k-3d.com)
*/

#include "noise.h"

#pragma warning(disable : 4305)

namespace k3d
{

namespace detail
{

#define NOISE_B 0x100
#define NOISE_BM 0xff
#define NOISE_N 0x1000

#define LERP(t,x0,x1)  ((x0) + (t)*((x1)-(x0)))

#define s_curve(t) ( t * t * (3.0 - 2.0 * t) )

#define setup(i, b0, b1, r0, r1) \
	t = vec[i] + NOISE_N; \
	b0 = ((int)t) & NOISE_BM; \
	b1 = (b0+1) & NOISE_BM; \
	r0 = t - (int)t;\
	r1 = r0 - 1.;

static float randtable[] =
{
	0.80874050111, 0.19330423902, 0.56358531449, 0.00125125889,
	0.89596240120, 0.35029145177, 0.47987304300, 0.58500930815,
	0.85894344920, 0.17410809656, 0.74660481582, 0.82284005249,
	0.01498458815, 0.30399487289, 0.51353495895, 0.71050141911,
	0.16589861751, 0.14731284524, 0.36445204016, 0.09140293588,
	0.00466933195, 0.11908322398, 0.44569231239, 0.98852504044,
	0.57118442335, 0.53166295358, 0.37788018433, 0.00891140477,
	0.66304513688, 0.16623432112, 0.60716574602, 0.60176396985,
	0.60768456069, 0.05703909421, 0.35212256233, 0.45078890347,
	0.30195013276, 0.51988280892, 0.80260628071, 0.78331858272,
	0.92571794794, 0.95590075381, 0.72667622913, 0.87597277749,
	0.23532822657, 0.46208075198, 0.14233832820, 0.53935361797,
	0.84365367595, 0.77965636158, 0.20960112308, 0.86223944823,
	0.39243751335, 0.61149937437, 0.99969481491, 0.99679555651,
	0.02374340037, 0.84014404736, 0.29728080081, 0.26621295816,
	0.05621509445, 0.67720572527, 0.09262367626, 0.37586596271,
	0.27289651173, 0.27588732566, 0.91879024628, 0.00878933073,
	0.72649311808, 0.83761101108, 0.69118320261, 0.58790856655,
	0.46845912046, 0.74373607593, 0.20535905026, 0.48493911557,
	0.10827967162, 0.74443800165, 0.94915616321, 0.45796075320,
	0.60896633808, 0.73500778222, 0.38523514512, 0.59904782250,
	0.22510452589, 0.15155491806, 0.36133915220, 0.57240516373,
	0.98998992889, 0.51710562456, 0.80288094729, 0.42515335551,
	0.65730765709, 0.16898098697, 0.34556108280, 0.75154881436,
	0.50480666524, 0.69975890378, 0.06353953673, 0.49189733573,
	0.90511795404, 0.14157536546, 0.94958342235, 0.14749595630,
	0.07037568285, 0.42655720695, 0.30304879910, 0.69289223914,
	0.87725455489, 0.15323343608, 0.68318735313, 0.96661275063,
	0.17789239174, 0.19135105441, 0.58204901273, 0.82168034913,
	0.50392162847, 0.15555284280, 0.47526474807, 0.81719412824,
	0.56874294259, 0.27958006531, 0.40559099094, 0.73201696829,
	0.47529526658, 0.72191534165, 0.75585192419, 0.68224127934,
	0.03509628590, 0.83468123417, 0.36780907620, 0.12302011170,
	0.10467848750, 0.42622150334, 0.66298409986, 0.51701406903,
	0.34598834193, 0.54954680013, 0.92138431959, 0.94933927427,
	0.31687368389, 0.84698019349, 0.37498092593, 0.47172460097,
	0.29779961547, 0.98297067171, 0.27188940092, 0.45609912412,
	0.76131473739, 0.19598986785, 0.56727805414, 0.73918881802,
	0.89016388440, 0.50090029603, 0.39765617847, 0.83944212165,
	0.05050813318, 0.57258827479, 0.99462874233, 0.02746665853,
	0.62675862911, 0.84304330576, 0.19406720176, 0.53132724998,
	0.12332529679, 0.84215826899, 0.19785149693, 0.65761284219,
	0.94106875820, 0.31406598102, 0.74312570574, 0.10992767113,
	0.73308511612, 0.14026306955, 0.33631397443, 0.28608050783,
	0.74721518601, 0.60023804437, 0.70799890133, 0.83462019715,
	0.06100650044, 0.00161748100, 0.14447462386, 0.25272377697,
	0.11560411390, 0.21057771538, 0.85262611774, 0.80623798334,
	0.45451216163, 0.11377300333, 0.01425214393, 0.55320902127,
	0.07388531144, 0.54344309824, 0.68614764855, 0.75222022156,
	0.29035309915, 0.69621875668, 0.20194097720, 0.43671987060,
	0.53257850887, 0.57786797693, 0.23242896817, 0.43668935209,
	0.96304208502, 0.50413525803, 0.16019165624, 0.62868129521,
	0.33594775231, 0.18994720298, 0.92480239265, 0.69576097903,
	0.99801629688, 0.45744193854, 0.99517807550, 0.17835016938,
	0.43772698141, 0.09439374981, 0.62517166662, 0.09750663778,
	0.29001739555, 0.89461958678, 0.04843287454, 0.93151646474,
	0.20197149571, 0.41071810053, 0.76906643880, 0.22730185858,
	0.46635334330, 0.45161290323, 0.60414441359, 0.62807092502,
	0.82879116184, 0.85479293191, 0.63472396008, 0.59782708213,
	0.37513351848, 0.56575212867, 0.72090823084, 0.62477492599,
	0.90508743553, 0.55513168737, 0.73790704062, 0.18427076022,
	0.69850764489, 0.60472426527, 0.18894009217, 0.24286629841,
	0.08038575396, 0.49446089053, 0.35129856258, 0.58461256752,
	0.69112216559, 0.62038026063, 0.61204870754, 0.74074526200,
	0.86773277993, 0.57603686636, 0.14911343730, 0.80452894681,
	0.04321420942, 0.72768333995, 0.61470381787, 0.91155735954,
	0.56920072024, 0.31501205481, 0.97653126621, 0.66777550584,
	0.86904507584, 0.10855433821, 0.17392498550, 0.30582598346,
	0.32691427351, 0.15488143559, 0.74431592761, 0.85122226630,
	0.82000183111, 0.64098025452, 0.07660145878, 0.07934812464,
	0.29874568926, 0.40897854549, 0.44825586718, 0.54509109775,
	0.32303842280, 0.15265358440, 0.50120548112, 0.46555986206,
	0.95907467879, 0.82668538469, 0.31388286996, 0.73799859615,
	0.94399853511, 0.30005798517, 0.72502822962, 0.87334818567,
	0.52458265938, 0.78496658223, 0.06573686941, 0.12723166601,
	0.87563707389, 0.07226783044, 0.95611438337, 0.60963774529,
	0.50505081332, 0.10480056154, 0.32212286752, 0.65385906552,
	0.55116428114, 0.91998046815, 0.29029206214, 0.22708822901,
	0.37913144322, 0.49253822443, 0.11453596606, 0.66280098880,
	0.38236640522, 0.50926236763, 0.79335917234, 0.49681081576,
	0.39518417920, 0.60628070925, 0.53215124973, 0.68816187017,
	0.62306588946, 0.10061952574, 0.70787682730, 0.00589007233,
	0.49690237129, 0.74733726005, 0.49150059511, 0.86324655904,
	0.35709707938, 0.55281228065, 0.78536332286, 0.38010803552,
	0.37424848170, 0.17658009583, 0.63084810938, 0.95571764275,
	0.61198767052, 0.95171971801, 0.74327829829, 0.13162633137,
	0.63921018097, 0.05590990936, 0.32984405042, 0.02783288064,
	0.59688100833, 0.86431470687, 0.84707174902, 0.13162633137,
	0.12646870327, 0.01467940306, 0.85396893216, 0.72164067507,
	0.06595049898, 0.21756645405, 0.61714529862, 0.70790734581,
	0.31940672018, 0.34098330638, 0.62410351878, 0.16891994995,
	0.80687887204, 0.80239265114, 0.66100039674, 0.36756492813,
	0.90060121464, 0.79818109684, 0.61110263375, 0.52653584399,
	0.25370036927, 0.40241706595, 0.63017670217, 0.14481032746,
	0.42780846583, 0.06616412854, 0.85518967254, 0.13653981140,
	0.22556230354, 0.54805139317, 0.30228583636, 0.57335123753,
	0.13470870083, 0.80803857540, 0.11062959685, 0.31134983367,
	0.78963591418, 0.89522995697, 0.78810998871, 0.28424939726,
	0.85665456099, 0.36112552263, 0.61522263253, 0.74379711295,
	0.24954985199, 0.22943815424, 0.86358226264, 0.22849208045,
	0.08142338328, 0.05380413221, 0.98483230079, 0.54240546892,
	0.25879696036, 0.09466841639, 0.42680135502, 0.52467421491,
	0.12509537034, 0.14654988250, 0.23276467177, 0.89153721732,
	0.05871761223, 0.04709006012, 0.08011108737, 0.93163853877,
	0.43278298288, 0.39860225227, 0.91470076601, 0.33640552995,
	0.84209723197, 0.53422650838, 0.83718375195, 0.94616534928,
	0.00433362835, 0.25916318247, 0.39768669698, 0.69353312784,
	0.24109622486, 0.39869380779, 0.95480208747, 0.52558977020,
	0.94528031251, 0.68401135289, 0.25513473922, 0.58555864132,
	0.94097720267, 0.00717184973, 0.89022492141, 0.43549913022,
	0.14242988372, 0.57667775506, 0.78615680410, 0.60155034028,
	0.41792046876, 0.00427259133, 0.38300729392, 0.22232734153,
	0.06485183264, 0.85509811701, 0.65993224891, 0.08224738304,
	0.80269783624, 0.69148838771, 0.66206854457, 0.81105990783,
	0.68950468459, 0.14276558733, 0.68562883389, 0.53013702811,
	0.86867885372, 0.03106784265, 0.77773369549, 0.72789696951,
	0.55198828089, 0.08545182653, 0.70656453139, 0.64452040162,
	0.14517654958, 0.27497177038, 0.05877864925, 0.94790490432,
	0.92248298593, 0.29224524674, 0.61998352000, 0.98178044984,
	0.15591906491, 0.21863460189, 0.69454023865, 0.36753440962,
	0.10641804254, 0.90227973266, 0.52143925291, 0.24054689169,
	0.78209784234, 0.08011108737, 0.44172490616, 0.90264595477,
	0.87038789026, 0.77587206641, 0.97439497055, 0.17178868984,
	0.75063325907, 0.00375377667, 0.45661793878, 0.21063875240,
	0.99261452071, 0.31113620411, 0.40470595416, 0.11404766991,
	0.24768822291, 0.18927579577, 0.25208288827, 0.03857539598,
	0.93908505509, 0.88537247841, 0.62031922361, 0.15350810266,
	0.65675832392, 0.64555803095, 0.77965636158, 0.19565416425,
	0.66417432173, 0.92129276406, 0.45545823542, 0.90914639729,
	0.42204046754, 0.56953642384, 0.63634144108, 0.15076143681,
	0.53645435957, 0.57856990265, 0.54057435835, 0.94302194281,
	0.03662221137, 0.35026093326, 0.39695425275, 0.25528733177,
	0.38941618091, 0.07028412732, 0.19650868252, 0.79525131993,
	0.15588854640, 0.19766838588, 0.07092501602, 0.59074678793,
	0.69734794153, 0.60429700613, 0.45432905057, 0.64433729057,
	0.83571886349, 0.39652699362, 0.68446913053, 0.44129764702,
	0.87600329600, 0.94946134831, 0.19959105197, 0.59221167638,
	0.89571825312, 0.18530838954, 0.98745689261, 0.39170506912,
	0.70851771599, 0.62733848079, 0.44206060976, 0.57444990387,
	0.40763573107, 0.26020081179, 0.28556169317, 0.04962309641,
	0.89605395672, 0.72844630268, 0.71019623402, 0.89532151250,
	0.30860316782, 0.90386669515, 0.39741203040, 0.39356669820,
	0.73369548631, 0.35355693228, 0.57057405316, 0.38810388501,
	0.13931699576, 0.73943296609, 0.73638111515, 0.74535355693,
	0.91113010041, 0.68044068728, 0.27259132664, 0.20017090365,
	0.90743736076, 0.10937833796, 0.51765495773, 0.36783959471,
	0.43852046266, 0.65483565783, 0.51796014283, 0.20307016205,
	0.07577745903, 0.07956175420, 0.09042634358, 0.68764305551,
	0.69728690451, 0.30722983489, 0.35508285775, 0.02780236213,
	0.67574083682, 0.06784264657, 0.39484847560, 0.14267403180,
	0.61571092868, 0.69402142399, 0.19870601520, 0.72493667409,
	0.54536576434, 0.09830011902, 0.54399243141, 0.65489669485,
	0.96996978668, 0.46403393658, 0.97457808161, 0.04962309641,
	0.69280068361, 0.67992187262, 0.53096102786, 0.72786645100,
	0.80977813044, 0.04995880001, 0.38862269967, 0.37211218604,
	0.78447828608, 0.31055635243, 0.06082338939, 0.16956083865,
	0.00915555284, 0.48466444899, 0.32190923795, 0.21246986297,
	0.75853755303, 0.60594500565, 0.50785851619, 0.43986327708,
	0.94232001709, 0.84264656514, 0.34980315561, 0.40107425153,
	0.56285287027, 0.05346842860, 0.99728385266, 0.62331003754,
	0.28421887875, 0.30640583514, 0.38676107059, 0.86355174413,
	0.01861629078, 0.39118625446, 0.61256752220, 0.02670369579,
	0.93676564837, 0.85793633839, 0.45570238350, 0.03103732414,
	0.80837427900, 0.61159092990, 0.72728659932, 0.40189825129,
	0.07901242103, 0.59501937925, 0.58259834590, 0.87542344432,
	0.58943449202, 0.59062471389, 0.46220282601, 0.79317606128,
	0.34357737968, 0.67275002289, 0.81606494339, 0.56920072024,
	0.23209326456, 0.34156315806, 0.26731162450, 0.01022370067,
	0.11685537278, 0.86986907559, 0.37174596393, 0.77101962340,
	0.18301950133, 0.89394817957, 0.62892544328, 0.72557756279,
	0.96520889920, 0.78008362072, 0.90200506607, 0.53556932279,
	0.95339823603, 0.79741813410, 0.90154728843, 0.19736320078,
	0.18546098209, 0.62871181372, 0.89034699545, 0.84881130406,
	0.73220007935, 0.25064851833, 0.26499221778, 0.61693166906,
	0.48094119083, 0.71388897366, 0.17093417158, 0.78810998871,
	0.12463759270, 0.85534226508, 0.80654316843, 0.08148442030,
	0.76204718162, 0.47297585986, 0.57332071902, 0.30747398297,
	0.01535081027, 0.99502548296, 0.72777489547, 0.19186986908,
	0.07211523789, 0.17413861507, 0.69164098025, 0.64763328959,
	0.55665761284, 0.83892330699, 0.67662587359, 0.27497177038,
	0.63823358867, 0.23252052370, 0.77950376904, 0.37122714927,
	0.28174687948, 0.97103793451, 0.20459608753, 0.23166600543,
	0.04696798608, 0.15182958464, 0.76921903134, 0.02288888211,
	0.34324167608, 0.64387951292, 0.34858241523, 0.00924710837,
	0.54518265328, 0.09067049165, 0.41651661733, 0.41456343272,
	0.87624744407, 0.65752128666, 0.33799249245, 0.69927060762,
	0.46577349162, 0.05398724326, 0.57002471999, 0.53547776727,
	0.48658711509, 0.86333811457, 0.42680135502, 0.25998718223,
	0.77846613971, 0.70940275277, 0.56581316568, 0.08743552965,
	0.86437574389, 0.29300820948, 0.85055085910, 0.86422315134,
	0.76561784722, 0.82863856929, 0.74684896390, 0.64491714225,
	0.52195806757, 0.91253395184, 0.90832239753, 0.87191381573,
	0.82305368206, 0.35401470992, 0.10763878292, 0.44016846217,
	0.65746024964, 0.00085451827, 0.16983550523, 0.42771691031,
	0.38566240425, 0.08981597339, 0.06372264779, 0.77355265969,
	0.14285714286, 0.15408795434, 0.80761131626, 0.68388927885,
	0.97842341380, 0.01855525376, 0.34614093448, 0.04846339305,
	0.63747062593, 0.91415143284, 0.05365153966, 0.64763328959,
	0.39194921720, 0.24259163182, 0.02105777154, 0.43171483505,
	0.82604449599, 0.02282784509, 0.78188421278, 0.65401165807,
	0.98348948637, 0.55040131840, 0.63426618244, 0.13843195898,
	0.78215887936, 0.30069887387, 0.21478926969, 0.46342356639,
	0.10251167333, 0.60802026429, 0.48060548723, 0.34253975036,
	0.45640430921, 0.15088351085, 0.75582140568, 0.98443556017,
	0.64165166173, 0.77550584429, 0.99874874111, 0.71163060396,
	0.28980376598, 0.68816187017, 0.94610431227, 0.62120426038,
	0.71236304819, 0.54823450423, 0.29038361766, 0.56715598010,
	0.95205542161, 0.49262977996, 0.54667806024, 0.24643696402,
	0.44575334941, 0.62337107456, 0.76516006958, 0.66982024598,
	0.20462660604, 0.98892178106, 0.74553666799, 0.79876094852,
	0.91839350566, 0.59129612110, 0.90981780450, 0.62947477645,
	0.88933988464, 0.44972075564, 0.48939481796, 0.50553910947,
	0.20300912503, 0.25196081423, 0.18292794580, 0.94039735099,
	0.04135258034, 0.62508011109, 0.94112979522, 0.73049104282,
	0.54448072756, 0.55635242775, 0.03451643422, 0.64134647664,
	0.12775048067, 0.99603259377, 0.06137272256, 0.44633320109,
	0.66023743400, 0.17279580065, 0.05035554064, 0.60090945158,
	0.41053498947, 0.65358439894, 0.31595812860, 0.33243812372,
	0.16644795068, 0.67851802118, 0.22162541581, 0.42750328074,
	0.43507187109, 0.80272835475, 0.50764488662, 0.49754325999,
	0.12857448042, 0.37589648122, 0.33948789941, 0.00451673940,
	0.64333017975, 0.89114047670, 0.73329874569, 0.82732627338,
	0.73717459639, 0.73397015290, 0.51448103275, 0.65059358501,
	0.11218604083, 0.36326181829, 0.46583452864, 0.95709097568,
	0.10867641224, 0.45478682821, 0.82122257149, 0.90466017640,
	0.41666920988, 0.03912472915, 0.76137577441, 0.86944181646,
	0.21127964110, 0.16486098819, 0.06775109104, 0.93612475967,
	0.61827448347, 0.11835077975, 0.59111301004, 0.11691640980,
	0.91564683981, 0.73155919065, 0.69029816584, 0.58009582812,
	0.37818536943, 0.80886257515, 0.29850154118, 0.66972869045,
	0.17056794946, 0.20441297647, 0.77251503037, 0.99749748222,
	0.19562364574, 0.58098086489, 0.18121890927, 0.38825647755,
	0.30399487289, 0.38172551653, 0.19025238807, 0.96014282662,
	0.77117221595, 0.04623554186, 0.19815668203, 0.12082277902,
	0.44862208930, 0.00177007355, 0.88103885006, 0.76104007080,
	0.49485763115, 0.46134830775, 0.80153813288, 0.42637409589,
	0.09274575030, 0.55940427870, 0.23850215155, 0.74938200018,
	0.05963316752, 0.85702078310, 0.89382610553, 0.92202520829,
	0.75606555376, 0.74721518601, 0.63203833125, 0.49089022492,
	0.19962157048, 0.79198583941, 0.90990936003, 0.25574510941,
	0.79036835841, 0.52217169713, 0.05124057741, 0.55284279916,
	0.55336161382, 0.73174230171, 0.02404858547, 0.70415356914,
	0.30542924284, 0.79692983795, 0.87554551836, 0.91421246986,
	0.65196691794, 0.05258339183, 0.48188726463, 0.39494003113,
	0.21973326823, 0.75618762780, 0.76857814264, 0.00085451827,
	0.03268532365, 0.67061372723, 0.05752739036, 0.15475936155,
	0.43626209296, 0.79815057833, 0.69786675619, 0.34586626789,
	0.47697378460, 0.29529709769, 0.63554795984, 0.98171941282,
	0.14322336497, 0.10098574786, 0.07928708762, 0.87435529649,
	0.37382122257, 0.30872524186, 0.91799676504, 0.14300973540,
	0.23999755852, 0.81420331431, 0.04663228248, 0.81383709220,
	0.20801416059, 0.26935636464, 0.25650807215, 0.89788506729,
	0.88769188513, 0.87484359264, 0.84615619373, 0.22937711722,
	0.17352824488, 0.04089480270, 0.60927152318, 0.84478286081,
	0.18506424146, 0.44926297800, 0.35590685751, 0.18375194555,
	0.91167943358, 0.54634235664, 0.62303537095, 0.89446699423,
	0.96932889798, 0.99932859279, 0.73064363536, 0.52259895627,
	0.37440107425, 0.14667195654, 0.84020508438, 0.86742759484,
	0.74181340983, 0.97375408185, 0.17096469008, 0.28437147130,
	0.81270790735, 0.64503921628, 0.03088473159, 0.71660512101,
	0.71889400922, 0.12463759270, 0.81817072054, 0.73592333750,
	0.88967558824, 0.91543321024, 0.64983062227, 0.37507248146,
	0.00756859035, 0.44886623737, 0.77349162267, 0.03216650899,
	0.91903439436, 0.15659047212, 0.41846980193, 0.22861415448,
	0.90652180548, 0.75447859127, 0.24286629841, 0.50169377728,
	0.22824793237, 0.77117221595, 0.03210547197, 0.21347697378,
	0.33161412397, 0.85201574755, 0.80639057588, 0.55659657582,
	0.64687032685, 0.86935026093, 0.99179052095, 0.53892635884,
	0.36689352092, 0.37104403821, 0.81279946287, 0.22772911771,
	0.11978514969, 0.39786980804, 0.56129642628, 0.03497421186,
	0.94018372143, 0.20148319956, 0.89901425214, 0.32493057039,
	0.44557023835, 0.89394817957, 0.84933011872, 0.12506485183,
	0.62404248177, 0.56007568590, 0.63475447859, 0.22211371197,
	0.13962218085, 0.86413159581, 0.01647999512, 0.42365794855,
	0.56144901883, 0.68166142766, 0.29197058016, 0.26740318003,
	0.42271187475, 0.78823206275, 0.60722678304, 0.95883053072,
	0.51652577288, 0.13577684866, 0.73946348460, 0.81838435011,
	0.33387249367, 0.74831385235, 0.00698873867, 0.02008117924,
	0.05050813318, 0.76790673544, 0.46226386303, 0.90212714011,
	0.96908474990, 0.68242439039, 0.02417065950, 0.63097018342,
	0.82235175634, 0.26593829157, 0.70638142033, 0.50044251839,
	0.84423352763, 0.26361888485, 0.59721671194, 0.64662617878,
	0.01449629200, 0.35160374767, 0.74816125980, 0.46006653035,
	0.38361766411, 0.40388195441, 0.62648396252, 0.47654652547,
	0.88387707144, 0.59685048982, 0.28882717368, 0.02761925108,
	0.53312784204, 0.38297677541, 0.93154698325, 0.43217261269,
	0.43742179632, 0.86519974364, 0.97424237800, 0.01007110813,
	0.49855037080, 0.60234382153, 0.33295693838, 0.69963682974,
	0.23303933836, 0.37821588794, 0.03900265511, 0.55961790826,
	0.60881374554, 0.96411023286, 0.85952330088, 0.93734550005,
	0.30530716880, 0.40702536088, 0.07644886624, 0.58397167882,
	0.30426953948, 0.87478255562, 0.08005005036, 0.49858088931,
	0.21622363964, 0.58842738121, 0.93288979766, 0.78151799066,
	0.25006866665, 0.87441633351, 0.56071657460, 0.27930539872,
	0.23108615375, 0.86660359508, 0.72713400678, 0.30140079958,
	0.57432782983, 0.89034699545, 0.44679097873, 0.19071016572,
	0.61919003876, 0.05923642689, 0.99523911252, 0.02877895444,
	0.23148289438, 0.11612292856, 0.16983550523, 0.25693533128,
	0.07962279122, 0.04928739280, 0.95980712302, 0.49488814966,
	0.49055452132, 0.51554918058, 0.27973265786, 0.33765678884,
	0.29215369121, 0.60106204413, 0.92471083712, 0.41227454451,
	0.54835657826, 0.95132297739, 0.40165410321, 0.47575304422,
	0.59285256508, 0.80166020692, 0.79442732017, 0.83492538224,
	0.26792199469, 0.93917661061, 0.31806390576, 0.87008270516,
	0.77489547411, 0.12762840663, 0.71877193518, 0.94332712790,
	0.04162724693, 0.02667317728, 0.17590868862, 0.47288430433,
	0.01446577349, 0.13611255226, 0.86230048524, 0.45478682821,
	0.77965636158, 0.79720450453, 0.81853694266, 0.21277504807,
	0.92150639363, 0.78142643513, 0.29874568926, 0.65849787896,
	0.02301095614, 0.49876400037, 0.24262215033, 0.84212775048,
	0.71532334361, 0.35621204260, 0.03088473159, 0.07571642201,
	0.18210394604, 0.29792168950, 0.18042542802, 0.40980254524,
	0.52562028871, 0.05822931608, 0.13473921934, 0.74489577929,
	0.89986877041, 0.15881832331, 0.80672627949, 0.35886715293,
	0.81807916501, 0.18143253883, 0.02923673208, 0.13180944243,
	0.53993346965, 0.34690389721, 0.57069612720, 0.24921414838,
	0.94528031251, 0.93252357555, 0.87075411237, 0.66490676595,
	0.80025635548, 0.55796990875, 0.15323343608, 0.10519730216,
	0.82241279336, 0.87231055635, 0.07251197851, 0.89513840144,
	0.45094149602, 0.67195654164, 0.54878383740, 0.38828699606,
	0.72804956206, 0.98214667196, 0.94946134831, 0.28138065737,
	0.19794305246, 0.13196203497, 0.27237769707, 0.59480574969,
	0.22992645039, 0.35532700583, 0.48735007782, 0.60051271096,
	0.67113254189, 0.94814905240, 0.69179357280, 0.47843867306,
	0.23581652272, 0.51356547746, 0.23752555925, 0.98257393109,
	0.53749198889, 0.32645649586, 0.48435926389, 0.89803765984,
	0.16922513504, 0.34815515610, 0.33652760399, 0.57078768273,
	0.73967711417, 0.92281868954, 0.21927549059, 0.29016998810,
	0.27008880886, 0.68669698172, 0.65044099246, 0.80700094607,
	0.85906552324, 0.72176274911, 0.06442457350, 0.76686910611,
	0.02078310495, 0.48429822687, 0.04437391278, 0.34156315806,
	0.60405285806, 0.54057435835, 0.61851863155, 0.83819086276,
	0.69673757134, 0.64003418073, 0.77736747337, 0.56868190558,
	0.42350535600, 0.15799432356, 0.86800744652, 0.75994140446,
	0.04214606159, 0.43836787011, 0.02142399365, 0.78646198920,
	0.24485000153, 0.06826990570, 0.26926480911, 0.50630207221,
	0.07101657155, 0.61552781762, 0.50645466475, 0.98205511643,
	0.33747367779, 0.86007263405, 0.95553453169, 0.36750389111,
	0.99270607624, 0.05096591082, 0.28952909940, 0.24188970611,
	0.30091250343, 0.22013000885, 0.06891079440, 0.17246009705,
	0.26276436659, 0.08368175298, 0.64961699271, 0.78685872982,
	0.17825861385, 0.56257820368, 0.21753593554, 0.13647877438,
	0.75884273812, 0.45371868038, 0.10690633869, 0.63228247932,
	0.69429609058, 0.28250984222, 0.05740531632, 0.33381145665,
	0.08120975372, 0.86849574267, 0.56758323923, 0.87325663015,
	0.61885433515, 0.63499862667, 0.99966429640, 0.79540391247,
	0.97888119144, 0.19425031281, 0.59312723167, 0.59477523118,
	0.25513473922, 0.23905148473, 0.04229865413, 0.23151341288,
	0.90005188147, 0.27246925260, 0.79494613483, 0.40681173132,
	0.05294961394, 0.20648823511, 0.72756126591, 0.37943662832,
	0.66811120945, 0.36124759667, 0.35398419141, 0.60457167272,
	0.34696493423, 0.89577929014, 0.22363963744, 0.97604297006,
	0.28235724967, 0.20166631062, 0.66512039552, 0.95352031007,
	0.21997741630, 0.29099398785, 0.53199865719, 0.35544907987,
	0.84380626850, 0.70934171575, 0.10672322764, 0.09961241493,
	0.49134800256, 0.00576799829, 0.81914731285, 0.67186498611,
	0.60582293161, 0.29062776574, 0.93533127842, 0.60463270974,
	0.83010345775, 0.91222876675, 0.87383648183, 0.74691000092,
	0.15012054811, 0.78423413800, 0.30487990967, 0.60591448714,
	0.99319437239, 0.39463484603, 0.68904690695, 0.54261909848,
	0.68742942595, 0.02743614002, 0.41160313730, 0.10193182165,
	0.18900112918, 0.22864467299, 0.16901150548, 0.25888851588,
	0.48323007904, 0.04760887478, 0.76329844050, 0.86220892972,
	0.56547746208, 0.08499404889, 0.96124149297, 0.05615405744,
	0.40870387890, 0.87865840632, 0.57390057070, 0.96255378887,
	0.82335886715, 0.65852839747, 0.41349528489, 0.40314951018,
	0.61626026185, 0.00222785119, 0.28312021241, 0.94518875698,
	0.95449690237, 0.42194891202, 0.65605639821, 0.04943998535,
	0.34055604724, 0.41438032167, 0.20087282937, 0.60396130253,
	0.10525833918, 0.00485244301, 0.80840479751, 0.62312692648,
	0.82424390393, 0.31678212836, 0.27686391797, 0.39442121647,
	0.56468398083, 0.19766838588, 0.31153294473, 0.85351115452,
	0.36848048341, 0.43324076052, 0.99511703848, 0.46906949065,
	0.87774285104, 0.58043153172, 0.38486892300, 0.41877498703,
	0.50914029359, 0.95104831080, 0.39338358715, 0.33741264077,
	0.08365123447, 0.77675710318, 0.60829493088, 0.41859187597,
	0.31727042451, 0.08734397412, 0.20792260506, 0.15118869594,
	0.07931760613, 0.33277382733, 0.62913907285, 0.47181615650,
	0.82497634816, 0.28070925016, 0.14282662435, 0.61452070681,
	0.10733359783, 0.22095400861, 0.99441511277, 0.43906979583,
	0.57625049593, 0.00735496078, 0.26459547716, 0.51899777215,
	0.36942655721, 0.96920682394, 0.10751670888, 0.77056184576,
	0.30262153996, 0.22266304514, 0.52171391949, 0.90398876919,
	0.76473281045, 0.41441084017, 0.31061738945, 0.47846919156,
	0.72417371136, 0.42747276223, 0.38911099582, 0.84160893582,
	0.75307473983, 0.08172856838, 0.09140293588, 0.73491622669,
	0.47502059999, 0.35963011567, 0.63780632954, 0.84698019349,
	0.34311960204, 0.18836024049, 0.41529587695, 0.09439374981,
	0.15427106540, 0.31031220435, 0.22931608020, 0.91338847011,
	0.20593890194, 0.27228614154, 0.18228705710, 0.34153263955,
	0.50190740684, 0.06543168432, 0.33527634510, 0.17246009705,
	0.91299172948, 0.44645527512, 0.80925931578, 0.96368297372,
	0.69924008911, 0.47209082308, 0.48191778314, 0.17926572466,
	0.29953917051, 0.27744376965, 0.93670461135, 0.24860377819,
	0.79506820887, 0.62990203558, 0.81963560900, 0.82244331187,
	0.95297097690, 0.28952909940, 0.26096377453, 0.00720236824,
	0.25617236854, 0.70751060518, 0.51744132817, 0.85946226386,
	0.82598345897, 0.09353923154, 0.49125644704, 0.92104861599,
	0.64326914274, 0.38386181219, 0.28263191626, 0.07669301431,
	0.97497482223, 0.14072084719, 0.43153172399, 0.51509140294,
	0.41166417432, 0.91479232154, 0.00518814661, 0.07461775567,
	0.42313913388, 0.10171819208, 0.18765831477, 0.75533310953,
	0.83718375195, 0.82421338542, 0.52742088076, 0.67793816950,
	0.14899136326, 0.87044892727, 0.49949644459, 0.27149266030,
	0.06674398022, 0.22067934202, 0.53837702567, 0.97079378643,
	0.19953001495, 0.19290749840, 0.19772942289, 0.20462660604,
	0.29438154241, 0.73879207739, 0.59349345378, 0.95678579058,
	0.92843409528, 0.92132328257, 0.78612628559, 0.67983031709,
	0.06189153722, 0.10544145024, 0.25898007141, 0.84432508316,
	0.35941648610, 0.89275795770, 0.14831995605, 0.90377513962,
	0.68456068606, 0.33146153142, 0.38157292398, 0.95669423505,
	0.67464217048, 0.43327127903, 0.38010803552, 0.88753929258,
	0.80205694754, 0.52784813990, 0.48283333842, 0.26709799493,
	0.82778405103, 0.48667867061, 0.74633014924, 0.79412213507,
	0.05413983581, 0.34717856380, 0.27878658406, 0.55903805658,
	0.28238776818, 0.36848048341, 0.76290169988, 0.91125217444,
	0.88161870174, 0.05014191107, 0.32456434828, 0.21124912259,
	0.87527085177, 0.14487136448, 0.40382091739, 0.24881740776,
	0.45170445875, 0.19684438612, 0.78365428632, 0.60811181982,
	0.80455946532, 0.63646351512, 0.43333231605, 0.76177251503,
	0.70940275277, 0.96319467757, 0.32267220069, 0.17444380016,
	0.74321726127, 0.04730368969, 0.86288033692, 0.19754631184,
	0.55269020661, 0.08236945708, 0.13083285012, 0.03604235969,
	0.07718131046, 0.21887874996, 0.53300576800, 0.56895657216,
	0.13370159001, 0.41126743370, 0.06445509201, 0.30460524308,
	0.26871547594, 0.50877407147, 0.82064271981, 0.25168614765,
	0.28443250832, 0.12173833430, 0.91653187658, 0.98794518876,
	0.34464552751, 0.29474776452, 0.47093111972, 0.70696127201,
	0.06732383190, 0.51722769860, 0.81112094485, 0.52510147404,
	0.50956755272, 0.10364085818, 0.55439924314, 0.82921842097,
	0.45130771813, 0.47392193365, 0.72310556352, 0.94668416395,
	0.88436536760, 0.36527603992, 0.54884487442, 0.91888180181,
	0.11243018891, 0.50740073855, 0.12253181555, 0.65971861934,
	0.36918240913, 0.79128391369, 0.54646443068, 0.27979369488,
	0.35966063417, 0.46385082553, 0.74434644612, 0.85811944945,
	0.63203833125, 0.37903988769, 0.58592486343, 0.44209112827,
	0.18570513016, 0.83666493728, 0.48258919034, 0.55760368664,
	0.22177800836, 0.04062013611, 0.10925626392, 0.00970488601,
	0.44273201697, 0.70760216071, 0.41029084140, 0.25595873898,
	0.80663472396, 0.85677663503, 0.54405346843, 0.13306070132,
	0.14322336497, 0.31336405530, 0.92382580035, 0.21826837977,
	0.53724784082, 0.95556505020, 0.01028473769, 0.60686056093,
	0.88570818201, 0.23706778161, 0.01239051485, 0.38688314463,
	0.53520310068, 0.01654103214, 0.15259254738, 0.59245582446,
	0.40556047243, 0.22446363720, 0.65272988067, 0.91671498764,
	0.50611896115, 0.39957884457, 0.09707937864, 0.96581926939,
	0.25077059236, 0.62541581469, 0.32840968047, 0.33530686361,
	0.47807245094, 0.89867854854, 0.53352458266, 0.28995635853,
	0.58275093844, 0.46034119694, 0.67458113346, 0.32816553240,
	0.66472365490, 0.12097537156, 0.92229987487, 0.41276284066,
	0.10202337718, 0.11157567064, 0.24738303781, 0.14868617817,
	0.45374919889, 0.49073763237, 0.82644123661, 0.68950468459,
	0.09921567431, 0.63991210669, 0.41850032044, 0.65343180639,
	0.52003540147, 0.87795648061, 0.36863307596, 0.83059175390,
	0.63060396130, 0.72569963683, 0.39469588305, 0.96227912229,
	0.06393627735, 0.85256508072, 0.81340983306, 0.37412640767,
	0.81896420179, 0.23560289315, 0.36237678152, 0.74358348338,
	0.17471846675, 0.56321909238, 0.27393414106, 0.78371532334,
	0.27942747276, 0.41380046999, 0.75710318308, 0.12424085208,
	0.31861323893, 0.24124881741, 0.54420606098, 0.17096469008,
	0.67024750511, 0.69841608936, 0.65804010132, 0.22507400739,
	0.95223853267, 0.32950834681, 0.05603198340, 0.64333017975,
	0.36854152043, 0.16296884060, 0.30762657552, 0.32068849757,
	0.26081118198, 0.23621326334, 0.96111941893, 0.27643665883,
	0.27100436415, 0.58281197546, 0.75603503525, 0.62483596301,
	0.02157658620, 0.71413312174, 0.33790093692, 0.71214941862,
	0.71987060152, 0.12637714774, 0.63069551683, 0.70055238502,
	0.59193700980, 0.86413159581, 0.52394177068, 0.38090151677,
	0.55348368786, 0.67177343059, 0.03079317606, 0.49516281625,
	0.02264473403, 0.76427503281, 0.57258827479, 0.34958952605,
	0.28223517563, 0.74806970428, 0.54695272683, 0.00064088870,
	0.29645680105, 0.72380748924, 0.83523056734, 0.79403057955,
	0.38938566240, 0.10177922910, 0.40559099094, 0.53865169225,
	0.69093905454, 0.16183965575, 0.66820276498, 0.09671315653,
	0.75658436842, 0.88824121830, 0.73851741081, 0.67076631977,
	0.26712851344, 0.11145359661, 0.26493118076, 0.56950590533,
	0.21491134373, 0.48789941099, 0.97747734001, 0.73345133824,
	0.48261970885, 0.34852137822, 0.61290322581, 0.80120242927,
	0.22061830500, 0.94799645985, 0.37366863002, 0.99850459304,
	0.92989898373, 0.14554277169, 0.57570116276, 0.63026825770,
	0.65575121311, 0.04864650410, 0.64232306894, 0.45793023469,
	0.19959105197, 0.65074617756, 0.09997863704, 0.74239326151,
	0.75411236915, 0.99453718680, 0.73213904233, 0.23435163427,
	0.86886196478, 0.89178136540, 0.54765465255, 0.89330729087,
	0.15192114017, 0.73107089450, 0.71990112003, 0.61595507675,
	0.61449018830, 0.40473647267, 0.15277565844, 0.75621814631,
	0.06372264779, 0.88604388562, 0.82998138371, 0.71678823206,
	0.45136875515, 0.48808252205, 0.60362559893, 0.39475692007,
	0.58305612354, 0.33573412275, 0.57878353221, 0.76180303354,
	0.09564500870, 0.08145390179, 0.74974822230, 0.36033204138,
	0.10818811609, 0.39667958617, 0.04446546831, 0.99707022309,
	0.45301675466, 0.26972258675, 0.43391216773, 0.41496017335,
	0.89309366131, 0.64174321726, 0.26132999664, 0.22498245186,
	0.23206274606, 0.41169469283, 0.44932401502, 0.87896359142,
	0.72304452651, 0.34171575060, 0.73937192907, 0.96441541795,
	0.68248542741, 0.89422284616, 0.62144840846, 0.60338145085,
	0.71550645466, 0.53166295358, 0.10879848628, 0.23563341166,
	0.45844904935, 0.51091036714, 0.01510666219, 0.56794946135,
	0.24582659383, 0.70238349559, 0.88964506973, 0.97085482345,
	0.60676900540, 0.46629230628, 0.26667073580, 0.10354930265,
	0.99816888943, 0.11664174322, 0.38618121891, 0.04907376324,
	0.68025757622, 0.00439466536, 0.04083376568, 0.73992126225,
	0.14108706931, 0.75087740715, 0.70662556841, 0.95641956847,
	0.97601245155, 0.66298409986, 0.61275063326, 0.67827387310,
	0.62849818415, 0.86132389294, 0.90740684225, 0.07110812708,
	0.89861751152, 0.15219580676, 0.57515182958, 0.88900418104,
	0.62529374065, 0.17548142949, 0.26294747765, 0.57390057070,
	0.44853053377, 0.85863826411, 0.67027802362, 0.83431501205,
	0.05822931608, 0.70876186407, 0.66011535997, 0.98297067171,
	0.83535264138, 0.97415082247, 0.74691000092, 0.81597338786,
	0.29178746910, 0.20551164281, 0.11764885403, 0.31412701804,
	0.21591845454, 0.24976348155, 0.84249397259, 0.94778283029,
};

static int rp = 0, erp = 0;;

static unsigned int randinx()
{
    unsigned int result;

    result = ( unsigned int ) ( randtable[ rp ] * 1664525L ) + 1013904223L;
    rp ++;
    if ( rp == erp )
    {
        rp = 0;
    }
    return result;

}

static void normalize2( float v[ 2 ] )
{
    float s;

    s = sqrt( v[ 0 ] * v[ 0 ] + v[ 1 ] * v[ 1 ] );
    v[ 0 ] = v[ 0 ] / s;
    v[ 1 ] = v[ 1 ] / s;
}

static void normalize3( float v[ 3 ] )
{
    float s;

    s = sqrt( v[ 0 ] * v[ 0 ] + v[ 1 ] * v[ 1 ] + v[ 2 ] * v[ 2 ] );
    v[ 0 ] = v[ 0 ] / s;
    v[ 1 ] = v[ 1 ] / s;
    v[ 2 ] = v[ 2 ] / s;
}

class CqRandom
{
public:
    CqRandom()
    {
        Reseed( rand() );
    }
    CqRandom( unsigned int Seed )
    {
        Reseed( Seed );
    }

    /** Get a random integer in the range (0 <= value < 2^32).
     */
    unsigned int RandomInt()
    {
        if ( --m_left == 0 ) NextState();

        unsigned long y = *m_next++;
        y ^= ( y >> 11 );
        y ^= ( y << 7 ) & 0x9d2c5680UL;
        y ^= ( y << 15 ) & 0xefc60000UL;
        y ^= ( y >> 18 );
        return y;
    }

    /** Get a random integer in the specified range (0 <= value < Range).
     * \param Range Integer max value.
     */
    unsigned int RandomInt( unsigned int Range )
    {
        double n = RandomFloat( Range );
        return ( unsigned int ) n;
    }

    /** Get a random float (0.0 <= value < 1.0).
     */
    float	RandomFloat()
    {
        return static_cast<float>( ( double ) RandomInt() * ( 1.0 / 4294967296.0 ) );
    }

    /** Get a random float in the specified range (0 <= value < Range).
     * \param Range The max value for the range.
     */
    float	RandomFloat( float Range )
    {
        return ( RandomFloat() * Range );
    }

    /** Apply a new seed value to the random number generator.
     */
    void	Reseed( unsigned int Seed )
    {
        m_state[ 0 ] = Seed & 0xffffffffUL;
        for ( int j = 1; j < N; j++ )
        {
            m_state[ j ] = ( 1812433253UL * ( m_state[ j - 1 ] ^ ( m_state[ j - 1 ] >> 30 ) ) + j );
            m_state[ j ] &= 0xffffffffUL;  /* for >32 bit machines */
        }
        m_left = 1;
        m_initf = 1;
    }

protected:
    enum {
        N = 624,
        M = 397
    };

    unsigned long m_state[ N ];
    int m_left;
    int m_initf;
    unsigned long* m_next;

#define MT_MATRIX_A	0x9908b0dfUL
#define MT_UMASK	0x80000000UL
#define MT_LMASK	0x7fffffffUL
#define MT_TWIST(u,v)	((((u) & MT_UMASK | (v) & MT_LMASK) >> 1) ^ ((v)&1UL ? MT_MATRIX_A : 0UL))

    void	NextState()
    {
        m_left = N;
        m_next = m_state;

        unsigned long* p = m_state;

        int j;
        for ( j = N - M + 1; --j; p++ )
        {
            *p = p[ M ] ^ MT_TWIST( p[ 0 ], p[ 1 ] );
        }
        for ( j = M; --j; p++ )
        {
            *p = p[ M - N ] ^ MT_TWIST( p[ 0 ], p[ 1 ] );
        }
        *p = p[ M - N ] & MT_TWIST( p[ 0 ], m_state[ 0 ] );
    }
};

class CqNoise
{
public:
	CqNoise()
	{
		init( 665 );
	}

	static	float	FGNoise1( float x )
	{
		int bx0, bx1;
		float rx0, rx1, sx, t, u, v, vec[ 1 ];

		vec[ 0 ] = x;

		setup( 0, bx0, bx1, rx0, rx1 );
		sx = s_curve( rx0 );

		u = rx0 * m_g1[ m_p[ bx0 ] ];
		v = rx1 * m_g1[ m_p[ bx1 ] ];

		return ( LERP( sx, u, v ) );
	}
	
	static	float	FGNoise2( float x, float y )
	{
		int bx0, bx1, by0, by1, b00, b10, b01, b11;
		float rx0, rx1, ry0, ry1, *q, sx, sy, a, b, t, u, v, vec[ 2 ];
		register int i, j;

		vec[ 0 ] = x;
		vec[ 1 ] = y;

		setup( 0, bx0, bx1, rx0, rx1 );
		setup( 1, by0, by1, ry0, ry1 );

		i = m_p[ bx0 ];
		j = m_p[ bx1 ];

		b00 = m_p[ i + by0 ];
		b10 = m_p[ j + by0 ];
		b01 = m_p[ i + by1 ];
		b11 = m_p[ j + by1 ];

		sx = s_curve( rx0 );
		sy = s_curve( ry0 );

#define at2(rx,ry) ( rx * q[0] + ry * q[1] )

		q = m_g2[ b00 ]; u = at2( rx0, ry0 );
		q = m_g2[ b10 ]; v = at2( rx1, ry0 );
		a = LERP( sx, u, v );

		q = m_g2[ b01 ]; u = at2( rx0, ry1 );
		q = m_g2[ b11 ]; v = at2( rx1, ry1 );
		b = LERP( sx, u, v );

		return ( LERP( sy, a, b ) );
	}
	
	static	float	FGNoise3( float x, float y, float z )
	{
		int bx0, bx1, by0, by1, bz0, bz1, b00, b10, b01, b11;
		float rx0, rx1, ry0, ry1, rz0, rz1, *q, sy, sz, a, b, c, d, t, u, v, vec[ 3 ];
		register int i, j;

		vec[ 0 ] = x;
		vec[ 1 ] = y;
		vec[ 2 ] = z;

		setup( 0, bx0, bx1, rx0, rx1 );
		setup( 1, by0, by1, ry0, ry1 );
		setup( 2, bz0, bz1, rz0, rz1 );

		i = m_p[ bx0 ];
		j = m_p[ bx1 ];

		b00 = m_p[ i + by0 ];
		b10 = m_p[ j + by0 ];
		b01 = m_p[ i + by1 ];
		b11 = m_p[ j + by1 ];

		t = s_curve( rx0 );
		sy = s_curve( ry0 );
		sz = s_curve( rz0 );

#define at3(rx,ry,rz) ( rx * q[0] + ry * q[1] + rz * q[2] )

		q = m_g3[ b00 + bz0 ]; u = at3( rx0, ry0, rz0 );
		q = m_g3[ b10 + bz0 ]; v = at3( rx1, ry0, rz0 );
		a = LERP( t, u, v );

		q = m_g3[ b01 + bz0 ]; u = at3( rx0, ry1, rz0 );
		q = m_g3[ b11 + bz0 ]; v = at3( rx1, ry1, rz0 );
		b = LERP( t, u, v );

		c = LERP( sy, a, b );

		q = m_g3[ b00 + bz1 ]; u = at3( rx0, ry0, rz1 );
		q = m_g3[ b10 + bz1 ]; v = at3( rx1, ry0, rz1 );
		a = LERP( t, u, v );

		q = m_g3[ b01 + bz1 ]; u = at3( rx0, ry1, rz1 );
		q = m_g3[ b11 + bz1 ]; v = at3( rx1, ry1, rz1 );
		b = LERP( t, u, v );

		d = LERP( sy, a, b );

		return ( LERP( sz, c, d ) );
	}

	static	void	init( int seed )
	{
		int i, j, k, result;
		CqRandom m_random;


		if ( erp == 0 )
		{
				m_random.Reseed( seed );
				result = sizeof( randtable ) / sizeof( float );
				erp = result - 1;

				for ( i = 0; i < NOISE_B; i++ )
				{
						m_p[ i ] = i;
						m_g1[ i ] = static_cast<float>( ( static_cast<int>
																								( randinx() ) % ( NOISE_B + NOISE_B ) ) - NOISE_B ) / NOISE_B;
						for ( j = 0; j < 2; j++ )
								m_g2[ i ][ j ] = static_cast<float>( (
																													static_cast<int>( randinx() ) % ( NOISE_B + NOISE_B ) ) - NOISE_B ) / NOISE_B;
						normalize2( m_g2[ i ] );

						for ( j = 0; j < 3; j++ )
								m_g3[ i ][ j ] = static_cast<float>( (
																													static_cast<int>( randinx() ) % ( NOISE_B + NOISE_B ) ) - NOISE_B ) / NOISE_B;
						normalize3( m_g3[ i ] );
				}

				while ( --i )
				{
						k = m_p[ i ];
						m_p[ i ] = m_p[ j = static_cast<int>( randinx() ) % NOISE_B ];
						m_p[ j ] = k;
				}

				for ( i = 0; i < NOISE_B + 2; i++ )
				{
						m_p[ NOISE_B + i ] = m_p[ i ];
						m_g1[ NOISE_B + i ] = m_g1[ i ];
						for ( j = 0; j < 2; j++ )
								m_g2[ NOISE_B + i ][ j ] = m_g2[ i ][ j ];
						for ( j = 0; j < 3; j++ )
								m_g3[ NOISE_B + i ][ j ] = m_g3[ i ][ j ];
				}

		}
	}
	
	static	float	glattice( int ix, int iy, int iz, float fx, float fy, float fz );

private:
    static	int	m_p[ NOISE_B + NOISE_B + 2 ];
    static	float	m_g3[ NOISE_B + NOISE_B + 2 ][ 3 ];
    static	float	m_g2[ NOISE_B + NOISE_B + 2 ][ 2 ];
    static	float	m_g1[ NOISE_B + NOISE_B + 2 ];
};

int CqNoise::m_p[ NOISE_B + NOISE_B + 2 ];
float CqNoise::m_g3[ NOISE_B + NOISE_B + 2 ][ 3 ];
float CqNoise::m_g2[ NOISE_B + NOISE_B + 2 ][ 2 ];
float CqNoise::m_g1[ NOISE_B + NOISE_B + 2 ];

CqNoise noise_data;

} // namespace detail

const double noise(const double U)
{
	return 0.5 * (1.0 + detail::CqNoise::FGNoise1(U));
}

const double noise(const double U, const double V)
{
	return 0.5 * (1.0 + detail::CqNoise::FGNoise2(U, V));
}

const double noise(const vector3& Pt)
{
	return 0.5 * (1.0 + detail::CqNoise::FGNoise3(Pt[0], Pt[1], Pt[2]));
}

} // namespace k3d


