// K-3D
// Copyright (c) 1995-2005, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\author Timothy M. Shead (tshead@k-3d.com)
*/

#include "idag.h"
#include "idocument.h"
#include "idocument_read_format.h"
#include "idocument_write_format.h"
#include "imaterial.h"
#include "inode.h"
#include "inode_collection.h"
#include "ipersistent.h"
#include "iplugin_factory.h"
#include "iproperty.h"
#include "iproperty_collection.h"
#include "irenderman.h"
#include "mesh.h"
#include "mesh_selection.h"
#include "property.h"
#include "result.h"
#include "serialization.h"
#include "string_cast.h"
#include "xml.h"

using namespace k3d::xml;

#include <boost/lexical_cast.hpp>

namespace k3d
{

namespace detail
{

class save_dependencies
{
public:
	typedef std::map<iproperty*, inode*> map_t;

	save_dependencies(map_t& Map, element& Element, const ipersistent::save_context& Context) :
		m_map(Map),
		m_element(Element),
		m_context(Context)
	{
	}

	void operator()(const idag::dependency_t& Dependency)
	{
		// Sanity checks ...
		iproperty* const from_property = Dependency.second;
		if(!from_property)
			return;
		inode* const from_node = m_map[from_property];
		return_if_fail(from_node);

		iproperty* const to_property = Dependency.first;
		return_if_fail(to_property);
		inode* const to_node = m_map[to_property];
		return_if_fail(to_node);

		m_element.append(
			element("dependency",
				attribute("from_node", m_context.lookup.lookup_id(from_node)),
				attribute("from_property", from_property->property_name()),
				attribute("to_node", m_context.lookup.lookup_id(to_node)),
				attribute("to_property", to_property->property_name())));
	}

private:
	map_t& m_map;
	element& m_element;
	const ipersistent::save_context& m_context;
};

class load_dependencies
{
public:
	load_dependencies(idag::dependencies_t& Dependencies, const ipersistent::load_context& Context) :
		m_dependencies(Dependencies),
		m_context(Context)
	{
	}

	void operator()(const element& Dependency)
	{
		if("dependency" != Dependency.name)
			return;

		ipersistent_lookup::id_type from_node_id = attribute_value<ipersistent_lookup::id_type>(Dependency, "from_node", 0);
		if(!from_node_id)
			from_node_id = attribute_value<ipersistent_lookup::id_type>(Dependency, "from_object", 0);
		return_if_fail(from_node_id);

		inode* const from_node = dynamic_cast<inode*>(m_context.lookup.lookup_object(from_node_id));
		if(!from_node)
		{
			log() << error << "Missing dependency source node [" << from_node_id << "]" << std::endl;
			return;
		}

		const std::string from_property_name = attribute_text(Dependency, "from_property");
		return_if_fail(from_property_name.size());
		iproperty* const from_property = get_property(*from_node, from_property_name);
		if(!from_property)
		{
			log() << error << "Missing dependency source property [" << from_node->name() << "." << from_property_name << "]" << std::endl;
			return;
		}

		ipersistent_lookup::id_type to_node_id = attribute_value<ipersistent_lookup::id_type>(Dependency, "to_node", 0);
		if(!to_node_id)
			to_node_id = attribute_value<ipersistent_lookup::id_type>(Dependency, "to_object", 0);
		return_if_fail(to_node_id);

		inode* const to_node = dynamic_cast<inode*>(m_context.lookup.lookup_object(to_node_id));
		if(!to_node)
		{
			log() << error << "Missing dependency target node [" << to_node_id << "]" << std::endl;
			return;
		}

		const std::string to_property_name = attribute_text(Dependency, "to_property");
		return_if_fail(to_property_name.size());
		iproperty* const to_property = get_property(*to_node, to_property_name);
		if(!to_property)
		{
			log() << error << "Missing dependency target property [" << to_node->name() << "." << to_property_name << "]" << std::endl;
			return;
		}

		m_dependencies[to_property] = from_property;
	}

private:
	idag::dependencies_t& m_dependencies;
	const ipersistent::load_context& m_context;
};

template<typename data_t, typename iterator_t>
bool save_parameter(element& XMLParameters, iterator_t Parameter, const std::string& Type)
{
	if(typeid(data_t) != Parameter->second.type())
		return false;


	XMLParameters.append(element("parameter", attribute("name", Parameter->first), attribute("type", Type), attribute("value", boost::any_cast<data_t>(Parameter->second))));

	return true;
}

void save_parameters(element& Element, const parameters_t& Parameters)
{
	for(parameters_t::const_iterator parameter = Parameters.begin(); parameter != Parameters.end(); ++parameter)
		{
			if(save_parameter<ri::integer>(Element, parameter, "integer")) continue;
			if(save_parameter<ri::real>(Element, parameter, "real")) continue;
			if(save_parameter<ri::string>(Element, parameter, "string")) continue;
			if(save_parameter<ri::point>(Element, parameter, "point")) continue;
			if(save_parameter<ri::vector>(Element, parameter, "vector")) continue;
			if(save_parameter<ri::normal>(Element, parameter, "normal")) continue;
			if(save_parameter<ri::color>(Element, parameter, "color")) continue;
			if(save_parameter<ri::hpoint>(Element, parameter, "hpoint")) continue;
			if(save_parameter<bool>(Element, parameter, "bool")) continue;
	//		if(save_parameter<ri::matrix>(Element, parameter, "matrix")) continue;

			log() << error << "Unknown parameter type - parameter [" << parameter->first << "] will not be serialized" << std::endl;
		}
}

void save_parameters(element& Element, const parameters_t& Parameters, const ri::storage_class_t StorageClass)
{
	if(Parameters.empty())
		return;

	element& xml_parameters = Element.append(element("parameters", attribute("storageclass", StorageClass)));
	save_parameters(xml_parameters, Parameters);
}

template<typename ContainerT>
void save_varying_parameters(element& Element, const ContainerT& Parameters)
{
	if(Parameters.empty())
		return;

	element& xml_parameters = Element.append(element("parameters", attribute("storageclass", "varying")));
	for(typename ContainerT::const_iterator parameters = Parameters.begin(); parameters != Parameters.end(); ++parameters)
		{
			element& xml_parameters2 = xml_parameters.append(element("parameters"));
			save_parameters(xml_parameters2, *parameters);
		}
}

void save_tags(element& Element, const parameters_t& Parameters)
{
	if(Parameters.empty())
		return;

	element& xml_parameters = Element.append(element("parameters", attribute("storageclass", "tag")));
	save_parameters(xml_parameters, Parameters);
}

template<typename data_t>
bool load_parameter(const std::string& XMLType, const std::string& Name, const std::string& Type, const std::string& Value, k3d::parameters_t& Parameters)
{
	if(XMLType != Type)
		return false;

	Parameters[Name] = k3d::from_string<data_t>(Value, data_t());
	return true;
}

void load_parameters(const element& Element, k3d::parameters_t& Parameters)
{
	for(element::elements_t::const_iterator xml_parameter = Element.children.begin(); xml_parameter != Element.children.end(); ++xml_parameter)
		{
			if(xml_parameter->name != "parameter")
				continue;

			const std::string name = attribute_text(*xml_parameter, "name");
			if(name.empty())
				{
					log() << error << k3d_file_reference << " unnamed parameter will not be loaded" << std::endl;
					continue;
				}

			const std::string type = attribute_text(*xml_parameter, "type");
			if(type.empty())
				{
					log() << error << k3d_file_reference << " parameter [" << name << "] with unknown type will not be loaded" << std::endl;
					continue;
				}

			const std::string value = attribute_text(*xml_parameter, "value");

			if(load_parameter<k3d::ri::integer>("integer", name, type, value, Parameters)) continue;
			if(load_parameter<k3d::ri::real>("real", name, type, value, Parameters)) continue;
			if(load_parameter<k3d::ri::string>("string", name, type, value, Parameters)) continue;
			if(load_parameter<k3d::ri::point>("point", name, type, value, Parameters)) continue;
			if(load_parameter<k3d::ri::vector>("vector", name, type, value, Parameters)) continue;
			if(load_parameter<k3d::ri::normal>("normal", name, type, value, Parameters)) continue;
			if(load_parameter<k3d::ri::color>("color", name, type, value, Parameters)) continue;
			if(load_parameter<k3d::ri::hpoint>("hpoint", name, type, value, Parameters)) continue;
			if(load_parameter<bool>("bool", name, type, value, Parameters)) continue;
	//		if(load_parameter<k3d::ri::matrix>("matrix", name, type, value, Parameters)) continue;

			log() << error << k3d_file_reference << " parameter [" << name << "] with unknown type [" << type << "] will not be loaded" << std::endl;
		}
}

void load_parameters(const element& Element, const std::string& StorageClass, k3d::parameters_t& Parameters)
{
	for(element::elements_t::const_iterator xml_parameters = Element.children.begin(); xml_parameters != Element.children.end(); ++xml_parameters)
		{
			if(xml_parameters->name != "parameters")
				continue;

			std::string storage_class = attribute_text(*xml_parameters, "storageclass");
			if(storage_class.empty())
				storage_class = attribute_text(*xml_parameters, "type");
			if(storage_class != StorageClass)
				continue;

			load_parameters(*xml_parameters, Parameters);
		}
}

void load_parameters(const element& Element, const k3d::ri::storage_class_t StorageClass, k3d::parameters_t& Parameters)
{
	load_parameters(Element, k3d::string_cast(StorageClass), Parameters);
}

void load_varying_parameters(const element& Element, boost::array<k3d::parameters_t, 4>& Parameters)
{
	for(element::elements_t::const_iterator xml_parameters = Element.children.begin(); xml_parameters != Element.children.end(); ++xml_parameters)
		{
			if(xml_parameters->name != "parameters")
				continue;

			std::string storage_class = attribute_text(*xml_parameters, "storageclass");
			if(storage_class.empty())
				storage_class = attribute_text(*xml_parameters, "type");
			std::string keyword("varying");
			if(storage_class != k3d::string_cast(keyword))
				continue;

			if(xml_parameters->children.size() != 4)
				{
					log() << error << k3d_file_reference << " varying parameters with incorrect child count will not be loaded" << std::endl;
					continue;
				}

			load_parameters(xml_parameters->children[0], Parameters[0]);
			load_parameters(xml_parameters->children[1], Parameters[1]);
			load_parameters(xml_parameters->children[2], Parameters[2]);
			load_parameters(xml_parameters->children[3], Parameters[3]);
		}
}

template<typename ContainerT>
void load_varying_parameters(const element& Element, ContainerT& Parameters, const unsigned long VaryingCount)
{
	for(element::elements_t::const_iterator xml_parameters = Element.children.begin(); xml_parameters != Element.children.end(); ++xml_parameters)
		{
			if(xml_parameters->name != "parameters")
				continue;

			std::string storage_class = attribute_text(*xml_parameters, "storageclass");
			if(storage_class.empty())
				storage_class = attribute_text(*xml_parameters, "type");
			std::string keyword("varying");
			if(storage_class != k3d::string_cast(keyword))
				continue;

			if(xml_parameters->children.size() != VaryingCount)
				{
					log() << error << k3d_file_reference << " varying parameters require " << VaryingCount << " values, found " << xml_parameters->children.size() << ", will not be loaded" << std::endl;
					continue;
				}

			for(element::elements_t::const_iterator xml_parameters2 = xml_parameters->children.begin(); xml_parameters2 != xml_parameters->children.end(); ++xml_parameters2)
				{
					Parameters.push_back(k3d::parameters_t());
					load_parameters(*xml_parameters2, Parameters.back());
				}
		}
}

void load_tags(const element& Element, k3d::parameters_t& Parameters)
{
	load_parameters(Element, "tag", Parameters);
}

void save_selection(element& Element, const mesh_selection::records_t& Records, const std::string& ElementName)
{
	if(Records.empty())
		return;

	element& xml_records = Element.append(element(ElementName));

	for(mesh_selection::records_t::const_iterator record = Records.begin(); record != Records.end(); ++record)
	{
		xml_records.append(
			element("selection",
				attribute("begin", record->first),
				attribute("end", record->second.end),
				attribute("weight", record->second.weight)));
	}
}

void load_selection(const element& Element, mesh_selection::records_t& Records)
{
	for(element::elements_t::const_iterator xml_selection = Element.children.begin(); xml_selection != Element.children.end(); ++xml_selection)
	{
		if(xml_selection->name != "selection")
			continue;

		const size_t begin = attribute_value<size_t>(*xml_selection, "begin", 0);
		const size_t end = attribute_value<size_t>(*xml_selection, "end", 0);
		const double weight = attribute_value<double>(*xml_selection, "weight", 0.0);

		Records.insert(std::make_pair(begin, mesh_selection::record(end, weight)));
	}
}

} // namespace detail

/////////////////////////////////////////////////////////////////////////////
// save_dag

void save_dag(idocument& Document, element& XML, const ipersistent::save_context& Context)
{
	// Create a mapping of properties to objects ...
	detail::save_dependencies::map_t object_map;
	const inode_collection::nodes_t& objects = Document.nodes().collection();
	for(inode_collection::nodes_t::const_iterator object = objects.begin(); object != objects.end(); ++object)
		{
			iproperty_collection* const property_collection = dynamic_cast<iproperty_collection*>(*object);
			if(!property_collection)
				continue;

			const iproperty_collection::properties_t properties(property_collection->properties());
			for(iproperty_collection::properties_t::const_iterator property = properties.begin(); property != properties.end(); ++property)
				object_map[*property] = *object;
		}

	// Save all dependencies
	element& xml_dependencies = XML.append(element("dependencies"));
	std::for_each(Document.dag().dependencies().begin(), Document.dag().dependencies().end(), detail::save_dependencies(object_map, xml_dependencies, Context));
}

/////////////////////////////////////////////////////////////////////////////
// load_dag

void load_dag(idocument& Document, element& XML, const ipersistent::load_context& Context)
{
	// If we don't have any DAG information, we're done ...
	element* xml_dependencies = find_element(XML, "dependencies");

	if(!xml_dependencies)
		return;

	// Load data and update the DAG ...
	idag::dependencies_t dependencies;
	std::for_each(xml_dependencies->children.begin(), xml_dependencies->children.end(), detail::load_dependencies(dependencies, Context));
	Document.dag().set_dependencies(dependencies);
}

/////////////////////////////////////////////////////////////////////////////
// save_mesh

void save_mesh(const mesh& Mesh, element& XML, const ipersistent::save_context& Context)
{
	// Create a one-based index of points
	std::map<point*, unsigned long> point_map;
	point_map[0] = 0;
	for(mesh::points_t::const_iterator point = Mesh.points.begin(); point != Mesh.points.end(); ++point)
		point_map.insert(std::make_pair(*point, point_map.size()));

	// Save points ...
	if(Mesh.points.size())
		{
			element& xml_points = XML.append(element("points"));
			for(mesh::points_t::const_iterator point = Mesh.points.begin(); point != Mesh.points.end(); ++point)
				{
					element& xml_point = xml_points.append(element("point", attribute("position", (**point).position)));
					detail::save_parameters(xml_point, (*point)->vertex_data, ri::VERTEX);
					detail::save_tags(xml_point, (*point)->tags);
				}
		}

	// Save point groups ...
	if(Mesh.point_groups.size())
		{
			element& xml_point_groups = XML.append(element("pointgroups"));
			for(mesh::point_groups_t::const_iterator point_group = Mesh.point_groups.begin(); point_group != Mesh.point_groups.end(); ++point_group)
				{
					element& xml_point_group = xml_point_groups.append(element("group", attribute("material", Context.lookup.lookup_id((*point_group)->material))));

					std::ostringstream points_buffer;
					for(point_group::points_t::const_iterator point = (*point_group)->points.begin(); point != (*point_group)->points.end(); ++point)
						points_buffer << point_map[*point] << " ";
					xml_point_group.append(element("points", points_buffer.str()));

					detail::save_parameters(xml_point_group, (*point_group)->constant_data, ri::CONSTANT);
				}
		}

	// Save polyhedra ...
	if(Mesh.polyhedra.size())
		{
			// For each polyhedron ...
			element& xml_polyhedra = XML.append(element("polyhedra"));
			for(mesh::polyhedra_t::const_iterator polyhedron = Mesh.polyhedra.begin(); polyhedron != Mesh.polyhedra.end(); ++polyhedron)
				{
					element& xml_polyhedron = xml_polyhedra.append(element("polyhedron", attribute("type", (*polyhedron)->type)));

					detail::save_tags(xml_polyhedron, (*polyhedron)->tags);

					// Create a one-based index of edges ...
					std::map<split_edge*, unsigned long> edge_map;
					edge_map[0] = 0;
					for(polyhedron::faces_t::const_iterator face = (*polyhedron)->faces.begin(); face != (*polyhedron)->faces.end(); ++face)
						{
							split_edge* edge = (*face)->first_edge;
							do
								{
									edge_map.insert(std::make_pair(edge, edge_map.size()));
									edge = edge->face_clockwise;
								}
							while(edge != (*face)->first_edge);
						}

					// Save edges ...
					element& xml_edges = xml_polyhedron.append(element("edges"));
					for(polyhedron::faces_t::const_iterator face = (*polyhedron)->faces.begin(); face != (*polyhedron)->faces.end(); ++face)
						{
							split_edge* edge = (*face)->first_edge;
							do
								{
									element& xml_edge = xml_edges.append(
										element("edge",
											attribute("vertex", point_map[edge->vertex]),
											attribute("faceclockwise", edge_map[edge->face_clockwise]),
											attribute("companion", edge_map[edge->companion])));

									detail::save_parameters(xml_edge, edge->facevarying_data, ri::FACEVARYING);
									detail::save_tags(xml_edge, edge->tags);

									edge = edge->face_clockwise;
								}
							while(edge != (*face)->first_edge);
						}

					// Save faces ...
					if((*polyhedron)->faces.size())
						{
							element& xml_faces = xml_polyhedron.append(element("faces"));
							for(polyhedron::faces_t::const_iterator face = (*polyhedron)->faces.begin(); face != (*polyhedron)->faces.end(); ++face)
								{
									element& xml_face = xml_faces.append(element("face", attribute("firstedge", edge_map[(*face)->first_edge]), attribute("material", Context.lookup.lookup_id((*face)->material))));
									if((*face)->holes.size())
										{
											element& xml_holes = xml_face.append(element("holes"));
											for(face::holes_t::const_iterator hole = (*face)->holes.begin(); hole != (*face)->holes.end(); ++hole)
												{
													xml_holes.append(element("hole", attribute("firstedge", edge_map[(*hole)])));
												}
										}

									detail::save_parameters(xml_face, (*face)->uniform_data, ri::UNIFORM);
									detail::save_tags(xml_face, (*face)->tags);
								}
						}
				}
		}

	// Save linear curve groups ...
	if(Mesh.linear_curve_groups.size())
		{
			element& xml_linear_curve_groups = XML.append(element("linearcurvegroups"));
			for(mesh::linear_curve_groups_t::const_iterator group = Mesh.linear_curve_groups.begin(); group != Mesh.linear_curve_groups.end(); ++group)
				{
					element& xml_linear_curve_group = xml_linear_curve_groups.append(element("group", attribute("wrap", (*group)->wrap), attribute("material", Context.lookup.lookup_id((*group)->material))));

					element& xml_curves = xml_linear_curve_group.append(element("curves"));
					for(linear_curve_group::curves_t::const_iterator curve = (*group)->curves.begin(); curve != (*group)->curves.end(); ++curve)
						{
							element& xml_curve = xml_curves.append(element("curve"));

							std::ostringstream points_buffer;
							for(linear_curve::control_points_t::const_iterator control_point = (*curve)->control_points.begin(); control_point != (*curve)->control_points.end(); ++control_point)
								points_buffer << point_map[*control_point] << " ";
							xml_curve.append(element("controlpoints", points_buffer.str()));

							detail::save_parameters(xml_curve, (*curve)->uniform_data, ri::UNIFORM);
							detail::save_varying_parameters(xml_curve, (*curve)->varying_data);
						}
					detail::save_parameters(xml_curves, (*group)->constant_data, ri::CONSTANT);
				}
		}

	// Save cubic curve groups ...
	if(Mesh.cubic_curve_groups.size())
		{
			element& xml_cubic_curve_groups = XML.append(element("cubiccurvegroups"));
			for(mesh::cubic_curve_groups_t::const_iterator group = Mesh.cubic_curve_groups.begin(); group != Mesh.cubic_curve_groups.end(); ++group)
				{
					element& xml_cubic_curve_group = xml_cubic_curve_groups.append(element("group", attribute("wrap", (*group)->wrap), attribute("material", Context.lookup.lookup_id((*group)->material))));

					element& xml_curves = xml_cubic_curve_group.append(element("curves"));
					for(cubic_curve_group::curves_t::const_iterator curve = (*group)->curves.begin(); curve != (*group)->curves.end(); ++curve)
						{
							element& xml_curve = xml_curves.append(element("curve"));

							std::ostringstream points_buffer;
							for(cubic_curve::control_points_t::const_iterator control_point = (*curve)->control_points.begin(); control_point != (*curve)->control_points.end(); ++control_point)
								points_buffer << point_map[*control_point] << " ";
							xml_curve.append(element("controlpoints", points_buffer.str()));

							detail::save_parameters(xml_curve, (*curve)->uniform_data, ri::UNIFORM);
							detail::save_varying_parameters(xml_curve, (*curve)->varying_data);
						}
					detail::save_parameters(xml_curves, (*group)->constant_data, ri::CONSTANT);
				}
		}

	// Save NURBS curve groups ...
	if(Mesh.nucurve_groups.size())
		{
			element& xml_nucurve_groups = XML.append(element("nucurvegroups"));
			for(mesh::nucurve_groups_t::const_iterator group = Mesh.nucurve_groups.begin(); group != Mesh.nucurve_groups.end(); ++group)
				{
					element& xml_nucurve_group = xml_nucurve_groups.append(element("group", attribute("material", Context.lookup.lookup_id((*group)->material))));

					element& xml_curves = xml_nucurve_group.append(element("curves"));
					for(nucurve_group::curves_t::const_iterator curve = (*group)->curves.begin(); curve != (*group)->curves.end(); ++curve)
						{
							element& xml_curve = xml_curves.append(element("curve", attribute("order", (*curve)->order)));

							std::ostringstream knots_buffer;
							std::copy((*curve)->knots.begin(), (*curve)->knots.end(), std::ostream_iterator<double>(knots_buffer, " "));
							xml_curve.append(element("knotvector", knots_buffer.str()));

							std::ostringstream points_buffer;
							std::ostringstream weights_buffer;
							for(nucurve::control_points_t::const_iterator control_point = (*curve)->control_points.begin(); control_point != (*curve)->control_points.end(); ++control_point)
								{
									points_buffer << point_map[control_point->position] << " ";
									weights_buffer << control_point->weight << " ";
								}

							xml_curve.append(element("controlpoints", points_buffer.str()));
							xml_curve.append(element("weights", weights_buffer.str()));
						}
				}
		}

	// Save bilinear patches ...
	if(Mesh.bilinear_patches.size())
		{
			element& xml_patches = XML.append(element("bilinearpatches"));
			for(mesh::bilinear_patches_t::const_iterator patch = Mesh.bilinear_patches.begin(); patch != Mesh.bilinear_patches.end(); ++patch)
				{
					element& xml_patch = xml_patches.append(element("patch", attribute("material", Context.lookup.lookup_id((*patch)->material))));

					std::ostringstream points_buffer;
					for(bilinear_patch::control_points_t::const_iterator control_point = (*patch)->control_points.begin(); control_point != (*patch)->control_points.end(); ++control_point)
						points_buffer << point_map[*control_point] << " ";
					xml_patch.append(element("controlpoints", points_buffer.str()));

					detail::save_parameters(xml_patch, (*patch)->uniform_data, ri::UNIFORM);
					detail::save_varying_parameters(xml_patch, (*patch)->varying_data);
				}
		}

	// Save bicubic patches ...
	if(Mesh.bicubic_patches.size())
		{
			element& xml_patches = XML.append(element("bicubicpatches"));
			for(mesh::bicubic_patches_t::const_iterator patch = Mesh.bicubic_patches.begin(); patch != Mesh.bicubic_patches.end(); ++patch)
				{
					element& xml_patch = xml_patches.append(element("patch", attribute("material", Context.lookup.lookup_id((*patch)->material))));

					std::ostringstream points_buffer;
					for(bicubic_patch::control_points_t::const_iterator control_point = (*patch)->control_points.begin(); control_point != (*patch)->control_points.end(); ++control_point)
						points_buffer << point_map[*control_point] << " ";
					xml_patch.append(element("controlpoints", points_buffer.str()));

					detail::save_parameters(xml_patch, (*patch)->uniform_data, ri::UNIFORM);
					detail::save_varying_parameters(xml_patch, (*patch)->varying_data);
				}
		}

	// Save NURBS patches ...
	if(Mesh.nupatches.size())
		{
			element& xml_patches = XML.append(element("nupatches"));
			for(mesh::nupatches_t::const_iterator patch = Mesh.nupatches.begin(); patch != Mesh.nupatches.end(); ++patch)
				{
					element& xml_patch = xml_patches.append(element("patch", attribute("uorder", (*patch)->u_order), attribute("vorder", (*patch)->v_order), attribute("material", Context.lookup.lookup_id((*patch)->material))));

					std::ostringstream u_knots_buffer;
					std::copy((*patch)->u_knots.begin(), (*patch)->u_knots.end(), std::ostream_iterator<double>(u_knots_buffer, " "));
					xml_patch.append(element("uknotvector", u_knots_buffer.str()));

					std::ostringstream v_knots_buffer;
					std::copy((*patch)->v_knots.begin(), (*patch)->v_knots.end(), std::ostream_iterator<double>(v_knots_buffer, " "));
					xml_patch.append(element("vknotvector", v_knots_buffer.str()));

					std::ostringstream points_buffer;
					std::ostringstream weights_buffer;
					for(nupatch::control_points_t::const_iterator control_point = (*patch)->control_points.begin(); control_point != (*patch)->control_points.end(); ++control_point)
						{
							points_buffer << point_map[control_point->position] << " ";
							weights_buffer << control_point->weight << " ";
						}

					xml_patch.append(element("controlpoints", points_buffer.str()));
					xml_patch.append(element("weights", weights_buffer.str()));
				}
		}
}

/////////////////////////////////////////////////////////////////////////////
// load_mesh

void load_mesh(mesh& Mesh, element& XML, const ipersistent::load_context& Context)
{
	// Load points ...
	if(element* const xml_points = find_element(XML, "points"))
		{
			for(element::elements_t::const_iterator xml_point = xml_points->children.begin(); xml_point != xml_points->children.end(); ++xml_point)
				{
					if(xml_point->name != "point")
						continue;

					Mesh.points.push_back(new k3d::point(attribute_value<k3d::vector3>(*xml_point, "position", k3d::vector3(0, 0, 0))));
					detail::load_parameters(*xml_point, k3d::ri::VERTEX, Mesh.points.back()->vertex_data);
					detail::load_tags(*xml_point, Mesh.points.back()->tags);
				}
		}

	// Load point groups ...
	if(element* const xml_point_groups = find_element(XML, "pointgroups"))
		{
			for(element::elements_t::iterator xml_group = xml_point_groups->children.begin(); xml_group != xml_point_groups->children.end(); ++xml_group)
				{
					if(xml_group->name != "group")
						continue;

					k3d::point_group* const group = new k3d::point_group();
					group->material = dynamic_cast<imaterial*>(Context.lookup.lookup_object(attribute_value<ipersistent_lookup::id_type>(*xml_group, "material", 0)));
					Mesh.point_groups.push_back(group);

					element* const xml_points = find_element(*xml_group, "points");
					if(xml_points)
						{
							std::istringstream points_buffer(xml_points->text);
							for(std::istream_iterator<unsigned long> point(points_buffer); point != std::istream_iterator<unsigned long>(); ++point)
								{
									const unsigned long point_index = *point - 1;
									return_if_fail(point_index < Mesh.points.size());
									group->points.push_back(Mesh.points[point_index]);
								}
						}

					detail::load_parameters(*xml_group, k3d::ri::CONSTANT, group->constant_data);
				}
		}

	// Load polyhedra ...
	if(element* const xml_polyhedra = find_element(XML, "polyhedra"))
		{
			for(element::elements_t::iterator xml_polyhedron = xml_polyhedra->children.begin(); xml_polyhedron != xml_polyhedra->children.end(); ++xml_polyhedron)
				{
					if(xml_polyhedron->name != "polyhedron")
						continue;

					k3d::polyhedron* const polyhedron = new k3d::polyhedron();

					polyhedron->type = attribute_value(*xml_polyhedron, "type", k3d::polyhedron::POLYGONS);
					Mesh.polyhedra.push_back(polyhedron);

					detail::load_tags(*xml_polyhedron, polyhedron->tags);

					// Load edges ...
					typedef std::vector<split_edge*> edges_t;
					edges_t edges;
					if(element* const xml_edges = find_element(*xml_polyhedron, "edges"))
						{
							//const unsigned long edge_count = std::count_if(xml_edges->children.begin(), xml_edges->children.end(), same_name("edge"));
							// Above statement often crashes, count eges the old fashioned way
							unsigned long edge_count = 0;
							for(k3d::xml::element::elements_t::const_iterator xe = xml_edges->children.begin(); xe != xml_edges->children.end(); ++xe)
								{
									if((*xe).name == "edge")
										++edge_count;
								}

							edges.resize(edge_count);
							for(edges_t::iterator edge = edges.begin(); edge != edges.end(); ++edge)
								*edge = new k3d::split_edge(0, 0, 0);


							edges_t::iterator edge = edges.begin();
							for(element::elements_t::iterator xml_edge = xml_edges->children.begin(); xml_edge != xml_edges->children.end(); ++xml_edge)
								{
									if(xml_edge->name != "edge")
										continue;

									unsigned long vertex_index = attribute_value<unsigned long>(*xml_edge, "vertex", 0);
									return_if_fail(vertex_index <= Mesh.points.size());

									unsigned long face_clockwise_index = attribute_value<unsigned long>(*xml_edge, "faceclockwise", 0);
									return_if_fail(face_clockwise_index <= edges.size());

									unsigned long companion_index = attribute_value<unsigned long>(*xml_edge, "companion", 0);
									return_if_fail(companion_index <= edges.size());

									if(vertex_index)
										(*edge)->vertex = Mesh.points[vertex_index-1];

									if(face_clockwise_index)
										(*edge)->face_clockwise = edges[face_clockwise_index-1];

									if(companion_index)
										(*edge)->companion = edges[companion_index-1];

									detail::load_parameters(*xml_edge, k3d::ri::FACEVARYING, (*edge)->facevarying_data);
									detail::load_tags(*xml_edge, (*edge)->tags);

									++edge;
								}
						}

					// Load faces ...
					if(element* const xml_faces = find_element(*xml_polyhedron, "faces"))
						{
							for(element::elements_t::iterator xml_face = xml_faces->children.begin(); xml_face != xml_faces->children.end(); ++xml_face)
								{
									if(xml_face->name != "face")
										continue;

									unsigned long first_edge_index = attribute_value<unsigned long>(*xml_face, "firstedge", 0);
									return_if_fail(first_edge_index);
									return_if_fail(first_edge_index <= edges.size());

									k3d::face* const face = new k3d::face(
										first_edge_index ? edges[first_edge_index-1] : 0,
										dynamic_cast<imaterial*>(Context.lookup.lookup_object(attribute_value<ipersistent_lookup::id_type>(*xml_face, "material", 0))));
									polyhedron->faces.push_back(face);

									detail::load_parameters(*xml_face, k3d::ri::UNIFORM, face->uniform_data);
									detail::load_tags(*xml_face, face->tags);

									if(element* const xml_holes = find_element(*xml_face, "holes"))
										{
											for(element::elements_t::iterator xml_hole = xml_holes->children.begin(); xml_hole != xml_holes->children.end(); ++xml_hole)
												{
													if(xml_hole->name != "hole")
														continue;

													unsigned long first_edge_index = attribute_value<unsigned long>(*xml_hole, "firstedge", 0);
													return_if_fail(first_edge_index);
													return_if_fail(first_edge_index <= edges.size());

													face->holes.push_back(first_edge_index ? edges[first_edge_index-1] : 0);
												}
										}
								}
						}
				}
		}

	// Load linear curve groups ...
	if(element* const xml_linear_curve_groups = find_element(XML, "linearcurvegroups"))
		{
			for(element::elements_t::iterator xml_group = xml_linear_curve_groups->children.begin(); xml_group != xml_linear_curve_groups->children.end(); ++xml_group)
				{
					if(xml_group->name != "group")
						continue;

					k3d::linear_curve_group* const group = new k3d::linear_curve_group();
					group->wrap = attribute_value<bool>(*xml_group, "wrap", false);
					group->material = dynamic_cast<imaterial*>(Context.lookup.lookup_object(attribute_value<ipersistent_lookup::id_type>(*xml_group, "material", 0)));
					detail::load_parameters(*xml_group, k3d::ri::CONSTANT, group->constant_data);
					Mesh.linear_curve_groups.push_back(group);

					element* const xml_curves = find_element(*xml_group, "curves");
					if(!xml_curves)
						continue;

					for(element::elements_t::iterator xml_curve = xml_curves->children.begin(); xml_curve != xml_curves->children.end(); ++xml_curve)
						{
							if(xml_curve->name != "curve")
								continue;

							k3d::linear_curve* const curve = new k3d::linear_curve();
							detail::load_parameters(*xml_curve, k3d::ri::UNIFORM, curve->uniform_data);
							group->curves.push_back(curve);

							element* const xml_control_points = find_element(*xml_curve, "controlpoints");
							if(!xml_control_points)
								continue;

							std::istringstream points_buffer(xml_control_points->text);
							for(std::istream_iterator<unsigned long> control_point(points_buffer); control_point != std::istream_iterator<unsigned long>(); ++control_point)
								{
									const unsigned long control_point_index = *control_point - 1;
									return_if_fail(control_point_index < Mesh.points.size());
									curve->control_points.push_back(Mesh.points[control_point_index]);
								}
							detail::load_varying_parameters(*xml_curve, curve->varying_data, k3d::varying_count(*curve, group->wrap));
						}
				}
		}

	// Load cubic curve groups ...
	if(element* const xml_cubic_curve_groups = find_element(XML, "cubiccurvegroups"))
		{
			for(element::elements_t::iterator xml_group = xml_cubic_curve_groups->children.begin(); xml_group != xml_cubic_curve_groups->children.end(); ++xml_group)
				{
					if(xml_group->name != "group")
						continue;

					k3d::cubic_curve_group* const group = new k3d::cubic_curve_group();
					group->wrap = attribute_value<bool>(*xml_group, "wrap", false);
					group->material = dynamic_cast<imaterial*>(Context.lookup.lookup_object(attribute_value<ipersistent_lookup::id_type>(*xml_group, "material", 0)));
					detail::load_parameters(*xml_group, k3d::ri::CONSTANT, group->constant_data);
					Mesh.cubic_curve_groups.push_back(group);

					element* const xml_curves = find_element(*xml_group, "curves");
					if(!xml_curves)
						continue;

					for(element::elements_t::iterator xml_curve = xml_curves->children.begin(); xml_curve != xml_curves->children.end(); ++xml_curve)
						{
							if(xml_curve->name != "curve")
								continue;

							k3d::cubic_curve* const curve = new k3d::cubic_curve();
							detail::load_parameters(*xml_curve, k3d::ri::UNIFORM, curve->uniform_data);
							group->curves.push_back(curve);

							element* const xml_control_points = find_element(*xml_curve, "controlpoints");
							if(!xml_control_points)
								continue;

							std::istringstream points_buffer(xml_control_points->text);
							for(std::istream_iterator<unsigned long> control_point(points_buffer); control_point != std::istream_iterator<unsigned long>(); ++control_point)
								{
									const unsigned long control_point_index = *control_point - 1;
									return_if_fail(control_point_index < Mesh.points.size());
									curve->control_points.push_back(Mesh.points[control_point_index]);
								}

							detail::load_varying_parameters(*xml_curve, curve->varying_data, k3d::varying_count(*curve, group->wrap));
						}
				}
		}

	// Load NURBS curve groups ...
	if(element* const xml_nucurve_groups = find_element(XML, "nucurvegroups"))
		{
			for(element::elements_t::iterator xml_group = xml_nucurve_groups->children.begin(); xml_group != xml_nucurve_groups->children.end(); ++xml_group)
				{
					if(xml_group->name != "group")
						continue;

					k3d::nucurve_group* const group = new k3d::nucurve_group();
					group->material = dynamic_cast<imaterial*>(Context.lookup.lookup_object(attribute_value<ipersistent_lookup::id_type>(*xml_group, "material", 0)));
					Mesh.nucurve_groups.push_back(group);

					element* const xml_curves = find_element(*xml_group, "curves");
					if(!xml_curves)
						continue;

					for(element::elements_t::iterator xml_curve = xml_curves->children.begin(); xml_curve != xml_curves->children.end(); ++xml_curve)
						{
							if(xml_curve->name != "curve")
								continue;

							element* const xml_knot_vector = find_element(*xml_curve, "knotvector");
							element* const xml_control_points = find_element(*xml_curve, "controlpoints");
							element* const xml_weights = find_element(*xml_curve, "weights");

							if(xml_knot_vector && xml_control_points && xml_weights)
								{
									k3d::nucurve* const curve = new k3d::nucurve();
									curve->order = attribute_value<unsigned long>(*xml_curve, "order", 0);
									group->curves.push_back(curve);

									std::istringstream knots_buffer(xml_knot_vector->text);
									std::copy(std::istream_iterator<double>(knots_buffer), std::istream_iterator<double>(), std::back_inserter(curve->knots));

									std::istringstream points_buffer(xml_control_points->text);
									std::istringstream weights_buffer(xml_weights->text);

									std::istream_iterator<unsigned long> control_point(points_buffer);
									std::istream_iterator<double> weight(weights_buffer);
									for(; control_point != std::istream_iterator<unsigned long>() && weight != std::istream_iterator<double>(); ++control_point, ++weight)
										{
											const unsigned long control_point_index = *control_point - 1;
											return_if_fail(control_point_index < Mesh.points.size());
											curve->control_points.push_back(k3d::nucurve::control_point(Mesh.points[control_point_index], *weight));
										}
								}
						}
				}
		}


	// Load bilinear patches ...
	if(element* const xml_bilinear_patches = find_element(XML, "bilinearpatches"))
		{
			for(element::elements_t::iterator xml_patch = xml_bilinear_patches->children.begin(); xml_patch != xml_bilinear_patches->children.end(); ++xml_patch)
				{
					if(xml_patch->name != "patch")
						continue;

					k3d::bilinear_patch* const patch = new k3d::bilinear_patch();
					patch->material = dynamic_cast<imaterial*>(Context.lookup.lookup_object(attribute_value<ipersistent_lookup::id_type>(*xml_patch, "material", 0)));
					Mesh.bilinear_patches.push_back(patch);

					element* const xml_control_points = find_element(*xml_patch, "controlpoints");
					if(!xml_control_points)
						continue;

					unsigned long storage_index = 0;
					std::istringstream points_buffer(xml_control_points->text);
					for(std::istream_iterator<unsigned long> control_point(points_buffer); control_point != std::istream_iterator<unsigned long>(); ++control_point)
						{
							return_if_fail(storage_index < 4);

							const unsigned long control_point_index = *control_point - 1;
							return_if_fail(control_point_index < Mesh.points.size());
							patch->control_points[storage_index++] = Mesh.points[control_point_index];
						}

					detail::load_parameters(*xml_patch, k3d::ri::UNIFORM, patch->uniform_data);
					detail::load_varying_parameters(*xml_patch, patch->varying_data);
				}
		}

	// Load bicubic patches ...
	if(element* const xml_bicubic_patches = find_element(XML, "bicubicpatches"))
		{
			for(element::elements_t::iterator xml_patch = xml_bicubic_patches->children.begin(); xml_patch != xml_bicubic_patches->children.end(); ++xml_patch)
				{
					if(xml_patch->name != "patch")
						continue;

					k3d::bicubic_patch* const patch = new k3d::bicubic_patch();
					patch->material = dynamic_cast<imaterial*>(Context.lookup.lookup_object(attribute_value<ipersistent_lookup::id_type>(*xml_patch, "material", 0)));
					Mesh.bicubic_patches.push_back(patch);

					element* const xml_control_points = find_element(*xml_patch, "controlpoints");
					if(!xml_control_points)
						continue;

					unsigned long storage_index = 0;
					std::istringstream points_buffer(xml_control_points->text);
					for(std::istream_iterator<unsigned long> control_point(points_buffer); control_point != std::istream_iterator<unsigned long>(); ++control_point)
						{
							return_if_fail(storage_index < 16);

							const unsigned long control_point_index = *control_point - 1;
							return_if_fail(control_point_index < Mesh.points.size());
							patch->control_points[storage_index++] = Mesh.points[control_point_index];
						}

					detail::load_parameters(*xml_patch, k3d::ri::UNIFORM, patch->uniform_data);
					detail::load_varying_parameters(*xml_patch, patch->varying_data);
				}
		}

	// Load NURBS patches ...
	if(element* const xml_nupatches = find_element(XML, "nupatches"))
		{
			for(element::elements_t::iterator xml_patch = xml_nupatches->children.begin(); xml_patch != xml_nupatches->children.end(); ++xml_patch)
				{
					if(xml_patch->name != "patch")
						continue;

					element* const xml_u_knot_vector = find_element(*xml_patch, "uknotvector");
					element* const xml_v_knot_vector = find_element(*xml_patch, "vknotvector");
					element* const xml_control_points = find_element(*xml_patch, "controlpoints");
					element* const xml_weights = find_element(*xml_patch, "weights");

					if(xml_u_knot_vector && xml_v_knot_vector && xml_control_points && xml_weights)
						{
							k3d::nupatch* const patch = new k3d::nupatch();
							patch->u_order = attribute_value<unsigned long>(*xml_patch, "uorder", 0);
							patch->v_order = attribute_value<unsigned long>(*xml_patch, "vorder", 0);
							patch->material = dynamic_cast<imaterial*>(Context.lookup.lookup_object(attribute_value<ipersistent_lookup::id_type>(*xml_patch, "material", 0)));
							Mesh.nupatches.push_back(patch);

							std::istringstream u_knots_buffer(xml_u_knot_vector->text);
							std::copy(std::istream_iterator<double>(u_knots_buffer), std::istream_iterator<double>(), std::back_inserter(patch->u_knots));

							std::istringstream v_knots_buffer(xml_v_knot_vector->text);
							std::copy(std::istream_iterator<double>(v_knots_buffer), std::istream_iterator<double>(), std::back_inserter(patch->v_knots));

							std::istringstream points_buffer(xml_control_points->text);
							std::istringstream weights_buffer(xml_weights->text);

							std::istream_iterator<unsigned long> control_point(points_buffer);
							std::istream_iterator<double> weight(weights_buffer);
							for(; control_point != std::istream_iterator<unsigned long>() && weight != std::istream_iterator<double>(); ++control_point, ++weight)
								{
									const unsigned long control_point_index = *control_point - 1;
									return_if_fail(control_point_index < Mesh.points.size());
									patch->control_points.push_back(k3d::nupatch::control_point(Mesh.points[control_point_index], *weight));
								}
						}
				}
		}
}

bool import_file(idocument& Document, idocument_read_format& FormatFilter, const boost::filesystem::path& File)
{
	return FormatFilter.read_file(Document, File);
}

bool export_file(idocument& Document, idocument_write_format& FormatFilter, const boost::filesystem::path& File)
{
	return FormatFilter.write_file(Document, File);
}

void save_node(ipersistent& Node, xml::element& XML, const ipersistent::save_context& Context)
{
	inode* const node = dynamic_cast<inode*>(&Node);
	return_if_fail(node);

	xml::element& xml_node = XML.append(element("node",
		attribute("name", node->name()),
		attribute("class", node->factory().class_id()),
		attribute("id", Context.lookup.lookup_id(node))));
	Node.save(xml_node, Context);
}

void save_mesh_selection(const mesh_selection& Selection, xml::element& XML, const ipersistent::save_context& Context)
{
	detail::save_selection(XML, Selection.points, "points");
	detail::save_selection(XML, Selection.edges, "edges");
	detail::save_selection(XML, Selection.faces, "faces");
	detail::save_selection(XML, Selection.linear_curves, "linearcurves");
	detail::save_selection(XML, Selection.cubic_curves, "cubiccurves");
	detail::save_selection(XML, Selection.nucurves, "nucurves");
	detail::save_selection(XML, Selection.bilinear_patches, "bilinearpatches");
	detail::save_selection(XML, Selection.bicubic_patches, "bicubicpatches");
	detail::save_selection(XML, Selection.nupatches, "nupatches");
}

void load_mesh_selection(mesh_selection& Selection, xml::element& XML, const ipersistent::load_context& Context)
{
	for(element::elements_t::const_iterator xml_selection = XML.children.begin(); xml_selection != XML.children.end(); ++xml_selection)
		{
			if(xml_selection->name == "points")
				detail::load_selection(*xml_selection, Selection.points);
			if(xml_selection->name == "edges")
				detail::load_selection(*xml_selection, Selection.edges);
			if(xml_selection->name == "faces")
				detail::load_selection(*xml_selection, Selection.faces);
			if(xml_selection->name == "linearcurves")
				detail::load_selection(*xml_selection, Selection.linear_curves);
			if(xml_selection->name == "cubiccurves")
				detail::load_selection(*xml_selection, Selection.cubic_curves);
			if(xml_selection->name == "nucurves")
				detail::load_selection(*xml_selection, Selection.nucurves);
			if(xml_selection->name == "bilinearpatches")
				detail::load_selection(*xml_selection, Selection.bilinear_patches);
			if(xml_selection->name == "bicubicpatches")
				detail::load_selection(*xml_selection, Selection.bicubic_patches);
			if(xml_selection->name == "nupatches")
				detail::load_selection(*xml_selection, Selection.nupatches);
		}
}

} // namespace k3d


