// K-3D
// Copyright (c) 1995-2005, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\author Anders Dahnielson (anders@dahnielson.com)
*/

#include <k3dsdk/i18n.h>
#include <k3dsdk/module.h>

#include "basic_bitmap_modifier.h"

namespace libk3dbitmap
{

/////////////////////////////////////////////////////////////////////////////
// bitmap_color_monochrome

class bitmap_color_monochrome :
	public basic_bitmap_modifier
{
	typedef basic_bitmap_modifier base;

public:
	bitmap_color_monochrome(k3d::idocument& Document) :
		base(Document),
		m_red_weight(init_owner(*this) + init_name("red_weight") + init_label(_("Red weight")) + init_description(_("Scale Red componnt value")) + init_value(0.3)),
		m_green_weight(init_owner(*this) + init_name("green_weight") + init_label(_("Green weight")) + init_description(_("Scale Green component value")) + init_value(0.59)),
		m_blue_weight(init_owner(*this) + init_name("blue_weight") + init_label(_("Blue weight")) + init_description(_("Scale Blue component value")) + init_value(0.11))
	{
		m_red_weight.changed_signal().connect(sigc::mem_fun(*this, &bitmap_color_monochrome::on_value_change));
		m_green_weight.changed_signal().connect(sigc::mem_fun(*this, &bitmap_color_monochrome::on_value_change));
		m_blue_weight.changed_signal().connect(sigc::mem_fun(*this, &bitmap_color_monochrome::on_value_change));
	}

	void on_value_change()
	{
		m_red_cache   = m_red_weight.value();
		m_green_cache = m_green_weight.value();
		m_blue_cache  = m_blue_weight.value();

		on_reset_bitmap();
	}

        void on_filter(const k3d::pixel& I, k3d::pixel& O)
	{
		O.red = O.green = O.blue = (I.red * m_red_cache) + (I.green * m_green_cache) + (I.blue * m_blue_cache);
	}

	k3d::iplugin_factory& factory()
	{
		return get_factory();
	}

	static k3d::iplugin_factory& get_factory()
	{
		static k3d::plugin_factory<
			k3d::document_plugin<bitmap_color_monochrome>,
			k3d::interface_list<k3d::ibitmap_source,
			k3d::interface_list<k3d::ibitmap_sink> > > factory(
				k3d::uuid(0x73cd5356, 0x4d3d4eaf, 0xacda0d76, 0x8070a016),
				"BitmapColorMonochrome",
				_("Make any image black and white"),
				"Bitmap",
				k3d::iplugin_factory::STABLE);

		return factory;
	}

private:
	k3d_data(double, immutable_name, change_signal, with_undo, local_storage, no_constraint, writable_property, with_serialization) m_red_weight;
	k3d_data(double, immutable_name, change_signal, with_undo, local_storage, no_constraint, writable_property, with_serialization) m_green_weight;
	k3d_data(double, immutable_name, change_signal, with_undo, local_storage, no_constraint, writable_property, with_serialization) m_blue_weight;

	double m_red_cache;
	double m_green_cache;
	double m_blue_cache;
};

/////////////////////////////////////////////////////////////////////////////
// bitmap_color_monochrome_factory

k3d::iplugin_factory& bitmap_color_monochrome_factory()
{
	return bitmap_color_monochrome::get_factory();
}

} // namespace libk3dbitmap


