// K-3D
// Copyright (c) 1995-2005, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\author Anders Dahnielson (anders@dahnielson.com)
*/

#include <k3dsdk/i18n.h>
#include <k3dsdk/module.h>

#include "basic_bitmap_modifier.h"

namespace libk3dbitmap
{

/////////////////////////////////////////////////////////////////////////////
// bitmap_gamma

class bitmap_gamma :
	public basic_bitmap_modifier
{
	typedef basic_bitmap_modifier base;

public:
	bitmap_gamma(k3d::idocument& Document) :
		base(Document),
		m_value(init_owner(*this) + init_name("value") + init_label(_("Gamma value")) + init_description(_("Apply gamma value to each pixel")) + init_value(0.0))
	{
		m_value.changed_signal().connect(sigc::mem_fun(*this, &bitmap_gamma::on_value_change));
	}

	void on_value_change()
	{
		m_value_cache = m_value.value();

		on_reset_bitmap();
	}

	void on_filter(const k3d::pixel& I, k3d::pixel& O)
	{
		double I_red   = k3d::color_traits<double>::convert(I.red);
		double I_green = k3d::color_traits<double>::convert(I.green);
		double I_blue  = k3d::color_traits<double>::convert(I.blue);
		double I_alpha = k3d::color_traits<double>::convert(I.alpha);

		// Matte divide ...
		if(I_alpha != 0)
		{
			I_red   = I_red   / I_alpha;
			I_green = I_green / I_alpha;
			I_blue  = I_blue  / I_alpha;
		}

		// Color operation ...
		const double O_red   = std::pow(I_red,   1 / m_value_cache);
		const double O_green = std::pow(I_green, 1 / m_value_cache);
		const double O_blue  = std::pow(I_blue,  1 / m_value_cache);

		// Matte multiply ...
		O.red   = k3d::bitmap::pixel_type::sample_traits::convert(O_red   * I_alpha);
		O.green = k3d::bitmap::pixel_type::sample_traits::convert(O_green * I_alpha);
		O.blue  = k3d::bitmap::pixel_type::sample_traits::convert(O_blue  * I_alpha);
		O.alpha = I.alpha;
	}

	k3d::iplugin_factory& factory()
	{
		return get_factory();
	}

	static k3d::iplugin_factory& get_factory()
	{
		static k3d::plugin_factory<
			k3d::document_plugin<bitmap_gamma>,
			k3d::interface_list<k3d::ibitmap_source,
			k3d::interface_list<k3d::ibitmap_sink> > > factory(
				k3d::uuid(0xa2ff08c9, 0x96e54063, 0x907ad507, 0xec36dd1c),
				"BitmapGamma",
				_("Apply gamma value to each pixel"),
				"Bitmap",
				k3d::iplugin_factory::STABLE);

		return factory;
	}

private:
	k3d_data(double, immutable_name, change_signal, with_undo, local_storage, no_constraint, writable_property, with_serialization) m_value;

	double m_value_cache;
};

/////////////////////////////////////////////////////////////////////////////
// bitmap_gamma_factory

k3d::iplugin_factory& bitmap_gamma_factory()
{
	return bitmap_gamma::get_factory();
}

} // namespace libk3dbitmap


