// K-3D
// Copyright (c) 1995-2005, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\author Tim Shead (tshead@k-3d.com)
*/

#include <k3dsdk/application.h>
#include <k3dsdk/basic_math.h>
#include <k3dsdk/bezier.h>
#include <k3dsdk/classes.h>
#include <k3dsdk/color.h>
#include <k3dsdk/geometry.h>
#include <k3dsdk/i18n.h>
#include <k3dsdk/ibezier_channel.h>
#include <k3dsdk/ichannel.h>
#include <k3dsdk/itime_sink.h>
#include <k3dsdk/persistent.h>
#include <k3dsdk/module.h>
#include <k3dsdk/node.h>
#include <k3dsdk/property.h>
#include <k3dsdk/state_change_set.h>
#include <k3dsdk/string_modifiers.h>
#include <k3dsdk/time_source.h>
#include <k3dsdk/vectors.h>

using namespace k3d::xml;

#include <iterator>
#include <set>

// We have an unfortunate clash with X
#ifdef RootWindow
#undef RootWindow
#endif // RootWindow

namespace libk3dchannels
{

namespace detail
{

/////////////////////////////////////////////////////////////////////////////
// color_bezier_channel

/// Encapsulates a data value that can change over time, controlled by a Bezier curve
class color_bezier_channel :
	public k3d::persistent<k3d::node> ,
	public k3d::ichannel<k3d::color>,
	public k3d::ibezier_channel<k3d::color>,
	public k3d::itime_sink
{
	typedef k3d::persistent<k3d::node>  base;
	typedef k3d::ibezier_channel<k3d::color>::control_points_t control_points_t;
	typedef k3d::ibezier_channel<k3d::color>::values_t values_t;

public:
	color_bezier_channel(k3d::idocument& Document) :
		base(Document),
		m_input(init_owner(*this) + init_name("input") + init_label(_("Input")) + init_description(_("Input Value")) + init_value(0.0)),
		m_control_points(init_owner(*this) + init_value(control_points_t(1, k3d::vector2(0.0, 0.0)))),
		m_values(init_owner(*this) + init_value(values_t(1, k3d::color(1, 1, 1)))),
		m_output(init_owner(*this) + init_name("output") + init_label(_("Output Value")) + init_description(_("Output Value")) + init_slot(sigc::mem_fun(*this, &color_bezier_channel::get_value)))
	{
		m_input.changed_signal().connect(m_output.changed_signal().make_slot());
		m_control_points.changed_signal().connect(m_output.changed_signal().make_slot());
		m_values.changed_signal().connect(m_output.changed_signal().make_slot());
	}

	k3d::iproperty& time_sink_input()
	{
		return m_input;
	}

	k3d::color get_value()
	{
		return value();
	}

	const k3d::color value()
	{
		return value(m_input.value(), 0.001);
	}

	const k3d::color value(const double X, const double MaxError)
	{
		// Special-case the easiest configuration ...
		if(1 == m_control_points.internal_value().size())
			return m_values.internal_value().front();

		// Convert document time to our own local time ...
		const double time = X;

		// We're before the first control point, so that's the value ...
		if(time <= m_control_points.internal_value().front()[0])
			return m_values.internal_value().front();

		values_t::iterator value = m_values.internal_value().begin();
		for(control_points_t::iterator control_point = m_control_points.internal_value().begin(); control_point != m_control_points.internal_value().end() - 1; std::advance(control_point, 3), std::advance(value, 1))
			{
				if(time > (*(control_point + 3))[0])
					continue;

				double mix = 0.0;
				const double dx = (*(control_point+3))[0] - (*control_point)[0];
				if(dx)
					{
						double error = 0.0;
						unsigned long iterations = 0;
						double mix = k3d::bezier_function<3, k3d::vector2>(control_point, control_point + 4, time, (time - (*control_point)[0]) / dx, MaxError * dx, 1000, error, iterations);

						// Adjust the mix for our "sawtooth" pattern ...
						if((*control_point)[1] > (*(control_point+3))[1])
							mix = 1.0 - mix;

						// Linear interpolate the final value ...
//						return k3d::mix<k3d::color>(*value, *(value+1), mix);
						return k3d::color(k3d::mix(value->red, (value+1)->red, mix), k3d::mix(value->green, (value+1)->green, mix), k3d::mix(value->blue, (value+1)->blue, mix));
					}
				else
					{
						mix = (*control_point)[1];
					}

			}

		// We're after the last control point, so that's the value ...
		return m_values.internal_value().back();
	}

	changed_signal_t& changed_signal()
	{
		return m_output.changed_signal();
	}

	void set_curve(const control_points_t ControlPoints, const values_t Values)
	{
		// Sanity checks ...
		return_if_fail(ControlPoints.size());
		return_if_fail(ControlPoints.size() == ((((ControlPoints.size() - 1) / 3) * 3) + 1));
		return_if_fail(Values.size() == (((ControlPoints.size() - 1) / 3) + 1));

		m_control_points.set_value(ControlPoints);
		m_values.set_value(Values);
	}

	void get_curve(control_points_t& ControlPoints, values_t& Values)
	{
		ControlPoints = m_control_points.internal_value();
		Values = m_values.internal_value();
	}

	void save(element& Element, const k3d::ipersistent::save_context& Context)
	{
		base::save(Element, Context);

		element& nodes = Element.append(element("nodes"));

		control_points_t::iterator control_point = m_control_points.internal_value().begin();
		values_t::iterator value = m_values.internal_value().begin();
		for(unsigned int i = 0; i < m_control_points.internal_value().size(); ++i)
			{
				element& node = nodes.append(element("node", attribute("coords", *control_point)));

				if(0 == i % 3)
					{
						node.append(attribute("value", *value));
						++value;
					}

				++control_point;
			}
	}

	void load(element& Element, const k3d::ipersistent::load_context& Context)
	{
		base::load(Element, Context);

		element* const nodes = find_element(Element, "nodes");
		return_if_fail(nodes);

		// Because we already have control points by default, we need to start from scratch ...
		m_control_points.internal_value().clear();
		m_values.internal_value().clear();

		for(element::elements_t::iterator element = nodes->children.begin(); element != nodes->children.end(); ++element)
			{
				if(element->name == "node" || element->name == "valuenode")
					{
						m_control_points.internal_value().push_back(attribute_value<k3d::vector2>(*element, "coords", k3d::vector2(0.0, 0.0)));

						attribute* const value = find_attribute(*element, "value");
						if(value)
							m_values.internal_value().push_back(attribute_value<k3d::color>(*element, "value", k3d::color(0, 0, 0)));
					}
				else
					{
						k3d::log() << warning << k3d_file_reference << ": unknown element \"" << element->name << "\" will be ignored ... " << std::endl;
					}
			}

		// If for any reason we don't have the correct number of control points or values, reset ourselves to a sensible fallback position ...
		if(
			(m_control_points.internal_value().size()) &&
			(m_control_points.internal_value().size() == ((((m_control_points.internal_value().size() - 1) / 3) * 3) + 1)) &&
			(m_values.internal_value().size() == (((m_control_points.internal_value().size() - 1) / 3) + 1)))
		{
			return;
		}

		k3d::log() << error << k3d_file_reference << ": inconsistent node count [" << m_control_points.internal_value().size() << ", " << m_values.internal_value().size() << "] resetting to default" << std::endl;
		m_control_points.internal_value() = control_points_t(1, k3d::vector2(0.0, 0.0));
		m_values.internal_value() = values_t(1, k3d::color(0.0, 0.0, 0.0));
	}

	k3d::iplugin_factory& factory()
	{
		return get_factory();
	}

	static k3d::iplugin_factory& get_factory()
	{
		static k3d::plugin_factory<k3d::document_plugin<color_bezier_channel>, k3d::interface_list<k3d::ichannel<k3d::color>, k3d::interface_list<k3d::ibezier_channel<k3d::color> > > > factory(
			k3d::classes::ColorBezierChannel(),
			"ColorBezierChannel",
			_("Provides an animatable color property under Bezier spline control"),
			"Animation Color",
			k3d::iplugin_factory::DEPRECATED);

		return factory;
	}

private:
	friend class bezier_channel_properties;

	k3d_data(double, immutable_name, change_signal, with_undo, local_storage, no_constraint, writable_property, with_serialization) m_input;
	k3d_data(control_points_t, no_name, change_signal, with_undo, local_storage, no_constraint, no_property, no_serialization) m_control_points;
	k3d_data(values_t, no_name, change_signal, with_undo, local_storage, no_constraint, no_property, no_serialization) m_values;
	k3d_data(k3d::color, immutable_name, change_signal, no_undo, computed_storage, no_constraint, read_only_property, no_serialization) m_output;
};

} // namespace detail

/////////////////////////////////////////////////////////////////////////////
// color_bezier_channel_factory

k3d::iplugin_factory& color_bezier_channel_factory()
{
	return detail::color_bezier_channel::get_factory();
}

} // namespace libk3dchannels


