// K-3D
// Copyright (c) 1995-2005, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\author Timothy M. Shead (tshead@k-3d.com)
*/

#include <boost/filesystem/operations.hpp>

#include <k3dsdk/geometry.h>
#include <k3dsdk/node.h>
#include <k3dsdk/i18n.h>
#include <k3dsdk/imaterial.h>
#include <k3dsdk/persistent.h>
#include <k3dsdk/material.h>
#include <k3dsdk/material_client.h>
#include <k3dsdk/measurement.h>
#include <k3dsdk/mesh_source.h>
#include <k3dsdk/module.h>
#include <k3dsdk/options.h>

#include <iterator>

#ifdef K3D_PLATFORM_WIN32
	#define DEFAULT_FONT boost::filesystem::path("c:\\windows\\fonts\\arial.ttf", boost::filesystem::native)
#else // K3D_PLATFORM_WIN32
	#define DEFAULT_FONT boost::filesystem::path("/usr/X11R6/lib/X11/fonts/ttf/arial.ttf", boost::filesystem::native)
#endif // !K3D_PLATFORM_WIN32

#include <ft2build.h>
#include FT_FREETYPE_H
#include FT_OUTLINE_H

namespace libk3dfreetype2
{

namespace detail
{

/// Defines a closed contour
typedef std::vector<k3d::vector3> contour_t;

/// Returns the signed 2D area of a contour
const double area(const contour_t& Contour)
{
	double result = 0;

	for(unsigned long i = 0; i != Contour.size(); ++i)
		result += (Contour[i][0] * Contour[(i+1)%Contour.size()][1]) - (Contour[(i+1)%Contour.size()][0] * Contour[i][1]);

	return result * 0.5;
}

/// Returns true iff a contour is clockwise
const bool clockwise(const contour_t& Contour)
{
	return area(Contour) < 0;
}

/// Encapsulates the freetype FT_Library struct to provide RAII behavior
class freetype_library
{
public:
	freetype_library() :
		m_initialized(0 == FT_Init_FreeType(&m_library))
	{
	}

	~freetype_library()
	{
		if(m_initialized)
			FT_Done_FreeType(m_library);
	}

	operator bool()
	{
		return m_initialized;
	}

	operator FT_Library()
	{
		return m_library;
	}

private:
	const bool m_initialized;
	FT_Library m_library;
};

/// Encapsulates the freetype FT_Face struct to provide RAII behavior
class freetype_face
{
public:
	freetype_face(FT_Library Library, const boost::filesystem::path& Path) :
		m_initialized(0 == FT_New_Face(Library, Path.native_file_string().c_str(), 0, &m_face))
	{
	}

	~freetype_face()
	{
		if(m_initialized)
			FT_Done_Face(m_face);
	}

	operator bool()
	{
		return m_initialized;
	}

	operator FT_Face()
	{
		return m_face;
	}

	FT_Face& face()
	{
		return m_face;
	}

private:
	const bool m_initialized;
	FT_Face m_face;
};

/// Encapsulates a set of callback functions that will receive data about a glyph outline from the freetype library, and generate K-3D geometry to match
class freetype_outline
{
public:
	freetype_outline(k3d::mesh& Mesh, k3d::polyhedron& Polyhedron, k3d::imaterial* const Material, const unsigned long CurveDivisions, const double Scale, const k3d::vector3& Offset) :
		mesh(Mesh),
		polyhedron(Polyhedron),
		material(Material),
		curve_divisions(CurveDivisions),
		scale(Scale),
		offset(Offset)
	{
		ft_outline_funcs.move_to = raw_move_to_func;
		ft_outline_funcs.line_to = raw_line_to_func;
		ft_outline_funcs.conic_to = raw_conic_to_func;
		ft_outline_funcs.cubic_to = raw_cubic_to_func;
		ft_outline_funcs.shift = 0;
		ft_outline_funcs.delta = 0;
	}

	void decompose(FT_Outline& Outline)
	{
		// Generate a set of closed contours ...
		FT_Outline_Decompose(&Outline, &ft_outline_funcs, this);

		// Segregate contours into faces and holes, based on their orientation (clockwise or counter-clockwise, respectively)
		contours_t face_contours;
		contours_t hole_contours;
		for(contours_t::iterator contour = contours.begin(); contour != contours.end(); ++contour)
			{
				if(clockwise(*contour))
					face_contours.push_back(*contour);
				else
					hole_contours.push_back(*contour);
			}

		// Create faces ...
		std::vector<k3d::face*> faces;
		for(contours_t::iterator contour = face_contours.begin(); contour != face_contours.end(); ++contour)
			{
				std::vector<k3d::split_edge*> edges;
				for(contour_t::iterator point = contour->begin(); point != contour->end(); ++point)
					{
						mesh.points.push_back(new k3d::point(*point));
						edges.push_back(new k3d::split_edge(mesh.points.back()));
					}
				k3d::loop_edges(edges.begin(), edges.end());

				polyhedron.faces.push_back(new k3d::face(edges.front(), material));
				faces.push_back(polyhedron.faces.back());
			}

		// Add holes ...
		for(contours_t::iterator contour = hole_contours.begin(); contour != hole_contours.end(); ++contour)
			{
				return_if_fail(!faces.empty());

				std::vector<k3d::split_edge*> edges;
				for(contour_t::iterator point = contour->begin(); point != contour->end(); ++point)
					{
						mesh.points.push_back(new k3d::point(*point));
						edges.push_back(new k3d::split_edge(mesh.points.back()));
					}
				k3d::loop_edges(edges.begin(), edges.end());

				faces[0]->holes.push_back(edges.front());
			}
	}

private:
	void begin_contour(const k3d::vector3& From)
	{
		contours.push_back(contour_t());
		last_point = From;
	}

	void line_to(const k3d::vector3& To)
	{
		contours.back().push_back(To);
		last_point = To;
	}

	void conic_to(const k3d::vector3& From, const k3d::vector3& Control, const k3d::vector3& To)
	{
		std::vector<k3d::vector3> control_points;
		control_points.push_back(From);
		control_points.push_back(Control);
		control_points.push_back(To);

		for(unsigned long i = 0; i != curve_divisions; ++i)
			{
				contours.back().push_back(k3d::Bezier<k3d::vector3>(control_points, static_cast<double>(i+1) / static_cast<double>(curve_divisions)));
			}

		last_point = To;
	}

	void cubic_to(const k3d::vector3& From, const k3d::vector3& Control1, const k3d::vector3& Control2, const k3d::vector3& To)
	{
		std::vector<k3d::vector3> control_points;
		control_points.push_back(From);
		control_points.push_back(Control1);
		control_points.push_back(Control2);
		control_points.push_back(To);

		for(unsigned long i = 0; i != curve_divisions; ++i)
			{
				contours.back().push_back(k3d::Bezier<k3d::vector3>(control_points, static_cast<double>(i+1) / static_cast<double>(curve_divisions)));
			}

		last_point = To;
	}

	const k3d::vector3 convert(FT_Vector* RHS)
	{
		return (k3d::vector3(RHS->x, RHS->y, 0) * scale) + offset;
	}

	int move_to_func(FT_Vector* To)
	{
		begin_contour(convert(To));
		return 0;
	}

	int line_to_func(FT_Vector* To)
	{
		line_to(convert(To));
		return 0;
	}

	int conic_to_func(FT_Vector* Control, FT_Vector* To)
	{
		conic_to(last_point, convert(Control), convert(To));
		return 0;
	}

	int cubic_to_func(FT_Vector* Control1, FT_Vector* Control2, FT_Vector* To)
	{
		cubic_to(last_point, convert(Control1), convert(Control2), convert(To));
		return 0;
	}

	static int raw_move_to_func(FT_Vector* to, void* user)
	{
		return reinterpret_cast<freetype_outline*>(user)->move_to_func(to);
	}

	static int raw_line_to_func(FT_Vector* to, void* user)
	{
		return reinterpret_cast<freetype_outline*>(user)->line_to_func(to);
	}

	static int raw_conic_to_func(FT_Vector* control, FT_Vector* to, void* user)
	{
		return reinterpret_cast<freetype_outline*>(user)->conic_to_func(control, to);
	}

	static int raw_cubic_to_func(FT_Vector* control1, FT_Vector* control2, FT_Vector* to, void* user)
	{
		return reinterpret_cast<freetype_outline*>(user)->cubic_to_func(control1, control2, to);
	}

	k3d::mesh& mesh;
	k3d::polyhedron& polyhedron;
	k3d::imaterial* const material;
	const unsigned long curve_divisions;
	const double scale;
	const k3d::vector3 offset;
	FT_Outline_Funcs ft_outline_funcs;

	k3d::vector3 last_point;
	typedef std::vector<contour_t> contours_t;
	contours_t contours;
};

} // namespace detail

/////////////////////////////////////////////////////////////////////////////
// poly_text

class poly_text :
	public k3d::material_client<k3d::mesh_source<k3d::persistent<k3d::node> > >
{
	typedef k3d::material_client<k3d::mesh_source<k3d::persistent<k3d::node> > > base;

public:
	poly_text(k3d::idocument& Document) :
		base(Document),
		m_font_path(init_owner(*this) + init_name("font") + init_label(_("Font")) + init_description(_("Font path")) + init_value(DEFAULT_FONT) + init_path_mode(k3d::ipath_property::READ) + init_path_type(k3d::options::path::fonts())),
		m_text(init_owner(*this) + init_name("text") + init_label(_("Text")) + init_description(_("Text")) + init_value<std::string>("Text!")),
		m_curve_divisions(init_owner(*this) + init_name("curve_divisions") + init_label(_("Curve subdivisions")) + init_description(_("Bezier curves subdivision number")) + init_value(3) + init_constraint(constraint::minimum(1L)) + init_precision(0) + init_step_increment(1) + init_units(typeid(k3d::measurement::scalar))),
		m_height(init_owner(*this) + init_name("height") + init_label(_("Height")) + init_description(_("Font scale")) + init_value(10.0) + init_precision(2) + init_step_increment(0.01) + init_units(typeid(k3d::measurement::distance)))
	{
		m_material.changed_signal().connect(sigc::mem_fun(*this, &poly_text::on_reset_geometry));

		m_font_path.changed_signal().connect(sigc::mem_fun(*this, &poly_text::on_reset_geometry));
		m_text.changed_signal().connect(sigc::mem_fun(*this, &poly_text::on_reset_geometry));
		m_curve_divisions.changed_signal().connect(sigc::mem_fun(*this, &poly_text::on_reset_geometry));
		m_height.changed_signal().connect(sigc::mem_fun(*this, &poly_text::on_reset_geometry));

		m_output_mesh.need_data_signal().connect(sigc::mem_fun(*this, &poly_text::on_create_geometry));
	}

	void on_reset_geometry()
	{
		m_output_mesh.reset();
	}

	k3d::mesh* on_create_geometry()
	{
		const boost::filesystem::path font_path = m_font_path.value();
		const std::string text = m_text.value();
		const unsigned long curve_divisions = m_curve_divisions.value();
		const double height = m_height.value();
		k3d::imaterial* const material = m_material.value();

		std::auto_ptr<k3d::mesh> mesh(new k3d::mesh());

		k3d::polyhedron* const polyhedron = new k3d::polyhedron();
		mesh->polyhedra.push_back(polyhedron);

		detail::freetype_library ft_library;
		if(!ft_library)
			{
				k3d::log() << error << "Error initializing FreeType library" << std::endl;
				return 0;
			}

		detail::freetype_face ft_face(ft_library, font_path);
		if(!ft_face)
			{
				k3d::log() << error << "Error opening font file: " << font_path.native_file_string() << std::endl;
				return 0;
			}

		if(!FT_IS_SCALABLE(ft_face.face()))
			{
				k3d::log() << error << "Not a scalable font: " << font_path.native_file_string() << std::endl;
				return 0;
			}

		const double normalize_height = 1.0 / static_cast<double>(ft_face.face()->bbox.yMax - ft_face.face()->bbox.yMin);
		const double scale = normalize_height * height;

		// For each character in the string ...
		k3d::vector3 offset;
		for(std::string::const_iterator c = text.begin(); c != text.end(); ++c)
			{
				if(0 != FT_Load_Glyph(ft_face, FT_Get_Char_Index(ft_face, static_cast<FT_ULong>(*c)), FT_LOAD_NO_SCALE | FT_LOAD_IGNORE_TRANSFORM))
					{
						k3d::log() << error << "Error loading glyph for " << font_path.native_file_string() << "[" << *c << "]" << std::endl;
						continue;
					}

				detail::freetype_outline outline(*mesh, *polyhedron, material, curve_divisions, scale, offset);
				outline.decompose(ft_face.face()->glyph->outline);

				offset += k3d::vector3(ft_face.face()->glyph->metrics.horiAdvance, 0, 0) * normalize_height * height;
			}

		assert_warning(is_valid(*polyhedron));

		return mesh.release();
	}

	k3d::iplugin_factory& factory()
	{
		return get_factory();
	}

	static k3d::iplugin_factory& get_factory()
	{
		static k3d::plugin_factory<k3d::document_plugin<poly_text>, k3d::interface_list<k3d::imesh_source > > factory(
			k3d::uuid(0x9acaeaf1, 0x1fe74387, 0xae71cbb3, 0x9b5e33fd),
			"PolyText",
			_("Generates polygonal text using Freetype 2"),
			"Polygons Text",
			k3d::iplugin_factory::STABLE);

		return factory;
	}

private:
	k3d_data(boost::filesystem::path, immutable_name, change_signal, with_undo, local_storage, no_constraint, path_property, path_serialization) m_font_path;
	k3d_data(std::string, immutable_name, change_signal, with_undo, local_storage, no_constraint, writable_property, with_serialization) m_text;
	k3d_data(long, immutable_name, change_signal, with_undo, local_storage, with_constraint, measurement_property, with_serialization) m_curve_divisions;
	k3d_data(double, immutable_name, change_signal, with_undo, local_storage, no_constraint, measurement_property, with_serialization) m_height;
};

} // namespace libk3dfreetype2


K3D_MODULE_START(libk3dfreetype2, k3d::uuid(0xd0691ef7, 0x0d6c41c0, 0xa607bea2, 0x09d386f5),  Registry)
	Registry.register_factory(libk3dfreetype2::poly_text::get_factory());
K3D_MODULE_END


