// K-3D
// Copyright (c) 2005, Romain Behar
//
// Contact: romainbehar@yahoo.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\author Romain Behar (romainbehar@yahoo.com)
*/

#include <k3dsdk/i18n.h>
#include <k3dsdk/measurement.h>
#include <k3dsdk/mesh_modifier.h>
#include <k3dsdk/mesh_selection_sink.h>
#include <k3dsdk/module.h>
#include <k3dsdk/node.h>
#include <k3dsdk/persistent.h>
#include <k3dsdk/utility.h>
#include <k3dsdk/vectors.h>

namespace libk3dmesh
{

namespace detail
{

struct subdivide_edges
{
	subdivide_edges(const double Subdivisions, k3d::mesh& Mesh) :
		subdivisions(Subdivisions),
		mesh(Mesh)
	{
	}

	void operator()(k3d::split_edge* Edge)
	{
		// Unselect edge
		Edge->selection_weight = 0;
		Edge->visible_selection = false;

		const k3d::vector3 start_point = Edge->vertex->position;
		const k3d::vector3 end_point = Edge->face_clockwise->vertex->position;

		// Create new points
		typedef std::vector<k3d::point*> points_t;
		points_t new_points;
		typedef std::vector<k3d::split_edge*> edges_t;
		edges_t new_edges;

		new_edges.push_back(Edge);

		const k3d::vector3 step = (end_point - start_point) / (subdivisions + 1);

		k3d::split_edge* edge = Edge;
		for(double n = 0; n < subdivisions; ++n)
		{
			const k3d::vector3 new_position = start_point + (n + 1)*step;
			k3d::point* new_point = new k3d::point(new_position);
			mesh.points.push_back(new_point);

			new_points.push_back(new_point);

			// Select resulting points
			new_point->selection_weight = 1.0;
			new_point->visible_selection = true;

			k3d::split_edge* new_edge = new k3d::split_edge(new_point);
			new_edge->face_clockwise = edge->face_clockwise;
			edge->face_clockwise = new_edge;

			new_edges.push_back(new_edge);

			edge = new_edge;
		}

		if(!Edge->companion)
			return;

		// Subdivide companion
		edge = Edge->companion;
		edge->selection_weight = 0;
		edge->visible_selection = false;

		new_points.push_back(edge->vertex);

		points_t::reverse_iterator point = new_points.rbegin();
		edges_t::reverse_iterator companion = new_edges.rbegin();
		for(; point != new_points.rend(); ++point, ++companion)
		{
			k3d::split_edge* new_edge;

			if(point == new_points.rbegin())
			{
				new_edge = edge;
			}
			else
			{
				new_edge = new k3d::split_edge(*point);
				new_edge->face_clockwise = edge->face_clockwise;
				edge->face_clockwise = new_edge;
			}

			k3d::join_edges(*new_edge, **companion);

			edge = new_edge;
		}
	}

	const double subdivisions;
	k3d::mesh& mesh;
};

typedef std::set<k3d::split_edge*> edges_t;

/// Functor that gets selected edges, and only one from a companion pair
struct get_edges
{
	get_edges(edges_t& Edges, const bool Selected) :
		edges(Edges), selected(Selected)
	{
	}

	void operator()(k3d::split_edge& Edge)
	{
		if(selected && !Edge.selection_weight)
			return;

		// Test whether companion is already in the set
		if(Edge.companion)
			if(edges.find(Edge.companion) != edges.end())
				return;

		// Save edge
		edges.insert(&Edge);
	}

	edges_t& edges;
	const bool selected;
};

/// Function template in the spirit of std::for_each that applies a functor to every edge in a polyhedron
template<typename T>
void for_each_edge(k3d::polyhedron& Polyhedron, T Functor)
{
	for(k3d::polyhedron::faces_t::const_iterator face = Polyhedron.faces.begin(); face != Polyhedron.faces.end(); ++face)
	{
		k3d::split_edge* edge = (*face)->first_edge;
		do
		{
			Functor(*edge);

			edge = edge->face_clockwise;
		}
		while(edge != (*face)->first_edge);
	}
}

} // namespace detail

/////////////////////////////////////////////////////////////////////////////
// subdivide_edges_implementation

class subdivide_edges_implementation :
	public k3d::mesh_selection_sink<k3d::mesh_modifier<k3d::persistent<k3d::node> > >
{
	typedef k3d::mesh_selection_sink<k3d::mesh_modifier<k3d::persistent<k3d::node> > > base;

public:
	subdivide_edges_implementation(k3d::idocument& Document) :
		base(Document),
		m_vertex_number(init_owner(*this) + init_name("vertices") + init_label(_("Vertices")) + init_description(_("New vertex number (subdivisions - 1)")) + init_value(1L) + init_precision(0) + init_step_increment(1) + init_units(typeid(k3d::measurement::scalar)) + init_constraint(constraint::minimum(1L)))
	{
		m_input_mesh.changed_signal().connect(sigc::mem_fun(*this, &subdivide_edges_implementation::on_reset_geometry));
		m_vertex_number.changed_signal().connect(sigc::mem_fun(*this, &subdivide_edges_implementation::on_reset_geometry));
		m_mesh_selection.changed_signal().connect(sigc::mem_fun(*this, &subdivide_edges_implementation::on_reset_geometry));
		m_output_mesh.need_data_signal().connect(sigc::mem_fun(*this, &subdivide_edges_implementation::on_create_geometry));
	}

	void on_reset_geometry()
	{
		m_output_mesh.reset();
	}

	k3d::mesh* on_create_geometry()
	{
		// Get the input geometry ...
		k3d::mesh* const input = m_input_mesh.value();
		if(!input)
			return 0;

		// Create output geometry ...
		k3d::mesh* const output = new k3d::mesh();
		k3d::deep_copy(*input, *output);
		k3d::replace_selection(m_mesh_selection.value(), *output);

		const double vertex_number = static_cast<double>(m_vertex_number.value());

		for(k3d::mesh::polyhedra_t::iterator polyhedron = output->polyhedra.begin(); polyhedron != output->polyhedra.end(); ++polyhedron)
		{
			// Get selected egdes
			detail::edges_t selected_edges;
			detail::for_each_edge(**polyhedron, detail::get_edges(selected_edges, true));

			// If no edge is selected, subdivide all of them
			if(!selected_edges.size())
				detail::for_each_edge(**polyhedron, detail::get_edges(selected_edges, false));

			// Subdivide edges
			std::for_each(selected_edges.begin(), selected_edges.end(), detail::subdivide_edges(vertex_number, *output));
		}

		return output;
	}

	k3d::iplugin_factory& factory()
	{
		return get_factory();
	}

	static k3d::iplugin_factory& get_factory()
	{
		static k3d::plugin_factory<
			k3d::document_plugin<subdivide_edges_implementation>,
			k3d::interface_list<k3d::imesh_source,
			k3d::interface_list<k3d::imesh_sink > > > factory(
				k3d::uuid(0x62de51ff, 0xe70f423a, 0x953d899a, 0x1dbf5c38),
				"SubdivideEdges",
				"Subdivides edges by creating one or more vertices along",
				"Objects",
				k3d::iplugin_factory::STABLE);

		return factory;
	}

private:
	k3d_data(long, immutable_name, change_signal, with_undo, local_storage, with_constraint, measurement_property, with_serialization) m_vertex_number;
};

/////////////////////////////////////////////////////////////////////////////
// subdivide_edges_factory

k3d::iplugin_factory& subdivide_edges_factory()
{
	return subdivide_edges_implementation::get_factory();
}

} // namespace libk3dmesh

