// K-3D
// Copyright (c) 1995-2005, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\brief Implements the RenderManScript K-3D object, which can insert scripted data into RenderMan output
		\author Tim Shead (tshead@k-3d.com)
*/

#include <k3dsdk/create_plugins.h>
#include <k3dsdk/i18n.h>
#include <k3dsdk/iapplication.h>
#include <k3dsdk/iscript_engine.h>
#include <k3dsdk/mesh_source.h>
#include <k3dsdk/module.h>
#include <k3dsdk/node.h>
#include <k3dsdk/persistent.h>
#include <k3dsdk/scripting.h>
#include <k3dsdk/user_property_changed_signal.h>

#define DEFAULT_SCRIPT "#python\n\n\
import k3d\n\n\
positions = ((-5, 5, 0), (5, 5, 0), (5, -5, 0), (-5, -5, 0))\n\
points = []\n\
for position in positions:\n\
	points.append(k3d.OutputMesh.new_point(position))\n\
polyhedron = k3d.OutputMesh.new_polyhedron()\n\
edges = []\n\
for point in points:\n\
	edges.append(polyhedron.new_edge(point))\n\
for i in range(len(edges)):\n\
	edges[i].face_clockwise = edges[(i+1)\%len(edges)]\n\
face = polyhedron.new_face(edges[0])\n"

namespace libk3dscripting
{

/////////////////////////////////////////////////////////////////////////////
// mesh_source_script

class mesh_source_script :
	public k3d::mesh_source<k3d::persistent<k3d::node> >
{
	typedef k3d::mesh_source<k3d::persistent<k3d::node> > base;

public:
	mesh_source_script(k3d::idocument& Document) :
		base(Document),
		m_script(init_owner(*this) + init_name("script") + init_label(_("Script")) + init_description(_("Script that creates geometry")) + init_value<std::string>(DEFAULT_SCRIPT)),
		m_script_engine(0),
		m_user_property_changed_signal(*this)
	{
		m_script.changed_signal().connect(sigc::mem_fun(*this, &mesh_source_script::on_reset_geometry));
		m_user_property_changed_signal.connect(sigc::mem_fun(*this, &mesh_source_script::on_reset_geometry));
		
		m_output_mesh.need_data_signal().connect(sigc::mem_fun(*this, &mesh_source_script::on_create_geometry));
	}

	~mesh_source_script()
	{
		delete dynamic_cast<k3d::ideletable*>(m_script_engine);
	}

	void on_reset_geometry()
	{
		m_output_mesh.reset();
	}

	k3d::mesh* on_create_geometry()
	{
		// Create the mesh, then give the script a crack at it ...
		std::auto_ptr<k3d::mesh> mesh(new k3d::mesh());

		// Examine the script to determine the correct language ...
		std::istringstream script(m_script.value());
		k3d::iplugin_factory* const language = k3d::recognize_script_language(script);
		return_val_if_fail(language, mesh.release());

		// If the current script engine is for the wrong language, lose it ...
		if(m_script_engine && (m_script_engine->factory().class_id() != language->class_id()))
		{
			delete dynamic_cast<k3d::ideletable*>(m_script_engine);
			m_script_engine = 0;
		}

		// Create our script engine as-needed ...
		if(!m_script_engine)
			m_script_engine = k3d::create_plugin<k3d::iscript_engine>(language->class_id());

		// No script engine?  We're outta here ...
		return_val_if_fail(m_script_engine, mesh.release());

		k3d::iscript_engine::context_t context;
		context["Document"] = static_cast<k3d::iunknown*>(&document());
		context["Object"] = static_cast<k3d::iunknown*>(this);
		context["OutputMesh"] = static_cast<k3d::iunknown*>(mesh.get());
		return_val_if_fail(m_script_engine->execute(name() + "Script", script, context), mesh.release());

		return mesh.release();
	}

	k3d::iplugin_factory& factory()
	{
		return get_factory();
	}

	static k3d::iplugin_factory& get_factory()
	{
		static k3d::plugin_factory< k3d::document_plugin<mesh_source_script>, k3d::interface_list<k3d::imesh_source> > factory(
			k3d::uuid(0xc68d0187, 0xbb334026, 0xb4ca6a9f, 0x0d980cc9),
			"MeshSourceScript",
			_("Mesh source that uses a script to create geometry"),
			"Scripting",
			k3d::iplugin_factory::STABLE);

		return factory;
	}

private:
	k3d_data(std::string, immutable_name, change_signal, with_undo, local_storage, no_constraint, script_property, with_serialization) m_script;
	k3d::iscript_engine* m_script_engine;
	k3d::user_property_changed_signal m_user_property_changed_signal;
};

/////////////////////////////////////////////////////////////////////////////
// mesh_source_script_factory

k3d::iplugin_factory& mesh_source_script_factory()
{
	return mesh_source_script::get_factory();
}

} // namespace libk3dscripting


