// K-3D
// Copyright (c) 1995-2005, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\brief Implements the k3d::spin_button class, which provides a UI for numeric quantities
		\author Tim Shead <tshead@k-3d.com>
		\author Dan Erikson <derikson@montana.com>
		\author Romain Behar <romainbehar@yahoo.com>
*/

#include "hotkey_entry.h"
#include "interactive.h"
#include "spin_button.h"

#include <k3dsdk/basic_math.h>
#include <k3dsdk/i18n.h>
#include <k3dsdk/iproperty.h>
#include <k3dsdk/istate_recorder.h>
#include <k3dsdk/iwritable_property.h>
#include <k3dsdk/measurement.h>
#include <k3dsdk/state_change_set.h>
#include <k3dsdk/string_cast.h>

#include <gtkmm/arrow.h>
#include <gtkmm/button.h>
#include <gtkmm/entry.h>
#include <gtkmm/window.h>

#include <iomanip>
#include <sstream>

using namespace k3d::measurement;

namespace libk3dngui
{

namespace spin_button
{

namespace detail
{

/////////////////////////////////////////////////////////////////////////////
// data_proxy

/// Specialization of k3d::spin_button::data_proxy for use with k3d::iproperty objects
class property_proxy :
	public idata_proxy
{
public:
	typedef k3d::iproperty data_t;

	explicit property_proxy(data_t& Data, k3d::istate_recorder* const StateRecorder, const Glib::ustring& ChangeMessage) :
		idata_proxy(StateRecorder, ChangeMessage),
		m_readable_data(Data),
		m_writable_data(dynamic_cast<k3d::iwritable_property*>(&Data))
	{
	}

	bool writable()
	{
		return m_writable_data ? true : false;
	}

	double value()
	{
		const std::type_info& type = m_readable_data.property_type();
		if(type == typeid(double))
			return boost::any_cast<double>(m_readable_data.property_value());
		else if(type == typeid(float))
			return boost::any_cast<float>(m_readable_data.property_value());
		else if(type == typeid(long))
			return boost::any_cast<long>(m_readable_data.property_value());
		else if(type == typeid(unsigned long))
			return boost::any_cast<unsigned long>(m_readable_data.property_value());
		else if(type == typeid(int))
			return boost::any_cast<int>(m_readable_data.property_value());
		else if(type == typeid(unsigned int))
			return boost::any_cast<unsigned int>(m_readable_data.property_value());
		else
			k3d::log() << error << "unsupported property type" << std::endl;

		return 0.0;
	}

	void set_value(const double Value)
	{
		return_if_fail(m_writable_data);

		const std::type_info& type = m_readable_data.property_type();
		if(type == typeid(double))
			m_writable_data->property_set_value(Value);
		else if(type == typeid(float))
			m_writable_data->property_set_value(static_cast<float>(Value));
		else if(type == typeid(long))
			m_writable_data->property_set_value(static_cast<long>(k3d::round(Value)));
		else if(type == typeid(unsigned long))
			m_writable_data->property_set_value(static_cast<unsigned long>(k3d::round(Value)));
		else if(type == typeid(int))
			m_writable_data->property_set_value(static_cast<int>(k3d::round(Value)));
		else if(type == typeid(unsigned int))
			m_writable_data->property_set_value(static_cast<unsigned int>(k3d::round(Value)));
		else
			k3d::log() << error << "Unsupported property type" << std::endl;
	}

	changed_signal_t& changed_signal()
	{
		return m_readable_data.property_changed_signal();
	}

private:
	property_proxy(const property_proxy& RHS);
	property_proxy& operator=(const property_proxy& RHS);
	~property_proxy() {}

	data_t& m_readable_data;
	k3d::iwritable_property* const m_writable_data;
};

} // namespace detail

std::auto_ptr<idata_proxy> proxy(k3d::iproperty& Data, k3d::istate_recorder* const StateRecorder, const Glib::ustring& ChangeMessage)
{
	return std::auto_ptr<idata_proxy>(new detail::property_proxy(Data, StateRecorder, ChangeMessage));
}

/////////////////////////////////////////////////////////////////////////////
// control

control::control(k3d::icommand_node& Parent, const std::string& Name, std::auto_ptr<idata_proxy> Data) :
	base(2, 8, true),
	ui_component(Name, &Parent),
	m_entry(new hotkey_entry()),
	m_button_up(new Gtk::Button()),
	m_button_down(new Gtk::Button()),
	m_data(Data),
	m_step_increment(0.01),
	m_precision(2),
	m_units(&typeid(scalar)),
	m_dragging(false),
	m_tap_started(false)
{
	set_name("k3d-spin-button");

	m_entry->set_name("entry");
	m_entry->set_width_chars(8);
	m_entry->signal_focus_out_event().connect(sigc::mem_fun(*this, &control::on_entry_focus_out_event));
	m_entry->signal_activate().connect(sigc::mem_fun(*this, &control::on_entry_activated));
	attach(*manage(m_entry), 0, 6, 0, 2);

	if(m_data.get() && m_data->writable())
		{
			// Setup up and down buttons
			setup_arrow_button(m_button_up, Gtk::ARROW_UP, true);
			setup_arrow_button(m_button_down, Gtk::ARROW_DOWN, false);

			// Setup VBox containing arrows
			attach(*manage(m_button_up), 6, 7, 0, 1);
			attach(*manage(m_button_down), 6, 7, 1, 2);

			tooltips().set_tip(*m_entry, _("Enter a new value.  Real-world units and simple math expressions are allowed."));
			tooltips().set_tip(*m_button_up, _("LMB-Drag to modify, LMB-Click to step, Tap Shift and Control while dragging to change sensitivity."));
			tooltips().set_tip(*m_button_down, _("LMB-Drag to modify, LMB-Click to step, Tap Shift and Control while dragging to change sensitivity."));

			// Make sure buttons can't get the focus (makes tabbing difficult)
			m_button_up->unset_flags(Gtk::CAN_FOCUS);
			m_button_down->unset_flags(Gtk::CAN_FOCUS);
		}
	else
		{
			m_entry->set_editable(false);
		}

	// Synchronize the view with the data source ...
	data_changed();

	// We want to be notified if the data source changes ...
	if(m_data.get())
		m_data->changed_signal().connect(sigc::mem_fun(*this, &control::data_changed));
}

void control::setup_arrow_button(Gtk::Button* Button, const Gtk::ArrowType ArrowType, const bool Up)
{
	Gtk::Arrow* const arrow = manage(new Gtk::Arrow(ArrowType, Gtk::SHADOW_NONE));
	arrow->set_size_request(0, 0);

	Button->set_size_request(0, 0);
	Button->add(*arrow);

	Button->signal_pressed().connect(sigc::bind(sigc::mem_fun(*this, &control::on_drag_pressed), Up));
	Button->signal_released().connect(sigc::mem_fun(*this, &control::on_drag_released));

	Button->add_events(Gdk::BUTTON_MOTION_MASK | Gdk::KEY_PRESS_MASK | Gdk::KEY_RELEASE_MASK);
	Button->signal_motion_notify_event().connect(sigc::mem_fun(*this, &control::on_drag_motion_notify_event));
	Button->signal_key_press_event().connect(sigc::mem_fun(*this, &control::on_drag_key_press_event));
	Button->signal_key_release_event().connect(sigc::mem_fun(*this, &control::on_drag_key_release_event));
}

void control::set_precision(const unsigned long Precision)
{
	m_precision = Precision;
	data_changed();
}

void control::set_step_increment(const double StepIncrement)
{
	m_step_increment = StepIncrement;
	data_changed();
}

void control::set_units(const std::type_info& Units)
{
	m_units = &Units;
	data_changed();
}

void control::display_value(const double Value)
{
	// Format the value ...
	std::ostringstream buffer;
	buffer << std::fixed << std::setprecision(m_precision);
	k3d::measurement::format(buffer, Value, *m_units);

	// Put it into our edit control ...
	m_entry->set_text(buffer.str());
}

bool control::on_entry_focus_out_event(GdkEventFocus* Event)
{
	set_value();
	return false;
}

void control::on_entry_activated()
{
	set_value();

	// Set focus to the arrows such as hotkeys work again
	m_button_down->set_flags(Gtk::CAN_FOCUS);
	m_button_down->grab_focus();
	m_button_down->unset_flags(Gtk::CAN_FOCUS);
}

void control::set_value()
{
	// Get rid of the selection ...
	m_entry->select_region(0, 0);
	// Get the edit control text ...
	const std::string new_text = m_entry->get_text();
	// Record the command for posterity (tutorials) ...
	record_command("set_value", new_text);

	// Sanity checks ...
	return_if_fail(m_data.get());

	// Default our results to the current value, in case it doesn't parse ...
	double new_value = m_data->value();
	// Parse the input expression into value converting it to SI units automatically (it can do mathematical expressions, too, for fun)
	if(!k3d::measurement::parse(new_text, new_value, m_units))
		{
			k3d::log() << error << "Couldn't parse expression: " << new_text << " restoring original value" << std::endl;
			display_value(new_value);
			return;
		}

	// If the value has changed, record it ...
	if(new_value != m_data->value())
		{
			// Turn this into an undo/redo -able event ...
			if(m_data->state_recorder)
				m_data->state_recorder->start_recording(k3d::create_state_change_set());

			m_data->set_value(new_value);

			// Turn this into an undo/redo -able event ...
			if(m_data->state_recorder)
				m_data->state_recorder->commit_change_set(m_data->state_recorder->stop_recording(), m_data->change_message + ' ' + new_text);
		}
	else
		{
			display_value(new_value);
		}
}

void control::stop_editing(const std::string& NewValue)
{
	// Sanity checks ...
	return_if_fail(m_data.get());

	// Turn this into an undo/redo -able event ...
	if(m_data->state_recorder)
		m_data->state_recorder->commit_change_set(m_data->state_recorder->stop_recording(), m_data->change_message + ' ' + NewValue);
}

void control::data_changed()
{
	return_if_fail(m_data.get());
	display_value(m_data->value());
}

bool control::execute_command(const std::string& Command, const std::string& Arguments)
{
	if(Command == "set_value")
		{
			interactive::set_text(*m_entry, Arguments);
			m_entry->select_region(0, 0);
			set_value();
			return true;
		}

	return ui_component::execute_command(Command, Arguments);
}

void control::on_drag_pressed(const bool Up)
{
	// Sanity checks ...
	return_if_fail(m_data.get());

	// Save which button was pressed
	m_up_button_pressed = Up;

	m_button_up->set_flags(Gtk::CAN_FOCUS);
	m_button_down->set_flags(Gtk::CAN_FOCUS);
	m_button_up->grab_focus();
	m_button_down->grab_focus();

	// Get the current mouse coordinates ...
	int x = 0;
	int y = 0;
	Gdk::ModifierType modifiers;
	Gdk::Display::get_default()->get_pointer(x, y, modifiers);

	m_last_mouse = k3d::vector2(x, y);

	// Get the value of the underlying data at the time dragging began ...
	m_drag_value = m_data->value();
	// Calculate the increment we should use while dragging ...
	m_drag_increment = std::abs(m_step_increment) * 0.2;
	if(!m_drag_increment)
		m_drag_increment = 0.002;

	// Connect idle timeout handler, called every 200ms
	m_drag_timeout = Glib::signal_timeout().connect(sigc::mem_fun(*this, &control::on_drag_timeout), 200);
	m_drag_first_timeout = true;

	// Turn this into an undo/redo -able event ...
	if(m_data->state_recorder)
		m_data->state_recorder->start_recording(k3d::create_state_change_set());
}

bool control::on_drag_motion_notify_event(GdkEventMotion* Event)
{
	// Sanity checks ...
	return_val_if_fail(m_data.get(), false);

	// Get new mouse coordinates
	int x, y;
	Gdk::ModifierType modifiers;
	Gdk::Display::get_default()->get_pointer(x, y, modifiers);
	const k3d::vector2 mouse(x, y);

	// Don't switch to drag mode until the mouse really moved
	if(!m_dragging)
		{
			const double length = (mouse - m_last_mouse).Length();
			if(length < 10)
				return false;
		}

	// We really are dragging, now ...
	m_dragging = true;

	// Update everything ...
	const double horizontal_length = m_last_mouse[0] - mouse[0];
	const double vertical_length = m_last_mouse[1] - mouse[1];
	if(std::abs(horizontal_length) > std::abs(vertical_length))
		{
			// Dragging mostly horizontally : 1/10th unit increase
			m_drag_value += m_drag_increment * 0.1 * (mouse[0] - m_last_mouse[0]);
		}
	else
		{
			// Dragging mostly vertically : one unit increase
			m_drag_value += m_drag_increment * (m_last_mouse[1] - mouse[1]);
		}

	m_data->set_value(m_drag_value);
	m_last_mouse = mouse;

	// Wrap the mouse if it goes off the top-or-bottom of the screen ...
	const int screen_height = Gdk::Display::get_default()->get_default_screen()->get_height();
	const int border = 5;
	if(y < border)
		{
			m_last_mouse = k3d::vector2(mouse[0], screen_height - (border + 1));
			interactive::warp_pointer(m_last_mouse);
		}
	else if(screen_height - y < border)
		{
			m_last_mouse = k3d::vector2(mouse[0], (border + 1));
			interactive::warp_pointer(m_last_mouse);
		}

	// Wrap the mouse if it goes off the left-or-right of the screen ...
	const int screen_width = Gdk::Display::get_default()->get_default_screen()->get_width();
	if(x < border)
		{
			m_last_mouse = k3d::vector2(screen_width - (border + 1), mouse[1]);
			interactive::warp_pointer(m_last_mouse);
		}
	else if(screen_width - x < border)
		{
			m_last_mouse = k3d::vector2((border + 1), mouse[1]);
			interactive::warp_pointer(m_last_mouse);
		}

	return false;
}

void control::on_drag_released()
{
	// Sanity checks ...
	return_if_fail(m_data.get());

	// If the user really didn't drag anywhere and value wasn't changed yet
	if(!m_dragging && m_drag_first_timeout)
		{
			if(m_up_button_pressed)
				increment();
			else
				decrement();
		}

	// Record the command for posterity (tutorials) ...
	record_command("set_value", k3d::string_cast(m_data->value()));

	// Disconnect idle timeout
	m_drag_timeout.disconnect();

	// Turn this into an undo/redo -able event ...
	if(m_data->state_recorder)
		{
			// Format a limited-precision version of the new value, so we we don't create unreadably-long undo-node labels ...
			std::stringstream buffer;
			buffer << std::setprecision(3) << m_data->value();
			m_data->state_recorder->commit_change_set(m_data->state_recorder->stop_recording(), m_data->change_message + ' ' + buffer.str());
		}

	m_button_up->unset_flags(Gtk::CAN_FOCUS);
	m_button_down->unset_flags(Gtk::CAN_FOCUS);
	m_dragging = false;
}

bool control::on_drag_key_press_event(GdkEventKey* Event)
{
	if(!m_tap_started && Event->keyval == GDK_Shift_L || Event->keyval == GDK_Shift_R)
	{
		m_tap_started = true;
		m_drag_increment *= 10.0;
		return true;
	}
	else if(!m_tap_started && Event->keyval == GDK_Control_L || Event->keyval == GDK_Control_R)
	{
		m_tap_started = true;
		m_drag_increment *= 0.1;
		return true;
	}

	return false;
}

bool control::on_drag_key_release_event(GdkEventKey* Event)
{
	m_tap_started = false;
	return false;
}

bool control::on_drag_timeout()
{
	// Step increment if the user doesn't move
	if(!m_dragging)
		{
			if(m_drag_first_timeout)
				{
					// Don't change value on first timeout
					m_drag_first_timeout = false;
					return true;
				}

			if(m_up_button_pressed)
				increment();
			else
				decrement();
		}

	return true;
}

void control::increment()
{
	// Sanity checks ...
	return_if_fail(m_data.get());

	// Update everything ...
	m_data->set_value(m_data->value() + m_step_increment);
}

void control::decrement()
{
	// Sanity checks ...
	return_if_fail(m_data.get());

	// Update everything ...
	m_data->set_value(m_data->value() - m_step_increment);
}

const double control::read_edit_control()
{
	// Sanity checks ...
	return_val_if_fail(m_data.get(), 0.0);

	// Default our results to the current value, in case it doesn't parse ...
	double value = m_data->value();

	// Parse the input expression into value converting it to SI units automatically (it can do mathematical expressions, too, for fun)
	k3d::measurement::parse(m_entry->get_text(), value, m_units);

	return value;
}

} // namespace spin_button

} // namespace libk3dngui


