#ifndef NGUI_TRANSFORM_TOOL_H
#define NGUI_TRANSFORM_TOOL_H

// K-3D
// Copyright (c) 1995-2005, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\author Tim Shead (tshead@k-3d.com)
		\author Romain Behar (romainbehar@yahoo.com)
*/

#include "basic_input_model.h"
#include "navigation_input_model.h"
#include "selection.h"
#include "tool_selection.h"
#include "utility.h"
#include "viewport.h"

#include <k3dsdk/geometry.h>
#include <k3dsdk/gl.h>
#include <k3dsdk/i18n.h>
#include <k3dsdk/icamera.h>
#include <k3dsdk/iprojection.h>
#include <k3dsdk/iselectable.h>
#include <k3dsdk/mesh.h>
#include <k3dsdk/property.h>
#include <k3dsdk/transform.h>

namespace libk3dngui
{

namespace detail
{

/////////////////////////////////////////////////////////////////////////////
// abstract_tool

class abstract_tool
{
public:
	virtual ~abstract_tool() {}

	// Functions executed by transform_tool that rely on transform_tool's derived classes functions
	virtual std::string manipulator_name(const k3d::selection::id ID) = 0;
	typedef std::vector<std::string> manipulators_t;
	virtual std::string get_manipulator(const manipulators_t& Manipulators) = 0;
	virtual void set_manipulator(const std::string ManipulatorName) = 0;
	virtual std::string get_constraint_name() = 0;

	virtual void begin_mouse_move(const k3d::vector2& Coordinates) = 0;

	virtual void update_constraint(viewport::control& Viewport, const k3d::vector2& Coordinates) = 0;
};

typedef struct
{
	unsigned long index;
	k3d::vector3 initial_position;
	k3d::vector3 tweak_value;
} component_point_t;

typedef std::list<component_point_t> component_points_t;

/// Lists selected points in a mesh and returns average component position
k3d::vector3 get_selected_points(selection_mode_t SelectionMode, const k3d::mesh& Mesh, component_points_t& PointList);

} // namespace detail

/////////////////////////////////////////////////////////////////////////////
// transform_tool

class transform_tool :
	public tool_selection,
	public k3d::property_collection
{
	typedef tool_selection base;

public:
	transform_tool(k3d::idocument& Document, document_state& DocumentState, detail::abstract_tool& AbstractTool) :
		base(Document, DocumentState),
		k3d::property_collection(),
		m_document(Document),
		m_document_state(DocumentState),
		m_abstract_tool(AbstractTool),
		m_navigation_model(DocumentState),
		m_current_target(0),
		m_deleted_target(false),
		m_coordinate_system(init_owner(*this) + init_name("coordinate_system") + init_label(_("coordinate_system")) + init_description(_("Coordinate system")) + init_value(COORDINATE_SYSTEM_GLOBAL) + init_enumeration(coordinate_system_values())),
		m_visible_manipulators(init_owner(*this) + init_name("visible_manipulators") + init_label(_("visible_manipulators")) + init_description(_("Visible manipulators")) + init_value(true))
	{
		m_coordinate_system.changed_signal().connect(sigc::mem_fun(*this, &transform_tool::update_coordinate_system));
		m_visible_manipulators.changed_signal().connect(sigc::mem_fun(*this, &transform_tool::redraw_all));

		m_input_model.connect_mbutton_start_drag(sigc::mem_fun(m_navigation_model, &navigation_input_model::on_button1_start_drag));
		m_input_model.connect_mbutton_drag(sigc::mem_fun(m_navigation_model, &navigation_input_model::on_button1_drag));
		m_input_model.connect_mbutton_end_drag(sigc::mem_fun(m_navigation_model, &navigation_input_model::on_button1_end_drag));
		m_input_model.connect_rbutton_start_drag(sigc::mem_fun(m_navigation_model, &navigation_input_model::on_button2_start_drag));
		m_input_model.connect_rbutton_drag(sigc::mem_fun(m_navigation_model, &navigation_input_model::on_button2_drag));
		m_input_model.connect_rbutton_end_drag(sigc::mem_fun(m_navigation_model, &navigation_input_model::on_button2_end_drag));
		m_input_model.connect_scroll(sigc::mem_fun(m_navigation_model, &navigation_input_model::on_scroll));
	}

	~transform_tool()
	{
		clear_targets();
	}

	// Interface required for data containers
	k3d::idocument& document()
	{
		return m_document;
	}

	navigation_input_model& navigation_model()
	{
		return m_navigation_model;
	}

	basic_input_model& input_model()
	{
		return m_input_model;
	}

protected:
	/// Mouse handling functions
	void lbutton_down(viewport::control& Viewport, const k3d::vector2& Coordinates, const k3d::key_modifiers& Modifiers);
	void lmb_down_add();
	void lmb_down_subtract();
	void lmb_down_manipulator(const std::string& ManipulatorName);
	void lmb_down_selected();
	void lmb_down_deselected();
	void lmb_down_nothing();
	void lbutton_click(const viewport::control& Viewport, const k3d::vector2& Coordinates);
	void lmb_click_add();
	void lmb_click_subtract();
	void lmb_click_replace();
	void lmb_click_start_motion(const k3d::vector2& Coordinates);
	void lmb_click_stop_motion();
	void lmb_click_deselect_all();
	void lbutton_start_drag(viewport::control& Viewport, const k3d::vector2& Coordinates);
	void lmb_start_drag_start_motion(const k3d::vector2& Coordinates);
	void lmb_start_drag_box_select(viewport::control& Viewport, const k3d::vector2& Coordinates);
	void lmb_drag_box_select(viewport::control& Viewport, const k3d::vector2& Coordinates);
	void lbutton_end_drag(viewport::control& Viewport, const k3d::vector2& Coordinates);
	void lmb_end_drag_stop_motion();
	void lmb_end_drag_box_select(viewport::control& Viewport, const k3d::vector2& Coordinates);
	void mbutton_click(viewport::control& Viewport, const k3d::vector2& Coordinates, const k3d::key_modifiers& Modifiers);
	void mmb_click_toggle_manipulators_visibility();
	void mmb_click_manipulators_next_selection();
	void mmb_click_switch_coordinate_system();
	void mmb_click_next_constraint(viewport::control& Viewport, const k3d::vector2& Coordinates);
	void rbutton_click(const viewport::control& Viewport, const k3d::vector2& Coordinates);
	void rmb_click_selection_tool();
	void rmb_click_cancel_move();

	void cancel_mouse_move();
	std::string complete_mouse_move();

	// Common functions
/** \todo standardize set_motion in selection tool and transform tools */
	void set_motion(const motion_t Motion);

	k3d::vector3 world_position();
	k3d::matrix4 world_orientation();

	/// Manipulators scale to adjust to viewport size
	double m_manipulators_scale;
	/// Updates manipulators scale to show at constant size after camera update
	void update_manipulators_scale(viewport::control& Viewport, const k3d::vector3& Origin);
	/// Returns whether given direction faces the screen
	bool front_facing(viewport::control& Viewport, const k3d::normal3& Normal, const k3d::vector3& Origin);

	/// Handles off-screen warps, returns true when the mouse warped
	bool off_screen_warp(viewport::control& Viewport, k3d::vector2& NewCoordinates);

	/// Deletes all targets
	void clear_targets();
	/// Mark : one of the target nodes was deleted
	void target_deleted() { m_deleted_target = true; }

	/// Retrieves the current document selection, refreshing the target list
	void get_current_selection();
	/// Returns the number of selected targets (nodes or components depending on selection mode)
	unsigned long target_number();
	/// Updates target list whenever necessary (e.g. : one of them was deleted)
	void update_targets();
	/// Transform targets
	void init_rotation();
	void init_scaling();
	void move_targets(const k3d::normal3& Delta);
	void rotate_targets(const k3d::matrix4& Rotation);
	void scale_targets(const k3d::vector3& Scaling);

private:
	/// Stores an object to be moved interactively
	class itarget
	{
	public:
		itarget() :
			current_system_type(GLOBAL),
			modifier(0)
		{
		}

		virtual ~itarget() {}

		// Returns selected target number
		virtual unsigned long target_number() = 0;

		// Resets move
		virtual void reset() = 0;

		// Update coordinate system
		void set_local() { current_system_type = LOCAL; }
		void set_global() { current_system_type = GLOBAL; }
		void set_parent() { current_system_type = PARENT; }

		// World parameters
		virtual k3d::vector3 world_position() = 0;

		k3d::matrix4 world_orientation()
		{
			if(LOCAL == current_system_type)
				return k3d::extract_rotation(k3d::node_to_world_matrix(*node));

			if(PARENT == current_system_type)
				return k3d::extract_rotation(k3d::parent_to_world_matrix(*node));

			return k3d::identity3D();
		}

		// Modifier
		void set_transform_modifier(k3d::inode* Modifier)
		{
			modifier = Modifier;
			Modifier->deleted_signal().connect(sigc::mem_fun(*this, &itarget::reset_transform_modifier));
		}
		void reset_transform_modifier()
		{
			modifier = 0;
		}

		// Actions
		virtual void init_rotation() = 0;
		virtual void init_scaling() = 0;
		virtual void move(const k3d::normal3& Delta) = 0;
		virtual void rotate(const k3d::matrix4& RotationMatrix) = 0;
		virtual void scale(const k3d::vector3& Delta) = 0;

	protected:
		typedef enum
		{
			GLOBAL,
			LOCAL,
			PARENT
		} system_t;

		/// Stores the transformation type
		system_t current_system_type;
		/// Stores the target node
		k3d::inode* node;
		/// Stores the transformation to be applied
		k3d::inode* modifier;
	};

	class transform_target :
		public itarget
	{
	public:
		transform_target(k3d::inode* Node);

		virtual k3d::vector3 world_position();
		virtual unsigned long target_number();
		virtual void reset();

		virtual void init_rotation();
		virtual void init_scaling();

		virtual void move(const k3d::normal3& Delta);
		virtual void rotate(const k3d::matrix4& RotationMatrix);
		virtual void scale(const k3d::vector3& Delta);

		/// Inserts a transform modifier
		bool create_transform_modifier(const k3d::uuid& Class, const std::string& Name);

	private:
		k3d::matrix4 m_original_rotation;
		k3d::vector3 m_original_scaling;
	};

	class mesh_target :
		public itarget
	{
	public:
		mesh_target(selection_mode_t SelectionMode, k3d::inode* Node, k3d::iproperty& MeshSourceProperty);

		virtual k3d::vector3 world_position();
		virtual unsigned long target_number();
		virtual void reset();

		virtual void init_rotation();
		virtual void init_scaling();

		virtual void move(const k3d::normal3& Delta);
		virtual void rotate(const k3d::matrix4& RotationMatrix);
		virtual void scale(const k3d::vector3& Delta);

		/// Inserts a tweak modifier
		void create_mesh_modifier(const std::string& Name);

	private:
		void move_components(const k3d::normal3& Move);
		void rotate_components(const k3d::matrix4& Rotation);
		void scale_components(const k3d::vector3& Scaling);

		/// Stores the mesh_source property instead of the mesh itself because the k3d::mesh pointer can change
		k3d::iproperty& mesh_source_property;

		/// Stores the average component position
		k3d::vector3 component_center;
		/// Stores the average component position
		k3d::vector3 initial_component_center;
		/// Stores the list of selected points with their index
		detail::component_points_t selected_points;
		/// Stores the tweak array when moving components
		typedef std::vector<k3d::vector3> tweaks_t;
		tweaks_t tweaks;
	};

	/// Defines a collection of objects to be manipulated interactively
	typedef std::vector<itarget*> targets_t;
	/// Stores the current set of objects to be manipulated interactively
	targets_t m_targets;

	/// Enumerates coordinate-system behaviors
	typedef enum
	{
		COORDINATE_SYSTEM_LOCAL,
		COORDINATE_SYSTEM_GLOBAL,
		COORDINATE_SYSTEM_PARENT
	} coordinate_system_t;

	void set_coordinate_system(const coordinate_system_t CoordinateSystem);
	void update_coordinate_system();

protected:
	/// Stores the owning document
	k3d::idocument& m_document;
	document_state& m_document_state;
	/// Stores a reference to the derived tool
	detail::abstract_tool& m_abstract_tool;
	/// Provides interactive navigation behavior
	navigation_input_model m_navigation_model;
	/// Dispatches incoming user input events
	basic_input_model m_input_model;

	/// Stores manipulators size
	double m_manipulators_size;

	/// Stores the target number used to provide local coordinates
	unsigned long m_current_target;
	/// Set to true when one of the targets is deleted
	bool m_deleted_target;

	/// Defines coordinate system enumeration property
	friend std::ostream& operator << (std::ostream& Stream, const coordinate_system_t& Value)
	{
		switch(Value)
		{
			case COORDINATE_SYSTEM_LOCAL:
				Stream << "coordinate_system_local";
				break;
			case COORDINATE_SYSTEM_GLOBAL:
				Stream << "coordinate_system_global";
				break;
			case COORDINATE_SYSTEM_PARENT:
				Stream << "coordinate_system_parent";
				break;
		}
		return Stream;
	}

	friend std::istream& operator >> (std::istream& Stream, coordinate_system_t& Value)
	{
		std::string text;
		Stream >> text;

		if(text == "coordinate_system_local")
			Value = COORDINATE_SYSTEM_LOCAL;
		else if(text == "coordinate_system_global")
			Value = COORDINATE_SYSTEM_GLOBAL;
		else if(text == "coordinate_system_parent")
			Value = COORDINATE_SYSTEM_PARENT;
		else
			k3d::log() << __PRETTY_FUNCTION__ << ": unknown enumeration [" << text << "]" << std::endl;

		return Stream;
	}

	const k3d::ienumeration_property::enumeration_values_t& coordinate_system_values()
	{
		static k3d::ienumeration_property::enumeration_values_t values;
		if(values.empty())
		{
			values.push_back(k3d::ienumeration_property::enumeration_value_t("Local", "coordinate_system_local", "Moves objects in local coordinate system"));
			values.push_back(k3d::ienumeration_property::enumeration_value_t("Global", "coordinate_system_global", "Moves objects in global coordinate system"));
			values.push_back(k3d::ienumeration_property::enumeration_value_t("Parent", "coordinate_system_parent", "Moves objects in parent's coordinate system"));
		}

		return values;
	}

	/// Stores the current coordinate system behavior
	k3d_data(coordinate_system_t, immutable_name, change_signal, with_undo, local_storage, no_constraint, enumeration_property, no_serialization) m_coordinate_system;

	/// Stores the enabled/disabled state for the manipulators
	k3d_data(bool, immutable_name, change_signal, with_undo, local_storage, no_constraint, writable_property, no_serialization) m_visible_manipulators;
};

} // namespace libk3dngui

#endif // !NGUI_TRANSFORM_TOOL_H

