// K-3D
// Copyright (c) 1995-2006, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\author Tim Shead (tshead@k-3d.com)
*/

#include "fstream.h"
#include "i18n.h"
#include "irenderman.h"
#include "iuser_property.h"
#include "options.h"
#include "property.h"
#include "renderman_shader.h"
#include "user_properties.h"

namespace k3d
{

namespace ri
{

namespace detail
{

class pause_undo_recording
{
public:
	pause_undo_recording(idocument& Document) :
		m_document(Document)
	{
		if(m_document.state_recorder().current_change_set())
			m_change_set = m_document.state_recorder().stop_recording();
	}

	~pause_undo_recording()
	{
		if(m_change_set.get())
			m_document.state_recorder().start_recording(m_change_set);
	}

private:
	idocument& m_document;
	std::auto_ptr<state_change_set> m_change_set;
};

const std::string shader_type_path(const k3d::sl::shader::type_t ShaderType)
{
	switch(ShaderType)
	{
		case k3d::sl::shader::SURFACE:
			return k3d::options::path::surface_shaders();
		case k3d::sl::shader::LIGHT:
			return k3d::options::path::light_shaders();
		case k3d::sl::shader::VOLUME:
			return k3d::options::path::volume_shaders();
		case k3d::sl::shader::DISPLACEMENT:
			return k3d::options::path::displacement_shaders();
		case k3d::sl::shader::TRANSFORMATION:
			return k3d::options::path::transformation_shaders();
		case k3d::sl::shader::IMAGER:
			return k3d::options::path::imager_shaders();
	}

	assert_not_reached();
	return std::string();
}

} // namespace detail

/////////////////////////////////////////////////////////////////////////////
// shader

shader::shader(idocument& Document, k3d::sl::shader::type_t ShaderType) :
	base(Document),
	m_shader_type(ShaderType),
	m_shader_path(init_owner(*this) + init_name("shader_path") + init_label(_("Shader Path")) + init_description(_("Shader Path")) + init_value<boost::filesystem::path>("") + init_path_mode(k3d::ipath_property::READ) + init_path_type(detail::shader_type_path(ShaderType))),
	m_shader(0)
{
	m_shader_path.add_pattern_filter(ipath_property::pattern_filter(_("RenderMan shader (*.sl)"), "*.sl"));
	m_shader_connection = m_shader_path.changed_signal().connect(sigc::mem_fun(*this, &shader::on_shader_changed));
}

shader::~shader()
{
	delete m_shader;
}

void shader::load(xml::element& Element, const ipersistent::load_context& Context)
{
	// Disable argument list updates while loading so we don't create the same arguments twice
	m_shader_connection.disconnect();
	base::load(Element, Context);
	m_shader_connection = m_shader_path.changed_signal().connect(sigc::mem_fun(*this, &shader::on_shader_changed));

	load_metafile();
}

const path shader::shader_path()
{
	return m_shader_path.value();
}

const std::string shader::shader_name()
{
	if(m_shader)
		return m_shader->name;

	return "null";
}

parameter_list shader::shader_arguments(const render_state& State)
{
	parameter_list results;

	const iproperty_collection::properties_t& properties = base::properties();
	for(iproperty_collection::properties_t::const_iterator prop = properties.begin(); prop != properties.end(); ++prop)
	{
		if(dynamic_cast<iuser_property*>(*prop))
		{
			iproperty& property = **prop;
			const std::type_info& property_type = property.property_type();

			if(property_type == typeid(ri::real))
			{
				results.push_back(parameter(property.property_name(), CONSTANT, boost::any_cast<ri::real>(get_value(document().dag(), property))));
			}
			else if(property_type == typeid(ri::string))
			{
				results.push_back(parameter(property.property_name(), CONSTANT, boost::any_cast<ri::string>(get_value(document().dag(), property))));
			}
			else if(property_type == typeid(k3d::inode*)) // Node properties always are of type "inode*", so we have to query for the interface type we really want
			{
				if(k3d::ri::itexture* const texture = dynamic_cast<k3d::ri::itexture*>(boost::any_cast<k3d::inode*>(get_value(document().dag(), property))))
					results.push_back(parameter(property.property_name(), CONSTANT, static_cast<k3d::ri::string>(texture->renderman_texture_path(State).native_file_string())));
			}
			else if(property_type == typeid(ri::point))
			{
				results.push_back(parameter(property.property_name(), CONSTANT, boost::any_cast<ri::point>(get_value(document().dag(), property))));
			}
			else if(property_type == typeid(ri::vector))
			{
				results.push_back(parameter(property.property_name(), CONSTANT, boost::any_cast<ri::vector>(get_value(document().dag(), property))));
			}
			else if(property_type == typeid(ri::normal))
			{
				results.push_back(parameter(property.property_name(), CONSTANT, boost::any_cast<ri::normal>(get_value(document().dag(), property))));
			}
			else if(property_type == typeid(ri::hpoint))
			{
				results.push_back(parameter(property.property_name(), CONSTANT, boost::any_cast<ri::hpoint>(get_value(document().dag(), property))));
			}
			else if(property_type == typeid(ri::matrix))
			{
				results.push_back(parameter(property.property_name(), CONSTANT, boost::any_cast<ri::matrix>(get_value(document().dag(), property))));
			}
			else if(property_type == typeid(ri::color))
			{
				results.push_back(parameter(property.property_name(), CONSTANT, boost::any_cast<ri::color>(get_value(document().dag(), property))));
			}
			else
			{
				log() << error << k3d_file_reference << ": unknown storage type [" << property_type.name() << "] for shader argument [" << property.property_name() << "] will be ignored" << std::endl;
			}
		}
	}

	return results;
}

void shader::on_shader_changed(iunknown*)
{
	load_metafile();
	clear_arguments();
	setup_arguments();
}

void shader::load_metafile()
{
	// Try to load a shader description metafile
	const boost::filesystem::path shader_path = m_shader_path.value();
	const boost::filesystem::path metafile_path = boost::filesystem::path(shader_path.native_file_string() + ".slmeta", boost::filesystem::native);
	k3d::filesystem::ifstream metafile_stream(metafile_path);

	const sl::shaders_t shaders = sl::parse_metafile(metafile_stream, shader_path.native_file_string(), metafile_path.native_file_string());
	if(shaders.size() != 1)
	{
		log() << error << "Can't load metafile describing shader [" << shader_path.native_file_string() << "]" << std::endl;
		return;
	}
	const sl::shaders_t::const_iterator shader = shaders.begin();
	if(shader->type != m_shader_type)
	{
		log() << error << "Shader [" << shader_path.native_file_string() << "] is not the correct shader type" << std::endl;
		return;
	}

	m_shader = new sl::shader(*shader);
}

void shader::clear_arguments()
{
	const iproperty_collection::properties_t properties = base::properties();
	for(iproperty_collection::properties_t::const_iterator property = properties.begin(); property != properties.end(); ++property)
	{
		if(dynamic_cast<iuser_property*>(*property))
		{
			if(ipersistent* const persistent = dynamic_cast<ipersistent*>(*property))
				disable_serialization(*persistent);

			unregister_property(**property);
			delete dynamic_cast<ideletable*>(*property);
		}
	}
}

void shader::setup_arguments()
{
	return_if_fail(m_shader);

	// At the moment, we don't support undo-able changes to the argument-set, so temporarily disable any recording ...
	detail::pause_undo_recording pause(document());

	for(sl::shader::arguments_t::const_iterator argument = m_shader->arguments.begin(); argument != m_shader->arguments.end(); ++argument)
	{
		if(argument->output)
			continue;

		switch(argument->extended_type)
		{
			case sl::argument::EX_FLOAT:
			case sl::argument::EX_TIME:
			case sl::argument::EX_ANGLE:
			case sl::argument::EX_DISTANCE:
			case sl::argument::EX_AREA:
			case sl::argument::EX_VOLUME:
			case sl::argument::EX_MASS:
			case sl::argument::EX_FORCE:
			case sl::argument::EX_PRESSURE:
			{
				new user::double_property(
					init_owner(*this)
					+ init_name(argument->name.c_str())
					+ init_label(argument->label.c_str())
					+ init_description(argument->description.c_str())
					+ init_value(from_string<double>(argument->default_value, 0.0)));
				break;
			}
			case sl::argument::EX_STRING:
			case sl::argument::EX_SPACE:
			{
				new user::string_property(
					init_owner(*this)
					+ init_name(argument->name.c_str())
					+ init_label(argument->label.c_str())
					+ init_description(argument->description.c_str())
					+ init_value(argument->default_value));
				break;
			}
			case sl::argument::EX_TEXTURE:
			{
				new user::ri::texture_property(
					init_owner(*this)
					+ init_name(argument->name.c_str())
					+ init_label(argument->label.c_str())
					+ init_description(argument->description.c_str())
					+ init_value(static_cast<k3d::ri::itexture*>(0)));
				break;
			}
			case sl::argument::EX_POINT:
			{
				new user::point3_property(
					init_owner(*this)
					+ init_name(argument->name.c_str())
					+ init_label(argument->label.c_str())
					+ init_description(argument->description.c_str())
					+ init_value(from_string<point3>(argument->default_value, point3(0, 0, 0))));
				break;
			}
			case sl::argument::EX_VECTOR:
			{
				new user::vector3_property(
					init_owner(*this)
					+ init_name(argument->name.c_str())
					+ init_label(argument->label.c_str())
					+ init_description(argument->description.c_str())
					+ init_value(from_string<vector3>(argument->default_value, vector3(0, 0, 0))));
				break;
			}
			case sl::argument::EX_NORMAL:
			{
				new user::normal3_property(
					init_owner(*this)
					+ init_name(argument->name.c_str())
					+ init_label(argument->label.c_str())
					+ init_description(argument->description.c_str())
					+ init_value(from_string<normal3>(argument->default_value, normal3(0, 0, 0))));
				break;
			}
			case sl::argument::EX_HPOINT:
			{
				new user::point4_property(
					init_owner(*this)
					+ init_name(argument->name.c_str())
					+ init_label(argument->label.c_str())
					+ init_description(argument->description.c_str())
					+ init_value(from_string<point4>(argument->default_value, point4(0, 0, 0, 0))));
				break;
			}
			case sl::argument::EX_MATRIX:
			{
				new user::matrix4_property(
					init_owner(*this)
					+ init_name(argument->name.c_str())
					+ init_label(argument->label.c_str())
					+ init_description(argument->description.c_str())
					+ init_value(from_string<matrix4>(argument->default_value, matrix4())));
				break;
			}
			case sl::argument::EX_COLOR:
			{
				new user::color_property(
					init_owner(*this)
					+ init_name(argument->name.c_str())
					+ init_label(argument->label.c_str())
					+ init_description(argument->description.c_str())
					+ init_value(from_string<color>(argument->default_value, color(1, 1, 1))));
				break;
			}
			default:
			{
				log() << error << k3d_file_reference << " unknown extended argument type for [" << argument->name << "] will not receive storage" << std::endl;
			}
		}
	}
}

} // namespace ri

} // namespace k3d
