// K-3D
// Copyright (c) 2005-2006, Romain Behar
//
// Contact: romainbehar@yahoo.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\author Romain Behar (romainbehar@yahoo.com)
*/

#include <k3dsdk/gl.h>
#include <k3dsdk/i18n.h>
#include <k3dsdk/measurement.h>
#include <k3dsdk/mesh.h>
#include <k3dsdk/mesh_modifier.h>
#include <k3dsdk/module.h>
#include <k3dsdk/node.h>
#include <k3dsdk/persistent.h>
#include <k3dsdk/selection.h>
#include <k3dsdk/string_cast.h>
#include <k3dsdk/transformable.h>

#include "helpers.h"

namespace libk3dmesh
{

namespace detail
{

struct show_numbering
{
	show_numbering(const unsigned long Start, const unsigned long End, const k3d::color& Color, const k3d::gl::render_state& State) :
		start(Start),
		end(End),
		render_state(State),
		component_counter(0)
	{
		k3d::gl::color3d(Color);
	}

	void operator()(k3d::point& Point)
	{
		if(component_counter >= start && component_counter <= end)
		{
			const std::string text = k3d::string_cast(component_counter);

			const k3d::point3 position = Point.position;

			glRasterPos3d(position[0], position[1], position[2]);
			glListBase(render_state.gl_ascii_font_list_base);
			glCallLists(text.size(), GL_UNSIGNED_BYTE, text.c_str());
		}

		++component_counter;
	}

	void operator()(k3d::polyhedron& Polyhedron)
	{
	}

	void operator()(k3d::face& Face)
	{
		if(component_counter >= start && component_counter <= end)
		{
			const std::string text = k3d::string_cast(component_counter);

			const k3d::point3 position = helpers::center_point(Face);

			glRasterPos3d(position[0], position[1], position[2]);
			glListBase(render_state.gl_ascii_font_list_base);
			glCallLists(text.size(), GL_UNSIGNED_BYTE, text.c_str());
		}

		++component_counter;
	}

	const unsigned long start;
	const unsigned long end;
	const k3d::gl::render_state& render_state;

	unsigned long component_counter;
};

void annotate_edge(k3d::split_edge& Edge, const unsigned long Number, const k3d::gl::render_state& State)
{
	const std::string text = k3d::string_cast(Number);

	k3d::point3 position = (Edge.face_clockwise->vertex->position + Edge.vertex->position) * 0.5;

	if(Edge.companion)
		position = Edge.vertex->position + (1.0/3.0) * (Edge.face_clockwise->vertex->position - Edge.vertex->position);

	glRasterPos3d(position[0], position[1], position[2]);
	glListBase(State.gl_ascii_font_list_base);
	glCallLists(text.size(), GL_UNSIGNED_BYTE, text.c_str());
}

} // namespace detail

/////////////////////////////////////////////////////////////////////////////
// show_component_numbering

class show_component_numbering :
	public k3d::gl::drawable<k3d::transformable<k3d::mesh_modifier<k3d::persistent<k3d::node> > > >
{
	typedef k3d::gl::drawable<k3d::transformable<k3d::mesh_modifier<k3d::persistent<k3d::node> > > > base;

public:
	show_component_numbering(k3d::idocument& Document) :
		base(Document),
		m_vertex_text_color(init_owner(*this) + init_name("vertex_color") + init_label(_("Vertex color")) + init_description(_("Numbering color for vertices")) + init_value(k3d::color(0, 0, 1))),
		m_edge_text_color(init_owner(*this) + init_name("edge_color") + init_label(_("Edge color")) + init_description(_("Numbering color for edges")) + init_value(k3d::color(0, 1, 0))),
		m_hole_edge_text_color(init_owner(*this) + init_name("hole_edge_color") + init_label(_("Hole edge color")) + init_description(_("Numbering color for hole edges")) + init_value(k3d::color(0, 0.5, 0))),
		m_face_text_color(init_owner(*this) + init_name("face_color") + init_label(_("Face color")) + init_description(_("Numbering color for faces")) + init_value(k3d::color(1, 0, 0))),
		m_vertex_numbering_start(init_owner(*this) + init_name("first_vertex_number") + init_label(_("First vertex number")) + init_description(_("First vertex number shown")) + init_value(0L) + init_precision(0) + init_step_increment(1) + init_units(typeid(k3d::measurement::scalar))),
		m_vertex_numbering_end(init_owner(*this) + init_name("last_vertex_number") + init_label(_("Last vertex number")) + init_description(_("Last vertex number shown")) + init_value(1000000L) + init_precision(0) + init_step_increment(1) + init_units(typeid(k3d::measurement::scalar))),
		m_edge_numbering_start(init_owner(*this) + init_name("first_edge_number") + init_label(_("First edge number")) + init_description(_("First edge number shown")) + init_value(0L) + init_precision(0) + init_step_increment(1) + init_units(typeid(k3d::measurement::scalar))),
		m_edge_numbering_end(init_owner(*this) + init_name("last_edge_number") + init_label(_("Last edge number")) + init_description(_("Last edge number shown")) + init_value(1000000L) + init_precision(0) + init_step_increment(1) + init_units(typeid(k3d::measurement::scalar))),
		m_face_numbering_start(init_owner(*this) + init_name("first_face_number") + init_label(_("First face number")) + init_description(_("First face number shown")) + init_value(0L) + init_precision(0) + init_step_increment(1) + init_units(typeid(k3d::measurement::scalar))),
		m_face_numbering_end(init_owner(*this) + init_name("last_face_number") + init_label(_("Last face number")) + init_description(_("Last face number shown")) + init_value(1000000L) + init_precision(0) + init_step_increment(1) + init_units(typeid(k3d::measurement::scalar)))
	{
		m_vertex_text_color.changed_signal().connect(make_async_redraw_slot());
		m_edge_text_color.changed_signal().connect(make_async_redraw_slot());
		m_hole_edge_text_color.changed_signal().connect(make_async_redraw_slot());
		m_face_text_color.changed_signal().connect(make_async_redraw_slot());
		m_vertex_numbering_start.changed_signal().connect(make_async_redraw_slot());
		m_vertex_numbering_end.changed_signal().connect(make_async_redraw_slot());
		m_edge_numbering_start.changed_signal().connect(make_async_redraw_slot());
		m_edge_numbering_end.changed_signal().connect(make_async_redraw_slot());
		m_face_numbering_start.changed_signal().connect(make_async_redraw_slot());
		m_face_numbering_end.changed_signal().connect(make_async_redraw_slot());
	}

	/** \todo Improve the implementation so we don't have to do this */
	k3d::iunknown* on_rewrite_hint(iunknown* const Hint)
	{
		// Force updates to re-allocate our mesh, for simplicity
		return 0;
	}

	void on_create_mesh(const k3d::mesh& InputMesh, k3d::mesh& Mesh)
	{
		k3d::deep_copy(InputMesh, Mesh);
	}

	void on_update_mesh(const k3d::mesh& InputMesh, k3d::mesh& Mesh)
	{
	}

	void on_gl_draw(const k3d::gl::render_state& State)
	{
		k3d::gl::color3d(k3d::color(1, 1, 1));
		draw(State);
	}

	void on_gl_select(const k3d::gl::render_state& State, const k3d::gl::select_state& SelectState)
	{
		k3d::gl::push_selection_token(this);
		draw(State);
		k3d::gl::pop_selection_token();
	}

	void draw(const k3d::gl::render_state& State)
	{
		glDisable(GL_LIGHTING);
		glDisable(GL_TEXTURE_1D);
		glDisable(GL_TEXTURE_2D);
		glDisable(GL_BLEND);

		// Get input mesh
		k3d::mesh* const input = m_input_mesh.value();
		if(!input)
			return;

		// Update start/end values
		if(m_vertex_numbering_end.value() < m_vertex_numbering_start.value())
			m_vertex_numbering_end.set_value(m_vertex_numbering_start.value());
		if(m_edge_numbering_end.value() < m_edge_numbering_start.value())
			m_edge_numbering_end.set_value(m_edge_numbering_start.value());
		if(m_face_numbering_end.value() < m_face_numbering_start.value())
			m_face_numbering_end.set_value(m_face_numbering_start.value());

		// Process vertices and faces
		k3d::for_each_point(*input, detail::show_numbering(m_vertex_numbering_start.value(), m_vertex_numbering_end.value(), m_vertex_text_color.value(), State));
		k3d::for_each_face(*input, detail::show_numbering(m_face_numbering_start.value(), m_face_numbering_end.value(), m_face_text_color.value(), State));

		// Distinguish face edges and hole edges
		const k3d::color edge_color = m_edge_text_color.value();
		const k3d::color hole_edge_color = m_hole_edge_text_color.value();
		k3d::gl::color3d(edge_color);

		const unsigned long edge_start = m_edge_numbering_start.value();
		const unsigned long edge_end = m_edge_numbering_end.value();

		unsigned long edge_counter = 0;
		for(k3d::mesh::polyhedra_t::iterator polyhedron = input->polyhedra.begin(); polyhedron != input->polyhedra.end(); ++polyhedron)
		{

			for(k3d::polyhedron::faces_t::const_iterator face = (*polyhedron)->faces.begin(); face != (*polyhedron)->faces.end(); ++face)
			{
				for(k3d::split_edge* edge = (*face)->first_edge; edge; edge = edge->face_clockwise)
				{
					if(edge_counter >= edge_start && edge_counter <= edge_end)
						detail::annotate_edge(*edge, edge_counter, State);

					++edge_counter;

					if(edge->face_clockwise == (*face)->first_edge)
						break;
				}

				if(!(*face)->holes.size())
					continue;

				k3d::gl::color3d(hole_edge_color);

				// Face holes
				unsigned long hole_edge_counter = 0;
				for(k3d::face::holes_t::iterator hole = (*face)->holes.begin(); hole != (*face)->holes.end(); ++hole)
				{
					for(k3d::split_edge* edge = *hole; edge; edge = edge->face_clockwise)
					{
						detail::annotate_edge(*edge, hole_edge_counter, State);
						++hole_edge_counter;

						if(edge->face_clockwise == (*hole))
							break;
					}
				}

				k3d::gl::color3d(edge_color);
			}
		}
	}

	k3d::iplugin_factory& factory()
	{
		return get_factory();
	}

	static k3d::iplugin_factory& get_factory()
	{
		static k3d::plugin_factory<
			k3d::document_plugin<show_component_numbering>,
			k3d::interface_list<k3d::imesh_source,
			k3d::interface_list<k3d::imesh_sink > > > factory(
				k3d::uuid(0x9bed6d9a, 0x790a4637, 0x88de6e16, 0xd04e1068),
				"ShowComponentNumbering",
				"Shows internal structure numbering for points, edges and faces",
				"Utility",
				k3d::iplugin_factory::STABLE);

		return factory;
	}

private:
	k3d_data(k3d::color, immutable_name, change_signal, with_undo, local_storage, no_constraint, writable_property, with_serialization) m_vertex_text_color;
	k3d_data(k3d::color, immutable_name, change_signal, with_undo, local_storage, no_constraint, writable_property, with_serialization) m_edge_text_color;
	k3d_data(k3d::color, immutable_name, change_signal, with_undo, local_storage, no_constraint, writable_property, with_serialization) m_hole_edge_text_color;
	k3d_data(k3d::color, immutable_name, change_signal, with_undo, local_storage, no_constraint, writable_property, with_serialization) m_face_text_color;
	k3d_data(long, immutable_name, change_signal, with_undo, local_storage, no_constraint, measurement_property, with_serialization) m_vertex_numbering_start;
	k3d_data(long, immutable_name, change_signal, with_undo, local_storage, no_constraint, measurement_property, with_serialization) m_vertex_numbering_end;
	k3d_data(long, immutable_name, change_signal, with_undo, local_storage, no_constraint, measurement_property, with_serialization) m_edge_numbering_start;
	k3d_data(long, immutable_name, change_signal, with_undo, local_storage, no_constraint, measurement_property, with_serialization) m_edge_numbering_end;
	k3d_data(long, immutable_name, change_signal, with_undo, local_storage, no_constraint, measurement_property, with_serialization) m_face_numbering_start;
	k3d_data(long, immutable_name, change_signal, with_undo, local_storage, no_constraint, measurement_property, with_serialization) m_face_numbering_end;
};

/////////////////////////////////////////////////////////////////////////////
// show_component_numbering_factory

k3d::iplugin_factory& show_component_numbering_factory()
{
	return show_component_numbering::get_factory();
}

} // namespace libk3dmesh

