// K-3D
// Copyright (c) 1995-2006, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\author Tim Shead (tshead@k-3d.com)
*/

#include "file_chooser_dialog.h"
#include "messages.h"

#include <k3dsdk/i18n.h>
#include <k3dsdk/options.h>
#include <k3dsdk/result.h>
#include <k3dsdk/share.h>
#include <k3dsdk/system.h>

#include <boost/filesystem/convenience.hpp>
#include <boost/filesystem/operations.hpp>
#include <boost/filesystem/path.hpp>

#include <gtkmm/checkbutton.h>
#include <gtkmm/filechooserdialog.h>
#include <gtkmm/stock.h>

namespace libk3dngui
{

/////////////////////////////////////////////////////////////////////////////
// file_chooser_dialog

file_chooser_dialog::file_chooser_dialog(const Glib::ustring& Title, const std::string& PathType, Gtk::FileChooserAction Action, const boost::filesystem::path& InitialPath) :
	base(Title, Action),
	path_type(PathType),
	initial_path(InitialPath),
	compression(false),
	extra_widget_container(new Gtk::VBox(false, 0))
{
	base::set_extra_widget(*extra_widget_container);

	set_select_multiple(false);
	set_local_only();
}

file_chooser_dialog::file_chooser_dialog(const Glib::ustring& Title, const std::string& PathType, k3d::ipath_property::mode_t Action, const boost::filesystem::path& InitialPath) :
	base(Title),
	path_type(PathType),
	initial_path(InitialPath),
	compression(false),
	extra_widget_container(new Gtk::VBox(false, 0))
{
	base::set_extra_widget(*extra_widget_container);

	set_select_multiple(false);
	set_local_only();

	switch(Action)
	{
		case k3d::ipath_property::READ:
			set_action(Gtk::FILE_CHOOSER_ACTION_OPEN);
			break;
		case k3d::ipath_property::WRITE:
			set_action(Gtk::FILE_CHOOSER_ACTION_SAVE);
			break;
	}
}

file_chooser_dialog::file_chooser_dialog(Gtk::Window& Parent, const Glib::ustring& Title, const std::string& PathType, Gtk::FileChooserAction Action, const boost::filesystem::path& InitialPath) :
	base(Parent, Title, Action),
	path_type(PathType),
	initial_path(InitialPath),
	compression(false),
	extra_widget_container(new Gtk::VBox(false, 0))
{
	base::set_extra_widget(*extra_widget_container);

	set_select_multiple(false);
	set_local_only();
}

file_chooser_dialog::file_chooser_dialog(Gtk::Window& Parent, const Glib::ustring& Title, const std::string& PathType, k3d::ipath_property::mode_t Action, const boost::filesystem::path& InitialPath) :
	base(Parent, Title),
	path_type(PathType),
	initial_path(InitialPath),
	compression(false),
	extra_widget_container(new Gtk::VBox(false, 0))
{
	base::set_extra_widget(*extra_widget_container);

	set_select_multiple(false);
	set_local_only();

	switch(Action)
	{
		case k3d::ipath_property::READ:
			set_action(Gtk::FILE_CHOOSER_ACTION_OPEN);
			break;
		case k3d::ipath_property::WRITE:
			set_action(Gtk::FILE_CHOOSER_ACTION_SAVE);
			break;
	}
}

file_chooser_dialog::~file_chooser_dialog()
{
	delete extra_widget_container;
}

void file_chooser_dialog::add_pattern_filter(const Glib::ustring& Name, const Glib::ustring& Pattern)
{
	Gtk::FileFilter filter;
	filter.set_name(Name);
	filter.add_pattern(Pattern);
	add_filter(filter);
}

void file_chooser_dialog::add_mime_type_filter(const Glib::ustring& Name, const Glib::ustring& MimeType)
{
	Gtk::FileFilter filter;
	filter.set_name(Name);
	filter.add_mime_type(MimeType);
	add_filter(filter);
}

void file_chooser_dialog::add_all_files_filter()
{
	Gtk::FileFilter filter;
	filter.set_name(_("All Files"));
	filter.add_pattern("*");
	add_filter(filter);
}

void file_chooser_dialog::enable_compression(const bool Compression)
{
	compression = Compression;
}

void file_chooser_dialog::append_extension(const std::string& Extension)
{
	extension = Extension;
}

void file_chooser_dialog::add_extra_widget(Gtk::Widget& ExtraWidget)
{
	extra_widget_container->pack_start(ExtraWidget, Gtk::PACK_SHRINK, 0);
}

bool file_chooser_dialog::get_file_path(boost::filesystem::path& Result)
{
	bool compressed = false;
	return get_file_path(Result, compressed);
}

bool file_chooser_dialog::get_file_path(boost::filesystem::path& Result, bool& Compressed)
{
	// Create buttons that will be (optionally) added to the dialog.  We do this on the stack
	// so the same buttons don't get added to the widget multiple times if get_file_path() is
	// called more-than-once.
	
	Gtk::CheckButton compression_widget(_("Compress output file"));
	compression_widget.set_active(Compressed);
	
	Gtk::CheckButton append_extension_widget(_("Append filename extension automatically"));
	append_extension_widget.set_active(true);

	Gtk::Button cancel_widget(Gtk::Stock::CANCEL);
	cancel_widget.set_flags(cancel_widget.get_flags() | Gtk::CAN_DEFAULT);
	cancel_widget.show();
	
	Gtk::Button open_widget(Gtk::Stock::OPEN);
	open_widget.set_flags(open_widget.get_flags() | Gtk::CAN_DEFAULT);
	open_widget.show();

	Gtk::Button save_widget(Gtk::Stock::SAVE);
	save_widget.set_flags(save_widget.get_flags() | Gtk::CAN_DEFAULT);
	save_widget.show();
	
	// Add the K-3D share path as a shortcut ...	
	add_shortcut_folder(k3d::share_path().native_file_string());

	add_action_widget(cancel_widget, Gtk::RESPONSE_CANCEL);
	switch(get_action())
	{
		case Gtk::FILE_CHOOSER_ACTION_OPEN:
		case Gtk::FILE_CHOOSER_ACTION_SELECT_FOLDER:
			add_action_widget(open_widget, Gtk::RESPONSE_OK);
			set_default_response(Gtk::RESPONSE_OK);
			break;
		case Gtk::FILE_CHOOSER_ACTION_SAVE:
			// Setup the (optional) automatic extension control ...
			if(!extension.empty())
				extra_widget_container->pack_end(append_extension_widget, Gtk::PACK_SHRINK, 0);
			
			// Setup the (optional) compression control ...
			if(compression)
				extra_widget_container->pack_end(compression_widget, Gtk::PACK_SHRINK, 0);
			
			add_action_widget(save_widget, Gtk::RESPONSE_OK);
			set_default_response(Gtk::RESPONSE_OK);
			break;
		case Gtk::FILE_CHOOSER_ACTION_CREATE_FOLDER:
			add_action_widget(save_widget, Gtk::RESPONSE_OK);
			set_default_response(Gtk::RESPONSE_OK);
			break;
	}

	// Setup the initial path to display ...
	boost::filesystem::path start_path = initial_path;
	if(start_path.empty())
		start_path = k3d::options::get_path(path_type);

	if(start_path.empty())
		start_path = k3d::system::get_home_directory();

	if(boost::filesystem::exists(start_path) && boost::filesystem::is_directory(start_path))
		set_current_folder(start_path.native_file_string());
	else
		set_filename(start_path.native_file_string());

	// Run the dialog ...
	set_position(Gtk::WIN_POS_CENTER);
	extra_widget_container->show_all();
	if(Gtk::RESPONSE_OK != run())
		return false;

	// Optionally add an extension if the user didn't ...
	Result = boost::filesystem::path(get_filename(), boost::filesystem::native);
	if(!extension.empty() && append_extension_widget.get_active() && (boost::filesystem::extension(Result) != extension))
		Result = boost::filesystem::path(Result.native_file_string() + extension, boost::filesystem::native);

	// Prompt the user if they're about to overwrite an existing file ...
	if(!prompt_file_overwrite(Result))
		return false;

	// Record the path for posterity ...
	k3d::options::set_path(path_type, Result);

	switch(get_action())
	{
		case Gtk::FILE_CHOOSER_ACTION_OPEN:
		case Gtk::FILE_CHOOSER_ACTION_SELECT_FOLDER:
			break;
		case Gtk::FILE_CHOOSER_ACTION_SAVE:
			if(compression)
				Compressed = compression_widget.get_active();
			break;
		case Gtk::FILE_CHOOSER_ACTION_CREATE_FOLDER:
			break;
	}

	return true;
}

bool file_chooser_dialog::prompt_file_overwrite(const boost::filesystem::path& File)
{
	switch(get_action())
	{
		case Gtk::FILE_CHOOSER_ACTION_OPEN:
		case Gtk::FILE_CHOOSER_ACTION_SELECT_FOLDER:
			break;
		case Gtk::FILE_CHOOSER_ACTION_SAVE:
		case Gtk::FILE_CHOOSER_ACTION_CREATE_FOLDER:
			if(boost::filesystem::exists(File))
			{
				std::vector<std::string> buttons;
				buttons.push_back(_("Yes"));
				buttons.push_back(_("No"));
				if(1 != query_message("Overwrite " + File.native_file_string() + "?", _("Overwrite File:"), 1, buttons))
					return false;
			}
			break;
	}

	return true;
}

} // namespace libk3dngui

