// K-3D
// Copyright (c) 1995-2005, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\author Timothy M. Shead (tshead@k-3d.com)
*/

#include <k3dsdk/axis.h>
#include <k3dsdk/i18n.h>
#include <k3dsdk/imaterial.h>
#include <k3dsdk/measurement.h>
#include <k3dsdk/mesh_modifier.h>
#include <k3dsdk/mesh_selection_sink.h>
#include <k3dsdk/module.h>
#include <k3dsdk/node.h>
#include <k3dsdk/persistent.h>

#include <iterator>

namespace detail
{

typedef std::vector<k3d::split_edge*> edges_t;

/// Returns every point within the polyhedron that that is an "interior" point, along with the list of edges that adjoin it
typedef std::map<k3d::point*, edges_t > point_edges_map_t;
point_edges_map_t interior_points(k3d::polyhedron& Polyhedron)
{
	point_edges_map_t results;

	for(k3d::polyhedron::faces_t::iterator face = Polyhedron.faces.begin(); face != Polyhedron.faces.end(); ++face)
		{
			k3d::split_edge* first_edge = (*face)->first_edge;
			k3d::split_edge* edge = first_edge;

			do
				{
					assert_warning(edge->vertex);

					edges_t edges;
					for(k3d::split_edge* edge2 = edge; edge2 && edge2->companion; edge2 = edge2->companion->face_clockwise)
						{
							edges.push_back(edge2);

							if(edge2->companion->face_clockwise == edge)
								{
									results[edge->vertex] = edges;
									break;
								}
						}

					edge = edge->face_clockwise;
				}
			while(edge != first_edge);
		}

	return results;
}

} // namespace

namespace libk3ddeformation
{

/////////////////////////////////////////////////////////////////////////////
// smooth_points

class smooth_points :
	public k3d::mesh_selection_sink<k3d::mesh_modifier<k3d::persistent<k3d::node> > >
{
	typedef k3d::mesh_selection_sink<k3d::mesh_modifier<k3d::persistent<k3d::node> > > base;

public:
	smooth_points(k3d::idocument& Document) :
		base(Document),
		m_smoothing(init_owner(*this) + init_name("smoothing") + init_label(_("Smoothing")) + init_description(_("Smoothing factor")) + init_value(1.0) + init_step_increment(0.1) + init_units(typeid(k3d::measurement::scalar)))
	{
		m_mesh_selection.changed_signal().connect(make_reset_mesh_slot());

		m_smoothing.changed_signal().connect(make_update_mesh_slot());
	}

	void on_create_mesh(const k3d::mesh& InputMesh, k3d::mesh& Mesh)
	{
		k3d::deep_copy(InputMesh, Mesh);
		k3d::replace_selection(m_mesh_selection.value(), Mesh);
	}

	void on_update_mesh(const k3d::mesh& Source, k3d::mesh& Target)
	{
		// Sanity checks ...
		return_if_fail(Source.polyhedra.size() == Target.polyhedra.size());
		return_if_fail(Source.points.size() == Target.points.size());

		const double smoothing = m_smoothing.value();

		detail::point_edges_map_t points_edges;

		for(unsigned int i = 0; i != Source.polyhedra.size(); ++i)
			{
				points_edges = detail::interior_points(*Source.polyhedra[i]);
			}

		for(unsigned long i = 0; i != Target.points.size(); ++i)
			{
				Target.points[i]->position = Source.points[i]->position;

				detail::point_edges_map_t::iterator p = points_edges.find(Source.points[i]);
				if(p == points_edges.end())
					continue;

				detail::edges_t& edges = p->second;
				if(edges.empty())
					continue;

				k3d::point3 position(0, 0, 0);
				for(detail::edges_t::const_iterator edge = edges.begin(); edge != edges.end(); ++edge)
					position += (*edge)->companion->vertex->position;

				Target.points[i]->position = k3d::mix(Source.points[i]->position, k3d::mix(Source.points[i]->position, position / edges.size(), smoothing), Target.points[i]->selection_weight);
			}
	}

	k3d::iplugin_factory& factory()
	{
		return get_factory();
	}

	static k3d::iplugin_factory& get_factory()
	{
		static k3d::plugin_factory<
			k3d::document_plugin<smooth_points>,
			k3d::interface_list<k3d::imesh_source,
			k3d::interface_list<k3d::imesh_sink > > > factory(
				k3d::uuid(0x3d6d1c61, 0x706f4968, 0x93462489, 0x9a3134a7),
				"SmoothPoints",
				_("Smooths mesh points by averaging their positions with their neighbors'"),
				"Deformation",
				k3d::iplugin_factory::STABLE);

		return factory;
	}

private:
	k3d_data(double, immutable_name, change_signal, with_undo, local_storage, no_constraint, measurement_property, with_serialization) m_smoothing;
};

/////////////////////////////////////////////////////////////////////////////
// smooth_points_factory

k3d::iplugin_factory& smooth_points_factory()
{
	return smooth_points::get_factory();
}

} // namespace libk3ddeformation


