// K-3D
// Copyright (c) 1995-2004, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\author Romain Behar (romainbehar@yahoo.com)
*/

#include <k3dsdk/i18n.h>
#include <k3dsdk/node.h>
#include <k3dsdk/persistent.h>
#include <k3dsdk/measurement.h>
#include <k3dsdk/mesh_modifier.h>
#include <k3dsdk/module.h>

#include <iterator>

namespace libk3dmesh
{

/////////////////////////////////////////////////////////////////////////////
// make_bilinear_patches_implementation

class make_bilinear_patches_implementation :
	public k3d::mesh_modifier<k3d::persistent<k3d::node> >
{
	typedef k3d::mesh_modifier<k3d::persistent<k3d::node> > base;

public:
	make_bilinear_patches_implementation(k3d::idocument& Document) :
		base(Document)
	{
	}

	/** \todo Improve the implementation so we don't have to do this */
	k3d::iunknown* on_rewrite_hint(iunknown* const Hint)
	{
		// Force updates to re-allocate our mesh, for simplicity
		return 0;
	}

	void on_create_mesh(const k3d::mesh& InputMesh, k3d::mesh& Mesh)
	{
		std::map<k3d::point*, k3d::point*> point_map;
		point_map[0] = 0;
		for(k3d::mesh::points_t::const_iterator point = InputMesh.points.begin(); point != InputMesh.points.end(); ++point)
		{
			k3d::point* const new_point = new k3d::point(**point);
			Mesh.points.push_back(new_point);
			point_map[*point] = new_point;
		}

		// Collect faces ...
		for(k3d::mesh::polyhedra_t::const_iterator polyhedron = InputMesh.polyhedra.begin(); polyhedron != InputMesh.polyhedra.end(); ++polyhedron)
		{
			for(k3d::polyhedron::faces_t::const_iterator face = (*polyhedron)->faces.begin(); face != (*polyhedron)->faces.end(); ++face)
			{
				std::vector<k3d::split_edge*> edges;
				for(k3d::split_edge* edge = (*face)->first_edge; edge; edge = edge->face_clockwise)
				{
					edges.push_back(edge);
					if(edge->face_clockwise == (*face)->first_edge)
						break;
				}
				if(edges.size() != 4)
					continue;

				k3d::bilinear_patch* const new_patch = new k3d::bilinear_patch();
				Mesh.bilinear_patches.push_back(new_patch);

				new_patch->material = (*face)->material;
				new_patch->control_points[0] = point_map[edges[0]->vertex];
				new_patch->control_points[1] = point_map[edges[1]->vertex];
				new_patch->control_points[2] = point_map[edges[3]->vertex];// bicubic patch control points *aren't* in clockwise order!
				new_patch->control_points[3] = point_map[edges[2]->vertex];
			}
		}
	}

	void on_update_mesh(const k3d::mesh& InputMesh, k3d::mesh& Mesh)
	{
	}

	k3d::iplugin_factory& factory()
	{
		return get_factory();
	}

	static k3d::iplugin_factory& get_factory()
	{
		static k3d::plugin_factory<
			k3d::document_plugin<make_bilinear_patches_implementation>,
			k3d::interface_list<k3d::imesh_source,
			k3d::interface_list<k3d::imesh_sink> > > factory(
				k3d::uuid(0xe99f3672, 0x6e85457d, 0x9ce1f9dc, 0x6bb71691),
				"MakeBilinearPatches",
				_("Converts input quadrilaterals to bilinear patches"),
				"Patches",
				k3d::iplugin_factory::STABLE);

		return factory;
	}
};

/////////////////////////////////////////////////////////////////////////////
// make_bilinear_patches_factory

k3d::iplugin_factory& make_bilinear_patches_factory()
{
	return make_bilinear_patches_implementation::get_factory();
}

} // namespace libk3dmesh


