/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <qtimer.h>
#include <qpopupmenu.h>
#include <qprocess.h>
#include <qtooltip.h>

#include <kapplication.h>

#include "debug.h"
#include "action.h"
#include "kadu.h"
#include "message_box.h"
#include "userlist.h"
#include "config_dialog.h"
#include "status.h"
#include "modules.h"
#include "chat_manager.h"
#include "icons_manager.h"
#include "chat.h"
#include "custom_input.h"

#include "amarok.h"

#define MODULE_AMAROK_VERSION 1.19

// For ID3 tags signatures cutter
const char DEFAULT_SIGNATURES[] = "! WWW.POLSKIE-MP3.TK ! \n! www.polskie-mp3.tk ! ";

AmaroK* amarok;

extern "C" int amarok_init()
{
	amarok = new AmaroK();
	return 0;
}

extern "C" void amarok_close()
{
	delete amarok;
	amarok = NULL;
}

AmaroK::AmaroK()
{
	kdebugf();

	// amaroK app version
	QProcess proc;

	proc.addArgument("amarok");
	proc.addArgument("--version");

	proc.start();

	while ( proc.isRunning() );

	amarokAppVer = static_cast<QString>( proc.readStdout() );

	int ind = amarokAppVer.find("amaroK: ", 0);

	if (ind != -1)
		amarokAppVer = amarokAppVer.mid( (ind + 8), (amarokAppVer.length() - ind - 9) );
	else
		amarokAppVer = QString::null;

	// Chat windows menu
	menu = new QPopupMenu();
	popups[0] = menu->insertItem(tr("Put formated string"), this, SLOT(putSongTitle(int)));
	popups[1] = menu->insertItem(tr("Put song title"), this, SLOT(putSongTitle(int)));

	// amaroK module objects
	client = new DCOPClient();
        client->attach();
	config = new ConfigFile(ggPath(QString("/kadu-amarok.conf")));
	timer = new QTimer();

	// Statuses menu entry
	bool menuPos = config->readBoolEntry("amaroK", "dockMenu", false);
	if (menuPos)
		popups[2] = dockMenu->insertItem(tr("Enable amaroK statuses"), this, SLOT(toggleTimer(int)), 0, -1, 10);
	else
		popups[2] = kadu->mainMenu()->insertItem(tr("Enable amaroK statuses"), this, SLOT(toggleTimer(int)), 0, -1, 0);

	// Handle chat buttons
	connect(timer, SIGNAL(timeout()), this, SLOT(checkTitle()));
	connect(kadu, SIGNAL(changingStatus(UserStatus&, bool&)), this, SLOT(checkStatusDesc(UserStatus&, bool&)));

	// Configuration dialog
	ConfigDialog::addTab("amaroK", dataPath("kadu/modules/data/amarok/amarok.png"));

	ConfigDialog::registerSlotOnCreateTab("amaroK", this, SLOT(onCreateConfig()));
	ConfigDialog::registerSlotOnCloseTab("amaroK", this, SLOT(onDestroyConfig()));
	ConfigDialog::registerSlotOnApplyTab("amaroK", this, SLOT(onUpdateConfig()));

	ConfigDialog::addVGroupBox("amaroK", "amaroK", QT_TRANSLATE_NOOP("@default", "Formats"));
		ConfigDialog::addLineEdit(config, "amaroK", "Formats", QT_TRANSLATE_NOOP("@default", "amaroK message format for chat windows"), "chatString", "amaroK: %t [%c / %l]",
			QT_TRANSLATE_NOOP("@default", "%t - song title (artist - title), %f - file name, %l - song length (MM:SS),<br>%c - current song position (MM:SS), %p - percents of played song, %i - song title, %a - song artist, %b - album"));
		ConfigDialog::addLineEdit(config, "amaroK", "Formats", QT_TRANSLATE_NOOP("@default", "amaroK status format"), "statusString", "-=[%t]=-",
			QT_TRANSLATE_NOOP("@default", "%t - song title (artist - title), %f - file name, %l - song length (MM:SS),<br>%c - current song position (MM:SS), %p - percents of played song, %i - song title, %a - song artist, %b - album"));

	ConfigDialog::addVGroupBox("amaroK", "amaroK", QT_TRANSLATE_NOOP("@default","Cut signatures"));
		ConfigDialog::addCheckBox(config, "amaroK", "Cut signatures", QT_TRANSLATE_NOOP("@default", "Enable signatures cutting"), "signature", TRUE, QT_TRANSLATE_NOOP("@default", "Shell I cut these craps?"));
		ConfigDialog::addTextEdit(config, "amaroK", "Cut signatures", QT_TRANSLATE_NOOP("@default", "Signatures:"), "signatures", DEFAULT_SIGNATURES);

	ConfigDialog::addCheckBox(config, "amaroK", "amaroK", QT_TRANSLATE_NOOP("@default","Put statuses switch to dock menu"), "dockMenu", false, QT_TRANSLATE_NOOP("@default","Enable this to put \"amaroK statuses switch\"\ninto docked icon menu."));

	Action* amarok_action = new Action(icons_manager->loadIcon("Unmute"),
		tr("amaroK"), "amarokAction", Action::TypeChat);
	connect(amarok_action, SIGNAL(activated(const UserGroup*, const QWidget*, bool)),
		this, SLOT(amarokActionActivated(const UserGroup*, const QWidget*)));
	KaduActions.insert("amarokAction", amarok_action);
}

AmaroK::~AmaroK()
{
	kdebugf();
	int idx = dockMenu->indexOf(popups[2]);
	if (idx == -1)
		kadu->mainMenu()->removeItem(popups[2]);
	else
		dockMenu->removeItem(popups[2]);

	KaduActions.remove("amarokAction");

	disconnect(timer, SIGNAL(timeout()), this, SLOT(checkTitle()));
	disconnect(kadu, SIGNAL(changingStatus(UserStatus&, bool&)), this, SLOT(checkStatusDesc(UserStatus&, bool&)));

	ConfigDialog::unregisterSlotOnCreateTab("amaroK", this, SLOT(onCreateConfig()));
	ConfigDialog::unregisterSlotOnCloseTab("amaroK", this, SLOT(onDestroyConfig()));
	ConfigDialog::unregisterSlotOnApplyTab("amaroK", this, SLOT(onUpdateConfig()));
	ConfigDialog::removeControl("amaroK", "Put statuses switch to dock menu");
	ConfigDialog::removeControl("amaroK", "Signatures:");
	ConfigDialog::removeControl("amaroK", "Enable signatures cutting");
	ConfigDialog::removeControl("amaroK", "Cut signatures");
	ConfigDialog::removeControl("amaroK", "amaroK message format for chat windows");
	ConfigDialog::removeControl("amaroK", "amaroK status format");
	ConfigDialog::removeControl("amaroK", "Formats");
	ConfigDialog::removeTab("amaroK");

	client->detach();
	delete client;
	delete menu;
	delete timer;
	delete config;
}

void AmaroK::amarokActionActivated(const UserGroup* users, const QWidget* source)
{
	kdebugf();

	menu->popup(source->mapToGlobal(QPoint(0,20)));
}

bool AmaroK::isOn()
{
	kdebugf();
	QByteArray replyData, data;
	QCString replyType;
	return ((client->call("amarok", "player", "isPlaying()", data, replyType, replyData)) ? true : false);
}

bool AmaroK::isPlaying()
{
	bool b;
	QByteArray replyData, data;
	QCString replyType;
	if (!client->call("amarok", "player", "isPlaying()", data, replyType, replyData))
		return false;

	QDataStream reply(replyData, IO_ReadOnly);
	reply >> b;
	return b;
}

QString AmaroK::getTitle()
{
	kdebugf();
	QByteArray replyData, data;
	QCString replyType;
	if (!client->call("amarok", "player", "title()", data, replyType, replyData))
		return "";

	QDataStream reply(replyData, IO_ReadOnly);
	QString curTrack;
	reply >> curTrack;
	return curTrack.simplifyWhiteSpace();
}

QString AmaroK::getArtist()
{
	kdebugf();
	QByteArray replyData, data;
	QCString replyType;
	if (!client->call("amarok", "player", "artist()", data, replyType, replyData))
		return "";

	QDataStream reply(replyData, IO_ReadOnly);
	QString curArtist;
	reply >> curArtist;
	return curArtist.simplifyWhiteSpace();
}

QString AmaroK::getAlbum()
{
	kdebugf();
	QByteArray replyData, data;
	QCString replyType;
	if (!client->call("amarok", "player", "album()", data, replyType, replyData))
		return "";

	QDataStream reply(replyData, IO_ReadOnly);
	QString curAlbum;
	reply >> curAlbum;
	return curAlbum.simplifyWhiteSpace();
}

QString AmaroK::nowPlaying()
{
	kdebugf();
	QByteArray replyData, data;
	QCString replyType;
	if (!client->call("amarok", "player", "nowPlaying()", data, replyType, replyData))
		return "";

	QDataStream reply(replyData, IO_ReadOnly);
	QString curTrack;
	reply >> curTrack;
	QString title( curTrack.simplifyWhiteSpace() );

	// Now, if we want to cut nasty signatures, we do it!
	if (config->readBoolEntry("amaroK", "signature", TRUE))
	{
		QStringList sigList(QStringList::split('\n', config->readEntry("amaroK", "signatures", DEFAULT_SIGNATURES)));
		for (unsigned int i = 0; i < sigList.count(); i++)
			title.remove(sigList[i]);
	}

	return title;
}

int AmaroK::getLength()
{
	kdebugf();
	QByteArray replyData, data;
	QCString replyType;
	int t;
	if (!client->call("amarok", "player", "trackTotalTime()", data, replyType, replyData))
		return 0;

	QDataStream reply(replyData, IO_ReadOnly);
	reply >> t;
	return t;
}

int AmaroK::getCurrentPos()
{
	kdebugf();
	QByteArray replyData, data;
	QCString replyType;
	int pos;
	if (!client->call("amarok", "player", "trackCurrentTime()", data, replyType, replyData))
		return 0;

	QDataStream reply(replyData, IO_ReadOnly);
	reply >> pos;
	return pos;
}

void AmaroK::toggleTimer(int id)
{
	kdebugf();
	bool enabled;
	QPopupMenu* menu;
	int idx = dockMenu->indexOf(popups[2]);
	if (idx == -1)
		menu = kadu->mainMenu();
	else
		menu = dockMenu;

	enabled = menu->isItemChecked(popups[2]);

	if (!isOn() && !enabled)
	{
		MessageBox::msg(tr("amaroK isn't runned!"));
		return;
	}

	menu->setItemChecked(popups[2], !enabled);
	if (enabled)
		timer->stop();
	else
		timer->start(1000);
}

void AmaroK::checkTitle()
{
	if (!isOn())
	{
		timer->stop();
		kadu->mainMenu()->setItemChecked(popups[2], false);
		return;
	}
	if (!gadu->status().isOffline())
	{
		QString title = getTitle();
		if (title != currentTitle || !gadu->status().hasDescription())
		{
			currentTitle = title;
			gadu->status().setDescription(parse(config->readEntry("amaroK", "statusString")));
		}
	}
}

void AmaroK::putSongTitle(int ident)
{
	kdebugf();
	if (!isPlaying())
	{
		MessageBox::msg(tr("amaroK isn't playing!"));
		return;
	}
	Chat* chat = getCurrentChat();
	QString title;
	if (popups[0] == ident)
		title = parse(config_file.readEntry("amaroK", "chatString"));

	if (popups[1] == ident)
		title = getTitle();

	int x, y;

	HtmlDocument doc, doc2, doc3;
	chat->edit()->getCursorPosition(&y, &x);
	chat->edit()->insertAt(title, y, x);
	doc.parseHtml(chat->edit()->text());
	for ( int i = 0; i < doc.countElements(); i++ )
	{
		if (i == 7)
		{
			doc2.parseHtml(doc.elementText(i));
			for ( int j = 0; j < doc2.countElements(); j++ )
			{
				if (doc2.isTagElement(j))
					doc3.addTag(doc2.elementText(j));
				else
					doc3.addText(doc2.elementText(j));
			}
		} else {
			if (doc.isTagElement(i))
				doc3.addTag(doc.elementText(i));
			else
				doc3.addText(doc.elementText(i));
		}
	}
	chat->edit()->setText(doc3.generateHtml());
	chat->edit()->moveCursor(QTextEdit::MoveEnd, FALSE);
}

QString AmaroK::parse(QString str)
{
	kdebugf();
	uint sl = str.length();
	QString r;

	for ( uint i = 0; i < sl; i++ )
	{
		while (str[i] != '%' && i < sl)
		{
			r += str[i];
			i++;
		}

		if (str[i] == '%')
		{
			i++;
			switch (str[i].latin1())
			{
				case 't':
					r += nowPlaying();
					break;
				case 'a':
					r += getArtist();
					break;
				case 'i':
					r += getTitle();
					break;
				case 'b':
					r += getAlbum();
					break;
				case 'l':
				{
					char ms[10];
					int lgt = getLength(), m, s;
					m = lgt / 60;
					s = lgt % 60;
					if (s < 10)
						sprintf(ms, "%d:0%d", m, s);
					else
						sprintf(ms, "%d:%d", m, s);

					r += ms;
					break;
				}
				case 'c':
				{
					char ms[10];
					int lgt = getCurrentPos(), m, s;
					m = lgt / 60;
					s = lgt % 60;
					if (s < 10)
						sprintf(ms, "%d:0%d", m, s);
					else
						sprintf(ms, "%d:%d", m, s);

					r += ms;
					break;
				}
				case 'p':
				{
					char tmp[10];
					int perc = getCurrentPos() / (getLength()/1000);
					sprintf(tmp, "%d%%", perc);
					r += tmp;
					break;
				}
				case 'd':
				{
					r += descriptionToUse;
					break;
				}
				case 'v':
				{
					if ( !amarokAppVer.isEmpty() )
						r += amarokAppVer;
					break;
				}
				default:
					r += str[i];
			}
		}
	}
	return r;
}

Chat* AmaroK::getCurrentChat()
{
	kdebugf();
	ChatList cs = chat_manager->chats();
	uint i;
	for ( i = 0; i < cs.count(); i++ )
	{
		if (cs[i]->isActiveWindow())
		{
			break;
		}
	}
	if (i == cs.count())
		return 0;

	return cs[i];
}

void AmaroK::checkStatusDesc(UserStatus& status, bool &stop)
{
	QString str = config_file.readEntry("amaroK", "statusString");
	if (status.description().find("%amarok%") > -1)
	{
		QString newDesc = status.description().replace("%amarok%", parse(str));
		status.setDescription(newDesc);
		gadu->status().setStatus(status);
		stop = TRUE;
	}
}

void AmaroK::onCreateConfig()
{
	modules_manager->moduleIncUsageCount("amarok");
}

void AmaroK::onDestroyConfig()
{
	onUpdateConfig();
	modules_manager->moduleDecUsageCount("amarok");
}

void AmaroK::onUpdateConfig()
{
	int idx = dockMenu->indexOf(popups[2]);
	if (idx == -1)
		kadu->mainMenu()->removeItem(popups[2]);
	else
		dockMenu->removeItem(popups[2]);

	bool enabled = false;
	if (timer->isActive())
		enabled = true;

	bool menuPos = config->readBoolEntry("amaroK", "dockMenu", false);
	if (menuPos)
	{
		popups[2] = dockMenu->insertItem(tr("Enable amaroK statuses"), this, SLOT(toggleTimer(int)), 0, -1, 10);
		dockMenu->setItemChecked(popups[2], enabled);
	}
	else
	{
		popups[2] = kadu->mainMenu()->insertItem(tr("Enable amaroK statuses"), this, SLOT(toggleTimer(int)), 0, -1, 0);
		kadu->mainMenu()->setItemChecked(popups[2], enabled);
	}

	config->sync();
}
