#ifndef CMDLINE_H
#define CMDLINE_H

#include <qobject.h>
#include <qstringlist.h>

class Chat;
class UserGroup;
class CustomInput;
class CmdLineHistory;

/*!
 * \fn CHECK_CMD(cmdName)
 * Designed to use inside "CmdLine" command implementation method.
 * Should be called at the begining of the method, to avoid
 * executing, if emited command signal isn't delivered for current method.
 * \param cmdName Name of command which the method implements.
 */
#define CHECK_CMD(cmdName) \
	if (cmdName != cmd) return;

/*!
 * Handles chat input messages which starts with '/' character
 * and interpretates it as command.
 * \class CmdLine
 * \brief Chat input commands execution.
 */
class CmdLine : public QObject
{
	Q_OBJECT

	private:
		/*!
		 * \enum DelSide
		 * Defines if word to complete (replace) is on the right or left side ot cursor.
		 * \var DelSide LEFT
		 * Word (or its part) is on the left side of cursor.
		 * \var DelSide RIGHT
		 * Word (or its part) is on the right side of cursor.
		 */
		enum DelSide
		{
			LEFT,
			RIGHT
		};

		/*!
		 * \var DelSide delSide
		 * Keeps word deletion side value given to constructor for later use.
		 */
		DelSide delSide;

		/*!
		 * \var int deleteLength
		 * Keeps length of expression to delete befor inserting choosen one.
		 */
		int deleteLength;

		/*!
		 * \var QStringList cmdList
		 * List of valid (added) commands.
		 */
		QStringList cmdList;

		/*!
		 * \var QStringList ignoredCmdList
		 * List of commands to be ignored by commands handler.
		 * This list should include all commands used by another modules, so
		 * PowerKadu doesn't block them.
		 */
		QStringList ignoredCmdList;

		/*!
		 * \var CmdLineHistory* history
		 * Chat command line history handler. Allows to get previously
		 * entered strings in chat input by pressing ctrl+up and ctrl+down.
		 */
		CmdLineHistory* history;

		/*!
		 * \fn void readCfg()
		 * Reads ignored commands list from the config file.
		 */
		void readCfg();

		/*!
		 * \fn void writeCfg()
		 * Writes ignored commands list to config file.
		 */
		void writeCfg();
		/*!
		 * \fn void chatCreated(Chat *chat)
		 * Connects chat signals to local object slots.
		 * \param chat Signal emiter.
		 */
		void chatCreated(Chat *chat);

		/*!
		 * \enum CompletionType
		 * Used (internaly) to mark completion mode.
		 * \var CompletionType UNDEFINED
		 * Not defined yet.
		 * \var CompletionType COMMAND
		 * Command completion mode.
		 * \var CompletionType NICK_OR_UID
		 * Nick or UID (UIN/etc) completion mode.
		 * \var CompletionType EMOTICON
		 * Emoticons completion mode.
		 */
		enum CompletionType
		{
			UNDEFINED,
			COMMAND,
			NICK_OR_UID,
			EMOTICON
		};

	public:
		/*!
		 * \fn CmdLine()
		 * Default constructor. Connects to chat windows and adds "help" command.
		 */
		CmdLine();

		/*!
		 * \fn ~CmdLine()
		 * Disconnects from chat windows.
		 */
		~CmdLine();

		/*!
		 * \fn void addCmd(QString cmd, QObject *receiver, const char* slot)
		 * Adds given command to valid commands list and connects its slot
		 * to local \sa cmdCall"("Chat* chat, const UserGroup* users, QStringList &args")"
		 * signal.
		 * \param cmd Command name.
		 * \param receiver Object which has method implementing the command.
		 * \param slot Slot of above \a receiver, which implements the command.
		 * An example of usage:
		 * \code
		 * file.h:
		 * -------
		 * class Abc
		 * {
		 * 	Q_OBJECT
		 * 		public:
		 * 		Abc();
		 * 		public slots:
		 * 		void cmd(Chat* chat, const UserGroup* users, QString &cmd, QStringList &args);
		 * }
		 * 	file.cpp
		 * --------
		 * #include "powerkadu.h"
		 * Abc::Abc()
		 * {
		 * 	powerKadu->cmds()->addCmd("command_name", this,
		 * 		SLOT(cmd(Chat*, const UserGroup*, QString&, QStringList&, QCString&)));
		 * }
		 * 	Abc::cmd(Chat* chat, const UserGroup* users, QString &cmd, QStringList &args, QCString &msg)
		 * {
		 * 	CHECK_CMD("command_name");
		 * 	// command implementation
		 * }
		 * \endcode
		 */
		void addCmd(QString cmd, QObject *receiver, const char* slot);

		/*!
		 * \fn void sortCmds()
		 * Sorts all commands in the list. Called by main PK object after
		 * all other PK objects was created.
		 */
		void sortCmds();

		/*!
		 * \fn void complete(CustomInput *input)
		 * Parses content of chat input field and completes nick/UIN/command/etc.
		 * if it's needed.
		 * \param input Chat input field with contents to be complated.
		 */
		void complete(CustomInput *input);

		/*!
		 * \fn void put(QString word, Chat* chat)
		 * Puts given word as a completion for requeted (before) expression.
		 * \param word The word to put.
		 * \param chat Chat window to put the word into.
		 */
		void put(QString word, Chat* chat);

		/*!
		 * \fn void historyPrev(CustomInput *input)
		 * Gets proper (previous) string from input history handler and puts it into
		 * the input.
		 * \param input Chat input field to get history for.
		 */
		void historyPrev(CustomInput *input);

		/*!
		 * \fn void historyNext(CustomInput *input)
		 * Gets proper (next) string from input history handler and puts it into
		 * the input.
		 * \param input Chat input field to get history for.
		 */
		void historyNext(CustomInput *input);

		/*!
		 * \fn Chat* getChatByInput(CustomInput *input)
		 * Gets chat window for given input field.
		 * \param input Input to search chat window by.
		 * \return Found chat window or <code>null</code> when there is no such window found.
		 * \note It's kind of utility function.
		 */
		Chat* getChatByInput(CustomInput *input);

		/*!
		 * \fn QString chatUniqKey(Chat* chat)
		 * Generates unique key by users included in the chat. It's done by
		 * generating list of these users, sorting them and concatenating them with '_' character.
		 * \param chat Chat window to generate key for.
		 * \return Ready to use key string.
		 * \note It's kind of utility function.
		 */
		QString chatUniqKey(Chat* chat);

		/*!
		 * \fn CmdLineHistory* getCmdLineHistory()
		 * \return Pointer to chats input history handler object.
		 */
		CmdLineHistory* getCmdLineHistory();

	private slots:
		/*!
		 * \fn void helpCmd(Chat* chat, const UserGroup* users, QString &cmd, QStringList &args, QCString &message)
		 * 	"help" command implementation.
		 * \param chat Help message will be displayed in that chat window.
		 * \param users Ignored.
		 * \param args Help message will be about first element in that list.
		 * \param message Ignored.
		 * \todo Document the cmd parameter!!
		 */
		void helpCmd(Chat* chat, const UserGroup* users, QString &cmd, QStringList &args, QCString &message);

		/*!
		 * \fn void chatCreated(const UserGroup *group)
		 * Called when any new chat window is created.
		 * \param group Users group that chat is opened for.
		 */
		void chatCreated(const UserGroup *group);

		/*!
		 * \fn void handleChatMsg(const UserGroup* users, QCString& msg, bool &stop)
		 * Called when send message request is invoked in any chat window.
		 * \param users Users group that chat is opened for.
		 * \param msg Requested message content. It can be changed directly here.
		 * \param stop Can be set to <code>true</code> to avoid sending the message.
		 */
		void handleChatMsg(const UserGroup* users, QCString& msg, bool &stop);

		/*!
		 * \fn void onCreateTab()
		 * Called when PowerKadu tab is selected in config dialog.
		 */
		void onCreateTab();

		/*!
		 * \fn void onApplyTab()
		 * Called when changes made in config dialog are accepted.
		 */
		void onApplyTab();

		/*!
		 * \fn void onCloseTab()
		 * Called then another tab in config dialog is selected, or
		 * when config dialog is closed.
		 */
		void onCloseTab();

		void onProcessUnknownClicked();

		/*!
		 * \fn void onAddCommand()
		 * Called when "Add command" button is pressed.
		 */
		void onAddCommand();

		/*!
		 * \fn void onRemoveCommand()
		 * Called when "Remove command" is pressed.
		 */
		void onRemoveCommand();

		/*!
		 * \fn void onHighlighted(int)
		 * Called when an item in ignored commands listbox is selected.
		 */
		void onHighlighted(int);

		/*!
		 * \fn void onCommandChanged(const QString &)
		 * Called when text in "Command:" lineedit is changed.
		 */
		void onCommandChanged(const QString &);

		/*!
		 * \fn void setButtons()
		 * Updated buttons (enabled / disabled).
		 */
		void setButtons();

		/*!
		 * \fn void handleChatKey(QKeyEvent* e, CustomInput* input, bool &handled)
		 * Called when any key is pressed in chat input field.
		 * \param e Pointer to event object containing all informations about the event.
		 * \param input Input field of chat window which called the method.
		 */
		void handleChatKey(QKeyEvent* e, CustomInput* input, bool &handled);

		/*!
		 * \fn void handleChatKeyRelease(QKeyEvent* e, CustomInput* input, bool &handled)
		 * Called when any key is released in chat input field.
		 * \param e Pointer to event object containing all informations about the event.
		 * \param input Input field of chat window which called the method.
		 * \todo Document the handled parameter!!
		 */
		void handleChatKeyRelease(QKeyEvent* e, CustomInput* input, bool &handled);

	signals:
		/*!
		 * \fn void cmdCall(Chat* chat, const UserGroup* users, QString &cmd, QStringList &args, QCString &message)
		 * Called when command string sequence was entered in chat input
		 * and confirmed with "return" key. Message interpreted as command
		 * will be avoided from sending it.
		 * \param chat Chat window which command was entered in.
		 * \param users Users group that chat window is opened for.
		 * \param cmd command name.
		 * \param args Additional command arguments.
		 * \param message The whole message. Useful for commands that need to process an entire message
		 *        (e. g. massmsg).
		 */
		void cmdCall(Chat* chat, const UserGroup* users, QString &cmd, QStringList &args, QCString &msg);
};

#endif
