/***************************************************************************
 *   Weather module for Kadu communicator                                  *
 *   2004 Micha Maek (bysk)                                             *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include <qapplication.h>
#include <qprocess.h>
#include <qclipboard.h>
#include <qpushbutton.h>
#include <qtooltip.h>
#include <qlayout.h>
#include <qhbuttongroup.h>
#include <qpopupmenu.h>
#include <qlabel.h>

#include "message_box.h"
#include "icons_manager.h"

#include "showforecastframe.h"
#include "weather_global.h"
#include "textprogress.h"
#include "citysearchresult.h"

ShowForecastFrame::ShowForecastFrame( QWidget* parent )
:
	QFrame( parent ),
	currentPage_(-1)
{
	setWFlags(WDestructiveClose);
	setSizePolicy(QSizePolicy::Minimum, QSizePolicy::Minimum);
	//setMinimumHeight(50);
	//updateGeometry();
	
	fieldTranslator_.insert( "Pressure",    tr("Pressure") );
	fieldTranslator_.insert( "Rain",        tr("Rain") );
	fieldTranslator_.insert( "Snow",        tr("Snow") );
	fieldTranslator_.insert( "Wind speed",  tr("Wind speed") );
	fieldTranslator_.insert( "Description", tr("Description") );
	fieldTranslator_.insert( "Humidity",    tr("Humidity") );
	fieldTranslator_.insert( "Dew point",   tr("Dew point") );
	fieldTranslator_.insert( "Visibility",  tr("Visibility") );
	
	downloadMessage_ = new TextProgress(this);
	downloadMessage_->setMinimumWidth( 300 );
	downloadMessage_->setAlignment( Qt::AlignCenter );
	downloadMessage_->setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Expanding);
	downloadMessage_->hide();
	
	labelMessage_ = new QLabel(this);
	labelMessage_->setAlignment( Qt::AlignTop );
	labelMessage_->setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Expanding);
	labelMessage_->hide();
	//labelMessage_->setTextFormat(Qt::PlainText);
	
	buttonBox_ = new QHButtonGroup(this);
	buttonBox_->setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Minimum);
	buttonBox_->setExclusive(true);
	buttonBox_->setFrameShape(QFrame::NoFrame);
	buttonBox_->setInsideSpacing(1);
	buttonBox_->setInsideMargin(0);
	buttonBox_->hide();
	
	QVBoxLayout* vlayout = new QVBoxLayout(this);
	vlayout->setSpacing( 5 );
	vlayout->setMargin( 5 );
	vlayout->insertWidget( 0, downloadMessage_, 0, Qt::AlignJustify );
	vlayout->insertWidget( 1, labelMessage_ );
	vlayout->insertWidget( 2, buttonBox_ );
	
	contextMenu_ = new QPopupMenu(this);
	contextMenu_->insertItem(tr("Copy"), this, SLOT(menuCopy()));
	contextMenu_->insertItem(tr("Go to Web page"), this, SLOT(menuGoToPage()));
	contextMenu_->insertItem(tr("Change city..."), this, SIGNAL(changeCity()));
	
	connect( buttonBox_, SIGNAL(clicked(int)), this, SLOT(dayClicked(int)) );
	connect( &downloader_, SIGNAL(finished()), this, SLOT(downloadingFinished()));
	connect( &downloader_, SIGNAL(error(GetForecast::ErrorId,QString)), this, SLOT(downloadingError(GetForecast::ErrorId, QString)));
}

void ShowForecastFrame::start( const CitySearchResult& city )
{
	downloadMessage_->setText(tr("Forecast download"));
	if( downloader_.downloadForecast( city.server_, city.cityId_ ) )
	{
		downloadMessage_->hide();
		showForecast();
	}
	else
		downloadMessage_->show();
}

void ShowForecastFrame::downloadingFinished()
{	
	downloadMessage_->hide();
	showForecast();
}

void ShowForecastFrame::downloadingError(GetForecast::ErrorId err, QString url)
{
	downloadMessage_->hide();
	
	labelMessage_->setAlignment( Qt::AlignCenter );
	if(err == GetForecast::Connection)
		labelMessage_->setText( tr("Cannot load page %1").arg(url) );
	else if(err == GetForecast::Parser)
		labelMessage_->setText( tr("Parse error page %1").arg(url) );
	
	labelMessage_->show();
}

void ShowForecastFrame::showForecast()
{
	QValueList<ForecastDay>::const_iterator it;
	for(it = downloader_.getForecast().Days.begin(); it != downloader_.getForecast().Days.end(); it++)
	{
		const ForecastDay& day = (*it);
		QPushButton* btn = new QPushButton(buttonBox_);
		btn->setToggleButton(true);
		btn->setPixmap(QPixmap(day["Icon"]));
		btn->show();
		buttonBox_->insert(btn);
		
		QToolTip::add(btn, "<b>"+day["Name"] + "</b><br>" + day["Temperature"]);
	}
	buttonBox_->setButton(0);
	dayClicked(0);
	
	labelMessage_->show();
	buttonBox_->show();
}

void ShowForecastFrame::dayClicked(int id)
{
	setCurrentPage( id );
	emit dayChanged( currentPage_ );
}

void ShowForecastFrame::mousePressEvent(QMouseEvent * e)
{
	if(e->button() != Qt::RightButton || currentPage_ < 0)
		return;
	
	contextMenu_->popup(mapToGlobal(e->pos()));
}

void ShowForecastFrame::menuCopy()
{
	const ForecastDay& day = downloader_.getForecast().Days[currentPage_];
	QString Data = downloader_.getForecast().LocationName  + " - " + day["Name"] + "\n";
	
	bool first = true;
	for(ForecastDay::const_iterator it = day.begin(); it != day.end(); it++)
	{
		if(it.key() != "Name" && it.key() != "Icon")
		{
			if(first)
				first = false;
			else
				Data += "\n";
			Data += getFieldTranslation( it.key() ) + ": " + it.data();
		}
	}

	Data.replace("&deg;","");
	Data.replace("&nbsp;"," ");
	QApplication::clipboard()->setText( Data );
}

void ShowForecastFrame::menuGoToPage()
{
	QProcess browser;
	QStringList args;
	
	args = QStringList::split(" ", config_file.readEntry("Chat","WebBrowser"));
	
	PlainConfigFile wConfig(  WeatherGlobal::WeatherConfigPath + downloader_.getForecast().config );
	QString url = wConfig.readEntry("Default","Default host") + "/" + wConfig.readEntry("Default","Default path");
	url.replace("%s", downloader_.getForecast().LocationID);
	args.push_back(url);

	browser.setArguments(args);
	browser.start();
}

const QString& ShowForecastFrame::getFieldTranslation( const QString& field )
{
	QMap<QString,QString>::const_iterator translation = fieldTranslator_.find( field );
	if( translation != fieldTranslator_.end() )
		return translation.data();
	else
		return field;
}

void ShowForecastFrame::setCurrentPage( int page )
{
	currentPage_ = page;

	const ForecastDay& day = downloader_.getForecast().Days[currentPage_];
	
	QString Data("");
	Data += "<b><u>" + downloader_.getForecast().LocationName + "</u> " + day["Name"] +
			"</b><hr><table width=\"100%\"><tr><td align=\"left\">";
	Data += "<img src=\""+day["Icon"]+"\"><br>";
	Data += "<b>"+day["Temperature"]+"</b>";
	Data += "</td><td>";
	
	bool first = true;
	for(ForecastDay::const_iterator it = day.begin(); it != day.end(); it++)
	{
		if(it.key() != "Name" && it.key() != "Icon" && it.key() != "Temperature")
		{
			if(first)
				first = false;
			else
				Data += "<br>";
				
			
			Data += getFieldTranslation( it.key() ) + ": " + it.data();
		}
	}
	
	Data += "</td></tr></table>";
	
	labelMessage_->setText(Data);
}

/*
	ShowForecastFrame1
*/

ShowForecastFrame1::ShowForecastFrame1( QWidget* parent, const CitySearchResult& city )
:
	ShowForecastFrame( parent ),
	city_( city ),
	started_( false )
{

}

void ShowForecastFrame1::start()
{
	if( !started_ )
	{
		started_ = true;
		ShowForecastFrame::start( city_ );
	}
}

/*
	ShowForecastFrame2
*/

ShowForecastFrame2::ShowForecastFrame2( QWidget* parent, QString city, QString serverConfigFile )
:
	ShowForecastFrame( parent ),
	city_( city ),
	server_( serverConfigFile ),
	started_( false )
{
	connect( &search_, SIGNAL(finished()), this, SLOT(finished()) );
	connect( &search_, SIGNAL(error(QString)), this, SLOT(error(QString)) );
}

void ShowForecastFrame2::start()
{
	if( !started_ )
	{
		started_ = true;
		downloadMessage_->setText(tr("Searching for <b>%1</b>").arg(city_) );
		downloadMessage_->show();
		search_.findID( city_, server_ );
	}
}

void ShowForecastFrame2::finished()
{
	const CITYSEARCHRESULTS& results = search_.getResult();
	
	if( results.empty() )
	{
		downloadMessage_->hide();
		labelMessage_->show();
		labelMessage_->setAlignment( Qt::AlignCenter );
		labelMessage_->setText( tr("<b>%1</b> not found.").arg(city_) );
	}
	else
		ShowForecastFrame::start( results.first() );
}

void ShowForecastFrame2::error( QString url )
{
	downloadMessage_->hide();
	labelMessage_->show();
	labelMessage_->setAlignment( Qt::AlignCenter );
	labelMessage_->setText( tr("Cannot load page %1.").arg(url) );
}
