/***************************************************************************
*                                                                          *
*   This program is free software; you can redistribute it and/or modify   *
*   it under the terms of the GNU General Public License as published by   *
*   the Free Software Foundation; either version 3 of the License, or      *
*   (at your option) any later version.                                    *
*                                                                          *
***************************************************************************/




#include <QApplication>
#include <QDesktopWidget>
#include <QMap>
#include <QMessageBox>
#include <QMenu>
#include <QLayout>
#include <QList>
#include <QString>
#include <QStringList>
#include <QTimer>
#include <QX11Info>

#include "config_file.h"
#include "debug.h"
#include "gadu.h"
#include "kadu.h"

#include "activate.h"
#include "chat_manager.h"
#include "icons_manager.h"
#include "misc.h"
#include "pending_msgs.h"

#include "globalhotkeys.h"

#include <X11/Xlib.h>
#include <X11/Xatom.h>




GlobalHotkeys *globalHotkeys;




extern "C" int globalhotkeys_init()
{
	kdebugf();
	// create new globalHotkeys object
	globalHotkeys = new GlobalHotkeys();
	// unregister UI file
	MainConfigurationWindow::registerUiFile( dataPath("kadu/modules/configuration/globalhotkeys.ui"), globalHotkeys );
	kdebugf2();
	return 0;
}


extern "C" void globalhotkeys_close()
{
	kdebugf();
	// unregister UI file
	MainConfigurationWindow::unregisterUiFile( dataPath("kadu/modules/configuration/globalhotkeys.ui"), globalHotkeys );
	// delete globalHotkeys object
	delete globalHotkeys;
	globalHotkeys = NULL;
	kdebugf2();
}




Hotkey::Hotkey( bool _shift, bool _control, bool _alt, bool _altgr, bool _super, int _keycode, QString _comment )
{
	shift   = _shift;
	control = _control;
	alt     = _alt;
	altgr   = _altgr;
	super   = _super;
	keycode = _keycode;
	comment = _comment;
}


Hotkey::Hotkey( QString hotkeystring )
{
	// remove "white" characters from begining and end of the string
	hotkeystring = hotkeystring.stripWhiteSpace();
	// initial values
	shift   = false;
	control = false;
	alt     = false;
	altgr   = false;
	super   = false;
	keycode = 0;
	// return if the string is empty
	if( hotkeystring.isEmpty() ) return;
	// split hotkeystring into parts using "+" as the separator
	QStringList parts = QStringList::split( QString("+"), hotkeystring, true );
	// set Shift
	if( parts.contains( "Shift" ) > 0 )
		shift = true;
	// set Control
	if( parts.contains( "Control" ) > 0 )
		control = true;
	// set Alt
	if( parts.contains( "Alt" ) > 0 )
		alt = true;
	// set AltGr
	if( parts.contains( "AltGr" ) > 0 )
		altgr = true;
	// set Super
	if( parts.contains( "Super" ) > 0 )
		super = true;
	bool ok;
	keycode = parts.last().toInt( &ok );
	if( ! ok )  // if the last part of hotkey string is not an integer
	{
		KeySym keysym = XStringToKeysym( parts.last().ascii() );
		if( keysym != NoSymbol )
		{
			keycode = XKeysymToKeycode( QX11Info::display(), keysym );
		}
	}
	comment = hotkeystring;
}


bool Hotkey::equals( Hotkey *h )
{
	if( shift   != h->shift   ) return false;
	if( control != h->control ) return false;
	if( alt     != h->alt     ) return false;
	if( altgr   != h->altgr   ) return false;
	if( super   != h->super   ) return false;
	if( keycode != h->keycode ) return false;
	return true;
}




GlobalHotkeys::GlobalHotkeys() : QObject( NULL, "globalhotkeys" )
{
	// create default configuration if needed
	createDefaultConfiguration();
	// set display to NULL
	display = NULL;
	// create the recentchatsmenu popup menu and reparent it bypass the WM
	recentchatsmenu = new QMenu();
	recentchatsmenu->reparent( (QWidget*)0, Qt::WType_TopLevel | Qt::WX11BypassWM, QPoint( 0, 0 ), false );
	// create and connect() recentchatsmenu popup menu inactivity timer
	recentchatsmenuinactivitytimer = new QTimer( recentchatsmenu );
	connect( recentchatsmenuinactivitytimer, SIGNAL(timeout()), this, SLOT(recentchatsmenuinactivitytimerTimeout()) );
	// create and connect() the hotkeys timer
	hotkeysTimer = new QTimer( this );
	connect( hotkeysTimer, SIGNAL(timeout()), this, SLOT(checkPendingHotkeys()) );
	// not shown yet
	shown = false;
	// read the configuration and force its usage
	configurationUpdated();
}


GlobalHotkeys::~GlobalHotkeys()
{
	// stop the hotkeys timer
	if( hotkeysTimer->isActive() )
	{
		hotkeysTimer->stop();
	}
	// close X11 display
	if( display != NULL )
	{
		XCloseDisplay( display );
	}
	// clear hotkeys list
	for( QMap<QString,Hotkey*>::Iterator I = hotkeys.begin(); I != hotkeys.end(); I++ )
	{
		delete (*I);
	}
	hotkeys.clear();
	// remove configuration widgets if needed
	if( ! HEshowKadusMainWindow.isNull() )
	{
		if( ! HEshowKadusMainWindow.isNull()         ) delete (HotkeyEdit*) HEshowKadusMainWindow;
		if( ! HEhideKadusMainWindow.isNull()         ) delete (HotkeyEdit*) HEhideKadusMainWindow;
		if( ! HEshowHideKadusMainWindow.isNull()     ) delete (HotkeyEdit*) HEshowHideKadusMainWindow;
		if( ! HEopenIncomingChatWindow.isNull()      ) delete (HotkeyEdit*) HEopenIncomingChatWindow;
		if( ! HEopenAllIncomingChatWindows.isNull()  ) delete (HotkeyEdit*) HEopenAllIncomingChatWindows;
		if( ! HEminimizeOpenedChatWindows.isNull()   ) delete (HotkeyEdit*) HEminimizeOpenedChatWindows;
		if( ! HErestoreMinimizedChatWindows.isNull() ) delete (HotkeyEdit*) HErestoreMinimizedChatWindows;
		if( ! HEminimizeRestoreChatWindows.isNull()  ) delete (HotkeyEdit*) HEminimizeRestoreChatWindows;
		if( ! HEcloseAllChatWindows.isNull()         ) delete (HotkeyEdit*) HEcloseAllChatWindows;
		if( ! HEopenChatWith.isNull()                ) delete (HotkeyEdit*) HEopenChatWith;
		if( ! HEopenRecentChats.isNull()             ) delete (HotkeyEdit*) HEopenRecentChats;
	
		// remove configuration widgets for contacts shortcuts if needed
		if( ! contactsAddNewButton.isNull() ) delete (ConfigActionButton*) contactsAddNewButton;
		for( QList< QPointer<ConfigLineEdit> >::Iterator I = contactsNamesEditList.begin(); I != contactsNamesEditList.end(); I++ )
		{
			if( ! (*I).isNull() ) delete (ConfigLineEdit*) (*I);
		}
		for( QList< QPointer<HotkeyEdit> >::Iterator I = contactsHotkeyEditList.begin(); I != contactsHotkeyEditList.end(); I++ )
		{
			if( ! (*I).isNull() ) delete (HotkeyEdit*) (*I);
		}
		// remove configuration groups widgets
		if( groupKadu     ) delete groupKadu;
		if( groupChats    ) delete groupChats;
		if( groupContacts ) delete groupContacts;
	}
}


void GlobalHotkeys::mainConfigurationWindowCreated( MainConfigurationWindow *mainConfigurationWindow )
{
	// create groups boxes
	groupKadu     = mainConfigurationWindow->configGroupBox( "Shortcuts", "Global hotkeys", "Kadu"     );
	groupChats    = mainConfigurationWindow->configGroupBox( "Shortcuts", "Global hotkeys", "Chats"    );
	groupContacts = mainConfigurationWindow->configGroupBox( "Shortcuts", "Global hotkeys", "Contacts" );
	// create hotkey edit widgets
	HEshowKadusMainWindow         = new HotkeyEdit( "GlobalHotkeys", "ShowKadusMainWindow"        , "Show Kadu's main window"           , "", groupKadu  );
	HEhideKadusMainWindow         = new HotkeyEdit( "GlobalHotkeys", "HideKadusMainWindow"        , "Hide Kadu's main window"           , "", groupKadu  );
	HEshowHideKadusMainWindow     = new HotkeyEdit( "GlobalHotkeys", "ShowHideKadusMainWindow"    , "Show/hide Kadu's main window"      , "", groupKadu  );
	HEopenIncomingChatWindow      = new HotkeyEdit( "GlobalHotkeys", "OpenIncomingChatWindow"     , "Open incoming chat window"         , "", groupChats );
	HEopenAllIncomingChatWindows  = new HotkeyEdit( "GlobalHotkeys", "OpenAllIncomingChatWindows" , "Open all incoming chat windows"    , "", groupChats );
	HEminimizeOpenedChatWindows   = new HotkeyEdit( "GlobalHotkeys", "MinimizeOpenedChatWindows"  , "Minimize all opened chat windows"  , "", groupChats );
	HErestoreMinimizedChatWindows = new HotkeyEdit( "GlobalHotkeys", "RestoreMinimizedChatWindows", "Restore all minimized chat windows", "", groupChats );
	HEminimizeRestoreChatWindows  = new HotkeyEdit( "GlobalHotkeys", "MinimizeRestoreChatWindows" , "Minimize/restore all chat windows" , "", groupChats );
	HEcloseAllChatWindows         = new HotkeyEdit( "GlobalHotkeys", "CloseAllChatWindows"        , "Close all chat windows"            , "", groupChats );
	HEopenChatWith                = new HotkeyEdit( "GlobalHotkeys", "OpenChatWith"               , "Open chat with ..."                , "", groupChats );
	HEopenRecentChats             = new HotkeyEdit( "GlobalHotkeys", "OpenRecentChats"            , "Open recent chats ..."             , "", groupChats );

	// set hotkey edit widgets' values
	HEshowKadusMainWindow->setText(         config_file.readEntry( "GlobalHotkeys", "ShowKadusMainWindow"         ).stripWhiteSpace() );
	HEhideKadusMainWindow->setText(         config_file.readEntry( "GlobalHotkeys", "HideKadusMainWindow"         ).stripWhiteSpace() );
	HEshowHideKadusMainWindow->setText(     config_file.readEntry( "GlobalHotkeys", "ShowHideKadusMainWindow"     ).stripWhiteSpace() );
	HEopenIncomingChatWindow->setText(      config_file.readEntry( "GlobalHotkeys", "OpenIncomingChatWindow"      ).stripWhiteSpace() );
	HEopenAllIncomingChatWindows->setText(  config_file.readEntry( "GlobalHotkeys", "OpenAllIncomingChatWindows"  ).stripWhiteSpace() );
	HEminimizeOpenedChatWindows->setText(   config_file.readEntry( "GlobalHotkeys", "MinimizeOpenedChatWindows"   ).stripWhiteSpace() );
	HErestoreMinimizedChatWindows->setText( config_file.readEntry( "GlobalHotkeys", "RestoreMinimizedChatWindows" ).stripWhiteSpace() );
	HEminimizeRestoreChatWindows->setText(  config_file.readEntry( "GlobalHotkeys", "MinimizeRestoreChatWindows"  ).stripWhiteSpace() );
	HEcloseAllChatWindows->setText(         config_file.readEntry( "GlobalHotkeys", "CloseAllChatWindows"         ).stripWhiteSpace() );
	HEopenChatWith->setText(                config_file.readEntry( "GlobalHotkeys", "OpenChatWith"                ).stripWhiteSpace() );
	HEopenRecentChats->setText(             config_file.readEntry( "GlobalHotkeys", "OpenRecentChats"             ).stripWhiteSpace() );

	// add contacts group fields
	contactsAddNewButton = new ConfigActionButton( "Add new shortcut ...", "", groupContacts );
	groupContacts->widget()->layout()->removeWidget( contactsAddNewButton );
	groupContacts->addWidget( contactsAddNewButton, true );  // re-insert the button so that it takes full available horizontal space
	connect( contactsAddNewButton, SIGNAL( clicked() ), this, SLOT( contactsAddNewButtonPressed() ) );
	contactsshortcuts.clear();
	int c = 0;
	while( true )
	{
		QString _contacts = config_file.readEntry( "GlobalHotkeys", QString( "ContactsShortcuts_Contacts_%" ).replace( "%", QString().setNum( c ) ) ).stripWhiteSpace();
		QString _shortcut = config_file.readEntry( "GlobalHotkeys", QString( "ContactsShortcuts_Shortcut_%" ).replace( "%", QString().setNum( c ) ) ).stripWhiteSpace();
		if( _contacts.isEmpty() && _contacts.isEmpty() )
		{
			break;
		}
		contactsshortcuts.push_back( QPair<QStringList,QString>( QStringList::split( ",", _contacts, false ), _shortcut ) );
		c++;
	}
	c = 0;
	for( QList< QPair<QStringList,QString> >::Iterator I = contactsshortcuts.begin(); I != contactsshortcuts.end(); I++ )
	{
		ConfigLineEdit *contactnameedit   = new ConfigLineEdit( "GlobalHotkeys", QString( "ContactsShortcuts_Contacts_%" ).replace( "%", QString().setNum( c ) ), "Contacts (comma separated)", "", groupContacts );
		HotkeyEdit     *contacthotkeyedit = new HotkeyEdit(     "GlobalHotkeys", QString( "ContactsShortcuts_Shortcut_%" ).replace( "%", QString().setNum( c ) ), "Shortcut"                  , "", groupContacts );
		contactsNamesEditList.push_back(  contactnameedit   );
		contactsHotkeyEditList.push_back( contacthotkeyedit );
		contactnameedit->setText( (*I).first.join( ", " ) );
		contacthotkeyedit->setText( (*I).second );
		contactnameedit->show();
		contacthotkeyedit->show();
		c++;
	}
}


void GlobalHotkeys::createDefaultConfiguration()
{
	config_file.addVariable( "GlobalHotkeys", "ShowKadusMainWindow"        , "" );
	config_file.addVariable( "GlobalHotkeys", "HideKadusMainWindow"        , "" );
	config_file.addVariable( "GlobalHotkeys", "ShowHideKadusMainWindow"    , "" );
	config_file.addVariable( "GlobalHotkeys", "OpenIncomingChatWindow"     , "" );
	config_file.addVariable( "GlobalHotkeys", "OpenAllIncomingChatWindows" , "" );
	config_file.addVariable( "GlobalHotkeys", "MinimizeOpenedChatWindows"  , "" );
	config_file.addVariable( "GlobalHotkeys", "RestoreMinimizedChatWindows", "" );
	config_file.addVariable( "GlobalHotkeys", "MinimizeRestoreChatWindows" , "" );
	config_file.addVariable( "GlobalHotkeys", "CloseAllChatWindows"        , "" );
	config_file.addVariable( "GlobalHotkeys", "OpenChatWith"               , "" );
	config_file.addVariable( "GlobalHotkeys", "OpenRecentChats"            , "" );

}


void GlobalHotkeys::configurationUpdated()
{
	// update configuration data
	showKadusMainWindow         = config_file.readEntry( "GlobalHotkeys", "ShowKadusMainWindow"         ).stripWhiteSpace();
	hideKadusMainWindow         = config_file.readEntry( "GlobalHotkeys", "HideKadusMainWindow"         ).stripWhiteSpace();
	showHideKadusMainWindow     = config_file.readEntry( "GlobalHotkeys", "ShowHideKadusMainWindow"     ).stripWhiteSpace();
	openIncomingChatWindow      = config_file.readEntry( "GlobalHotkeys", "OpenIncomingChatWindow"      ).stripWhiteSpace();
	openAllIncomingChatWindows  = config_file.readEntry( "GlobalHotkeys", "OpenAllIncomingChatWindows"  ).stripWhiteSpace();
	minimizeOpenedChatWindows   = config_file.readEntry( "GlobalHotkeys", "MinimizeOpenedChatWindows"   ).stripWhiteSpace();
	restoreMinimizedChatWindows = config_file.readEntry( "GlobalHotkeys", "RestoreMinimizedChatWindows" ).stripWhiteSpace();
	minimizeRestoreChatWindows  = config_file.readEntry( "GlobalHotkeys", "MinimizeRestoreChatWindows"  ).stripWhiteSpace();
	closeAllChatWindows         = config_file.readEntry( "GlobalHotkeys", "CloseAllChatWindows"         ).stripWhiteSpace();
	openChatWith                = config_file.readEntry( "GlobalHotkeys", "OpenChatWith"                ).stripWhiteSpace();
	openRecentChats             = config_file.readEntry( "GlobalHotkeys", "OpenRecentChats"             ).stripWhiteSpace();

	// update contacts shortcuts configuration data and recreate configuration widgets
	contactsshortcuts.clear();
	int c = 0;
	while( true )
	{
		QString _contacts = config_file.readEntry( "GlobalHotkeys", QString( "ContactsShortcuts_Contacts_%" ).replace( "%", QString().setNum( c ) ) ).stripWhiteSpace();
		QString _shortcut = config_file.readEntry( "GlobalHotkeys", QString( "ContactsShortcuts_Shortcut_%" ).replace( "%", QString().setNum( c ) ) ).stripWhiteSpace();
		if( ( ! _contacts.isEmpty() ) || ( ! _shortcut.isEmpty() ) )
		{
			QStringList _contactslist = QStringList::split( ",", _contacts, false );
			for( QStringList::Iterator I = _contactslist.begin(); I != _contactslist.end(); I++ )
			{
				(*I) = (*I).stripWhiteSpace();
			}
			// remove duplicated contacts from the shortcut
			for( int k = _contactslist.size()-1; k > 0; k-- )
			{
				if( _contactslist.indexOf( _contactslist[k] ) != k )
				{
					_contactslist.removeAt( k );
				}
			}
			// remove contacts that do not exist from the shortcut
			for( QStringList::Iterator I = _contactslist.begin(); I != _contactslist.end(); )
			{
				if( ! userlist->containsAltNick( (*I), FalseForAnonymous ) )
				{
					I = _contactslist.erase( I );
				}
				else
				{
					I++;
				}
			}
			contactsshortcuts.push_back( QPair<QStringList,QString>( _contactslist, _shortcut ) );
		}
		else if( contactsNamesEditList.count() == 0 )
		{
			break;
		}
		if( ( contactsNamesEditList.count() > 0 ) && ( c >= contactsNamesEditList.count()-1 ) )
		{
			break;
		}
		c++;
	}
	c = 0;
	for( QList< QPair<QStringList,QString> >::Iterator I = contactsshortcuts.begin(); I != contactsshortcuts.end(); I++ )
	{
		config_file.writeEntry( "GlobalHotkeys", QString( "ContactsShortcuts_Contacts_%" ).replace( "%", QString().setNum( c ) ), (*I).first.join( ", " ) );
		config_file.writeEntry( "GlobalHotkeys", QString( "ContactsShortcuts_Shortcut_%" ).replace( "%", QString().setNum( c ) ), (*I).second             );
		c++;
	}
	config_file.removeVariable( "GlobalHotkeys", QString( "ContactsShortcuts_Contacts_%" ).replace( "%", QString().setNum( c ) ) );
	config_file.removeVariable( "GlobalHotkeys", QString( "ContactsShortcuts_Shortcut_%" ).replace( "%", QString().setNum( c ) ) );
	for( int cc = c+1; cc < contactsNamesEditList.count(); cc++ )
	{
		config_file.removeVariable( "GlobalHotkeys", QString( "ContactsShortcuts_Contacts_%" ).replace( "%", QString().setNum( cc ) ) );
		config_file.removeVariable( "GlobalHotkeys", QString( "ContactsShortcuts_Shortcut_%" ).replace( "%", QString().setNum( cc ) ) );
	}
	if( contactsNamesEditList.count() > 0 )
	{
		for( QList< QPointer<ConfigLineEdit> >::Iterator I = contactsNamesEditList.begin(); I != contactsNamesEditList.end(); I++ )
		{
			if( ! (*I).isNull() ) delete (ConfigLineEdit*) (*I);
		}
		contactsNamesEditList.clear();
		for( QList< QPointer<HotkeyEdit> >::Iterator I = contactsHotkeyEditList.begin(); I != contactsHotkeyEditList.end(); I++ )
		{
			if( ! (*I).isNull() ) delete (HotkeyEdit*) (*I);
		}
		contactsHotkeyEditList.clear();
		c = 0;
		for( QList< QPair<QStringList,QString> >::Iterator I = contactsshortcuts.begin(); I != contactsshortcuts.end(); I++ )
		{
			ConfigLineEdit *contactnameedit   = new ConfigLineEdit( "GlobalHotkeys", QString( "ContactsShortcuts_Contacts_%" ).replace( "%", QString().setNum( c ) ), "Contacts (comma separated)", "", groupContacts );
			HotkeyEdit     *contacthotkeyedit = new HotkeyEdit(     "GlobalHotkeys", QString( "ContactsShortcuts_Shortcut_%" ).replace( "%", QString().setNum( c ) ), "Shortcut"                  , "", groupContacts );
			contactsNamesEditList.push_back(  contactnameedit   );
			contactsHotkeyEditList.push_back( contacthotkeyedit );
			contactnameedit->setText( (*I).first.join( ", " ) );
			contacthotkeyedit->setText( (*I).second );
			contactnameedit->show();
			contacthotkeyedit->show();
			c++;
		}
	}
	// stop the hotkeys timer
	if( hotkeysTimer->isActive() )
	{
		hotkeysTimer->stop();
	}
	// new X11 display
	if( display != NULL )
	{
		XCloseDisplay( display );
	}
	display = XOpenDisplay( 0 );
	// clear hotkeys list
	for( QMap<QString,Hotkey*>::Iterator I = hotkeys.begin(); I != hotkeys.end(); I++ )
	{
		delete (*I);
	}
	hotkeys.clear();
	// update hotkeys list
	hotkeys.insert( "ShowKadusMainWindow"        , new Hotkey( showKadusMainWindow         ) );
	hotkeys.insert( "HideKadusMainWindow"        , new Hotkey( hideKadusMainWindow         ) );
	hotkeys.insert( "ShowHideKadusMainWindow"    , new Hotkey( showHideKadusMainWindow     ) );
	hotkeys.insert( "OpenIncomingChatWindow"     , new Hotkey( openIncomingChatWindow      ) );
	hotkeys.insert( "OpenAllIncomingChatWindows" , new Hotkey( openAllIncomingChatWindows  ) );
	hotkeys.insert( "MinimizeOpenedChatWindows"  , new Hotkey( minimizeOpenedChatWindows   ) );
	hotkeys.insert( "RestoreMinimizedChatWindows", new Hotkey( restoreMinimizedChatWindows ) );
	hotkeys.insert( "MinimizeRestoreChatWindows" , new Hotkey( minimizeRestoreChatWindows  ) );
	hotkeys.insert( "CloseAllChatWindows"        , new Hotkey( closeAllChatWindows         ) );
	hotkeys.insert( "OpenChatWith"               , new Hotkey( openChatWith                ) );
	hotkeys.insert( "OpenRecentChats"            , new Hotkey( openRecentChats             ) );

	c = 0;
	for( QList< QPair<QStringList,QString> >::Iterator I = contactsshortcuts.begin(); I != contactsshortcuts.end(); I++ )
	{
		hotkeys.insert( QString( "ContactsShortcut_%" ).replace( "%", QString().setNum( c ) ), new Hotkey( (*I).second ) );
		c++;
	}
	// grab hotkeys
	grabHotkeys( hotkeys );
	// start the hotkeys timer
	hotkeysTimer->start( GLOBALHOTKEYS_HOTKEYSTIMERINTERVAL, false );
}


void GlobalHotkeys::grabHotkeys( QMap<QString,Hotkey*> hotkeys )
{
	uint modifiers;
	for( QMap<QString,Hotkey*>::Iterator I = hotkeys.begin(); I != hotkeys.end(); I++ )
	{
		if( (*I)->keycode != 0 )
		{
			modifiers = 0;
			modifiers |= ( (*I)->shift   ? GLOBALHOTKEYS_X11SHIFTMASK   : 0 );
			modifiers |= ( (*I)->control ? GLOBALHOTKEYS_X11CONTROLMASK : 0 );
			modifiers |= ( (*I)->alt     ? GLOBALHOTKEYS_X11ALTMASK     : 0 );
			modifiers |= ( (*I)->altgr   ? GLOBALHOTKEYS_X11ALTGRMASK   : 0 );
			modifiers |= ( (*I)->super   ? GLOBALHOTKEYS_X11SUPERMASK   : 0 );
			// install empty messages handler to avoid warning being printed to the output
			messageshandled = 0;
			QtMsgHandler previousmsghandler = qInstallMsgHandler( EmptyMsgHandler );
			// pure hotkey
			XGrabKey(
				display, (*I)->keycode, modifiers,
				DefaultRootWindow( display ), False, GrabModeAsync, GrabModeAsync
				);
			// hotkey with CapsLock
			XGrabKey(
				display, (*I)->keycode, modifiers | GLOBALHOTKEYS_X11CAPSLOCKMASK,
				DefaultRootWindow( display ), False, GrabModeAsync, GrabModeAsync
				);
			// hotkey with NumLock
			XGrabKey(
				display, (*I)->keycode, modifiers | GLOBALHOTKEYS_X11NUMLOCKMASK,
				DefaultRootWindow( display ), False, GrabModeAsync, GrabModeAsync
				);
			// hotkey with CapsLock and NumLock
			XGrabKey(
				display, (*I)->keycode, modifiers | GLOBALHOTKEYS_X11CAPSLOCKMASK | GLOBALHOTKEYS_X11NUMLOCKMASK,
				DefaultRootWindow( display ), False, GrabModeAsync, GrabModeAsync
				);
			// sync the X11 connection, so that we are sure the X11 errors will be handled now
			XSync( display, false );
			// install the previous messages handler
			qInstallMsgHandler( previousmsghandler );
			// check for X11 errors
			if( messageshandled > 0 )
			{
				QMessageBox *messagebox = new QMessageBox(
						qApp->translate( "@default", "Kadu - Global hotkeys" ),
						qApp->translate( "@default", "Hotkey %% is used by another application." ).replace( "%%", (*I)->comment ),
						QMessageBox::Warning,
						QMessageBox::Ok, QMessageBox::NoButton, QMessageBox::NoButton,
						qApp->activeWindow()
					);
				messagebox->show();
				messageshandled = 0;
			}
		}
	}
}


void GlobalHotkeys::contactsAddNewButtonPressed()
{
	int c = 0;
	QList< QPointer<ConfigLineEdit> >::Iterator I_contactsNamesEditList;
	I_contactsNamesEditList  = contactsNamesEditList.begin();
	while( ( I_contactsNamesEditList != contactsNamesEditList.end() ) )
	{
		c++;
		I_contactsNamesEditList++;
	}
	ConfigLineEdit *contactnameedit   = new ConfigLineEdit( "GlobalHotkeys", QString( "ContactsShortcuts_Contacts_%" ).replace( "%", QString().setNum( c ) ), "Contacts (comma separated)", "", groupContacts );
	HotkeyEdit     *contacthotkeyedit = new HotkeyEdit(     "GlobalHotkeys", QString( "ContactsShortcuts_Shortcut_%" ).replace( "%", QString().setNum( c ) ), "Shortcut"                  , "", groupContacts );
	contactsNamesEditList.push_back(  contactnameedit   );
	contactsHotkeyEditList.push_back( contacthotkeyedit );
	contactnameedit->show();
	contacthotkeyedit->show();
}


void GlobalHotkeys::checkPendingHotkeys()
{
	XEvent event;
	bool mod_shift;
	bool mod_control;
	bool mod_alt;
	bool mod_altgr;
	bool mod_super;
	Hotkey *hotkey;
	// (*) calling hide() and show() too fast may result in X11 error
	// (*) "X Error: BadWindow (invalid Window parameter) 3" - it seems to be q Qt bug
	// (*) to avoid warnings being reported we have to handle messages
	QtMsgHandler previousmsghandler = qInstallMsgHandler( EmptyMsgHandler );
	// process any pending hotkeys
	while( XPending( display ) > 0 )
	{
		// get the next event
		XNextEvent( display, &event );
		// is it KeyPress event?
		if( event.type == KeyPress )
		{
			// read modifiers state from current event
			mod_shift   = ( ( event.xkey.state & GLOBALHOTKEYS_X11SHIFTMASK   ) != 0 );
			mod_control = ( ( event.xkey.state & GLOBALHOTKEYS_X11CONTROLMASK ) != 0 );
			mod_alt     = ( ( event.xkey.state & GLOBALHOTKEYS_X11ALTMASK     ) != 0 );
			mod_altgr   = ( ( event.xkey.state & GLOBALHOTKEYS_X11ALTGRMASK   ) != 0 );
			mod_super   = ( ( event.xkey.state & GLOBALHOTKEYS_X11SUPERMASK   ) != 0 );
			// create the hotkey
			hotkey = new Hotkey( mod_shift, mod_control, mod_alt, mod_altgr, mod_super, event.xkey.keycode );
			// check hotkeys and execute related commands
			if( hotkey->equals( hotkeys["ShowKadusMainWindow"] ) )
			{
				if( ! kadu->isVisible() )
				{
					// show and activate Kadu's main window
					kadu->show();
					kadu->raise();
					activateWindow( kadu->winId() );
				}
				else if( kadu->isVisible() && ( ! kadu->isActiveWindow() ) )
				{
					// hide Kadu's main window first to avoid virtual desktop switching
					kadu->hide();
					// show and activate Kadu's main window
					QTimer *showandactivatetimer = new QTimer( kadu );
					connect( showandactivatetimer, SIGNAL(timeout()), this, SLOT(showAndActivateToplevel()) );
					showandactivatetimer->start( GLOBALHOTKEYS_SHOWANDACTIVATETIMERINTERVAL, true );
				}
			}
			else if( hotkey->equals( hotkeys["HideKadusMainWindow"] ) )
			{
				if( kadu->isVisible() )
				{
					// hide Kadu's main window
					kadu->hide();
				}
			}
			else if( hotkey->equals( hotkeys["ShowHideKadusMainWindow"] ) )
			{
				if( ! kadu->isVisible() )
				{
					// show and activate Kadu's main window
					kadu->show();
					kadu->raise();
					activateWindow( kadu->winId() );
				}
				else if( kadu->isVisible() && ( ! kadu->isActiveWindow() ) )
				{
					// hide Kadu's main window first to avoid virtual desktop switching
					kadu->hide();
					// show and activate Kadu's main window
					QTimer *showandactivatetimer = new QTimer( kadu );
					connect( showandactivatetimer, SIGNAL(timeout()), this, SLOT(showAndActivateToplevel()) );
					showandactivatetimer->start( GLOBALHOTKEYS_SHOWANDACTIVATETIMERINTERVAL, true );
				}
				else if( kadu->isVisible() && ( kadu->isActiveWindow() ) )
				{
					// hide Kadu's main window
					kadu->hide();
				}
			}
			else if( hotkey->equals( hotkeys["OpenIncomingChatWindow"] ) )
			{
				if( pending.count() > 0 )
				{
					// open window for pending message(s)
					chat_manager->openPendingMsgs( true );
					// activate it
					QWidget *win = chat_manager->chats()[ chat_manager->chats().count() - 1 ];  // last created chat widget
					while( win->parent() )
						win = static_cast<QWidget *>(win->parent());
					win->show();
					win->raise();
					activateWindow( win->winId() );
				}
				else
				{
					// show window with new unread message(s)
					foreach( ChatWidget *chat, chat_manager->chats() )
					{
						if( chat->newMessagesCount() > 0 )
						{
							// reopen the chat (needed when Tabs module is active)
							chat_manager->openChatWidget( gadu, chat->users()->toUserListElements() );
							// activate the window
							QWidget *win = chat;
							while( win->parent() )
								win = static_cast<QWidget *>(win->parent());
							win->show();
							win->raise();
							activateWindow( win->winId() );
							// done - only one window
							break;
						}
					}
				}
			}
			else if( hotkey->equals( hotkeys["OpenAllIncomingChatWindows"] ) )
			{
				// open all windows for pending message(s)
				while( pending.count() > 0 )
				{
					// open the window
					chat_manager->openPendingMsgs( true );
					// activate it
					QWidget *win = chat_manager->chats()[ chat_manager->chats().count() - 1 ];  // last created chat widget
					while( win->parent() )
						win = static_cast<QWidget *>(win->parent());
					win->show();
					win->raise();
					activateWindow( win->winId() );
				}
				// show all windows with new unread message(s)
				foreach( ChatWidget *chat, chat_manager->chats() )
				{
					if( chat->newMessagesCount() > 0 )
					{
						// reopen the chat (needed when Tabs module is active)
						chat_manager->openChatWidget( gadu, chat->users()->toUserListElements() );
						// activate the window
						QWidget *win = chat;
						while( win->parent() )
							win = static_cast<QWidget *>(win->parent());
						win->show();
						win->raise();
						activateWindow( win->winId() );
					}
				}
			}
			else if( hotkey->equals( hotkeys["MinimizeOpenedChatWindows"] ) )
			{
				// minimize all windows (if needed)
				foreach( ChatWidget *chat, chat_manager->chats() )
				{
					QWidget *win = chat;
					while( win->parent() )
						win = static_cast<QWidget *>(win->parent());
					if( ! win->isMinimized() )
						win->showMinimized();
				}
			}
			else if( hotkey->equals( hotkeys["RestoreMinimizedChatWindows"] ) )
			{
				// restore all windows (if needed) and activate them
				foreach( ChatWidget *chat, chat_manager->chats() )
				{
					QWidget *win = chat;
					while( win->parent() )
						win = static_cast<QWidget *>(win->parent());
					if( win->isMinimized() )
						win->showNormal();
					// hide the window first to avoid virtual desktop switching
					win->hide();
					// show and activate the window
					QTimer *showandactivatetimer = new QTimer( win );
					connect( showandactivatetimer, SIGNAL(timeout()), this, SLOT(showAndActivateToplevel()) );
					showandactivatetimer->start( GLOBALHOTKEYS_SHOWANDACTIVATETIMERINTERVAL, true );
				}
			}
			else if( hotkey->equals( hotkeys["MinimizeRestoreChatWindows"] ) )
			{
				// check if all windows are minimized already
				bool allwindowsminimized = true;
				foreach( ChatWidget *chat, chat_manager->chats() )
				{
					QWidget *win = chat;
					while( win->parent() )
						win = static_cast<QWidget *>(win->parent());
					if( ! win->isMinimized() )
						allwindowsminimized = false;
				}
				if( ! allwindowsminimized )
				{
					// minimize all windows (if needed)
					foreach( ChatWidget *chat, chat_manager->chats() )
					{
						QWidget *win = chat;
						while( win->parent() )
							win = static_cast<QWidget *>(win->parent());
						if( ! win->isMinimized() )
							win->showMinimized();
					}
				}
				else
				{
					// restore all windows (if needed) and activate them
					foreach( ChatWidget *chat, chat_manager->chats() )
					{
						QWidget *win = chat;
						while( win->parent() )
							win = static_cast<QWidget *>(win->parent());
						if( win->isMinimized() )
							win->showNormal();
						// hide the window first to avoid virtual desktop switching
						win->hide();
						// show and activate the window
						QTimer *showandactivatetimer = new QTimer( win );
						connect( showandactivatetimer, SIGNAL(timeout()), this, SLOT(showAndActivateToplevel()) );
						showandactivatetimer->start( GLOBALHOTKEYS_SHOWANDACTIVATETIMERINTERVAL, true );
					}
				}
			}
			else if( hotkey->equals( hotkeys["CloseAllChatWindows"] ) )
			{
				// list of windows to close (needed by Tabs module - we cannot close the same window multiple times!)
				QList<QWidget*> wins;
				foreach( ChatWidget *chat, chat_manager->chats() )
				{
					// add the window to the list
					QWidget *win = chat;
					while( win->parent() )
						win = static_cast<QWidget *>(win->parent());
					if( wins.contains( win ) == 0 )  // if this window is not on the list yet
						wins.push_back( win );
				}
				// close the windows from the list
				for( QList<QWidget*>::Iterator I = wins.begin(); I != wins.end(); I++ )
				{
					(*I)->close();
				}
			}
			else if( hotkey->equals( hotkeys["OpenChatWith"] ) )
			{
				if( openchatwithwindow.isNull() )  // if the "open chat with ..." window is not opened
				{
					// open the window
					openchatwithwindow = new OpenChatWith( kadu );
					openchatwithwindow->move(
							( qApp->desktop()->width()  - openchatwithwindow->width()  ) / 2,
							( qApp->desktop()->height() - openchatwithwindow->height() ) / 2
						);
					openchatwithwindow->show();
					openchatwithwindow->raise();
					activateWindow( openchatwithwindow->winId() );
				}
				else
				{
					// close and delete the window
					openchatwithwindow->close( true );
				}
			}
			else if( hotkey->equals( hotkeys["OpenRecentChats"] ) )
			{
				if( ! recentchatsmenu->isVisible() )
				{
					// stop inactivity checking
					recentchatsmenuinactivitytimer->stop();
					// remove old popup menu items, clear recentchatsusers and recentchatswindows
					recentchatsmenu->clear();
					recentchatsusers.clear();
					recentchatswindows.clear();
					int n = 0;
					// add currently open chats to the popup menu
					if( ! chat_manager->chats().isEmpty() )
					{
						// for each currently open chat
						foreach( ChatWidget *chat, chat_manager->chats() )
						{
							// add popup menu item
							QStringList altnicks = chat->users()->altNicks();
							QString chat_users;
							if( altnicks.count() <= 5 )
							{
								chat_users = altnicks.join( ", " );
							}
							else
							{
								for( int i=0; i<4; i++ )
									chat_users.append( *altnicks.at(i) + ", " );
								chat_users.append( *altnicks.at(4) + ", ..." );
							}
							recentchatsmenu->insertItem( icons_manager->loadIcon( "OpenChat" ), chat_users, this, SLOT(openRecentChat(int)), 0, n );
							// add users
							recentchatsusers[n] = chat->users()->toUserListElements();
							// find and add the window
							QWidget *win = chat;
							while( win->parent() )
								win = static_cast<QWidget *>(win->parent());
							recentchatswindows[n] = win;
							// increase n
							n++;
						}
					}
					// add recently closed chats to the popup menu
					if( ! chat_manager->closedChatUsers().isEmpty() )
					{
						// for each recently closed chat
						foreach( UserListElements users, chat_manager->closedChatUsers() )
						{
							// add popup menu item
							QStringList altnicks = users.altNicks();
							QString chat_users;
							if( altnicks.count() <= 5 )
							{
								chat_users = altnicks.join( ", " );
							}
							else
							{
								for( int i=0; i<4; i++ )
									chat_users.append( *altnicks.at(i) + ", " );
								chat_users.append( *altnicks.at(4) + ", ..." );
							}
							recentchatsmenu->insertItem( icons_manager->loadIcon( "History" ), chat_users, this, SLOT(openRecentChat(int)), 0, n );
							// add users
							recentchatsusers[n] = users;
							// increase n
							n++;
						}
					}
					if( n == 0 )
					{
						// insert disabled item saying there are no recent chats
						recentchatsmenu->insertItem( qApp->translate( "@default", "no recent chats" ) );
						recentchatsmenu->setItemEnabled( recentchatsmenu->idAt( 0 ), false );
					}
					// popup in the center of the screen
					recentchatsmenu->popup( QPoint(
							( qApp->desktop()->width()  - recentchatsmenu->sizeHint().width()  ) / 2,
							( qApp->desktop()->height() - recentchatsmenu->sizeHint().height() ) / 2
						) );
					// activate first item
					if( n > 0 )
					{
						recentchatsmenu->setActiveItem( 0 );
					}
					// make the recentchatsmenu popup menu active window
					recentchatsmenu->setActiveWindow();
					recentchatsmenu->raise();
					activateWindow( recentchatsmenu->winId() );
					XSetInputFocus( QX11Info::display(), recentchatsmenu->winId(), RevertToNone, CurrentTime );
					// start inactivity checking
					recentchatsmenuinactivitytimer->start( GLOBALHOTKEYS_RECENTCHATSMENUINACTIVITYTIMERINTERVAL );
				}
				else
				{
					if( recentchatsmenu->isItemEnabled( recentchatsmenu->idAt( 0 ) ) )  // are there any active items in the popup menu?
					{
						// yes - select next item
						// stop inactivity checking
						recentchatsmenuinactivitytimer->stop();
						// find selected item
						unsigned int index = 0;
						while( ( index < recentchatsmenu->count() ) && ( ! recentchatsmenu->isItemActive( recentchatsmenu->idAt( index ) ) ) )
						{
							index++;
						}
						// find next item
						if( index < recentchatsmenu->count() )
						{
							index++;
							if( index >= recentchatsmenu->count() )
								index = 0;
						}
						else
						{
							index = 0;
						}
						// select the next item
						recentchatsmenu->setActiveItem( index );
						// make the recentchatsmenu popup menu active window
						recentchatsmenu->setActiveWindow();
						recentchatsmenu->raise();
						activateWindow( recentchatsmenu->winId() );
						XSetInputFocus( QX11Info::display(), recentchatsmenu->winId(), RevertToNone, CurrentTime );
						// start inactivity checking
						recentchatsmenuinactivitytimer->start( GLOBALHOTKEYS_RECENTCHATSMENUINACTIVITYTIMERINTERVAL );
					}
					else
					{
						// no - hide the popup menu
						recentchatsmenuinactivitytimer->stop();
						recentchatsmenu->hide();
					}
				}
			}


			else
			{
				for( QList< QPair<QStringList,QString> >::Iterator I = contactsshortcuts.begin(); I != contactsshortcuts.end(); I++ )
				{
					if( hotkey->equals( new Hotkey( (*I).second ) ) )
					{
						UserListElements users;
						for( QStringList::Iterator I2 = (*I).first.begin(); I2 != (*I).first.end(); I2++ )
						{
							users.push_back( userlist->byAltNick( (*I2) ) );
						}
						if( users.count() > 0 )
						{
							chat_manager->openPendingMsgs( users );
							// for each currently open chat
							foreach( ChatWidget *chat, chat_manager->chats() )
							{
								if( chat->users()->toUserListElements() == users )
								{
									// find the window
									QWidget *win = chat;
									while( win->parent() )
										win = static_cast<QWidget *>(win->parent());
									// hide chat window first to avoid virtual desktop switching
									win->hide();
									// show and activate chat window
									QTimer *showandactivatetimer = new QTimer( win );
									connect( showandactivatetimer, SIGNAL(timeout()), this, SLOT(showAndActivateToplevel()) );
									showandactivatetimer->start( GLOBALHOTKEYS_SHOWANDACTIVATETIMERINTERVAL, true );
									usleep( 1000*GLOBALHOTKEYS_SHOWANDACTIVATETIMERINTERVAL/10 );
								}
							}
						}
					}
				}
			}
			// destroy the hotkey
			delete hotkey;
		}
	}
	// (*) restore previous messages handler
	qInstallMsgHandler( previousmsghandler );
}


void GlobalHotkeys::showAndActivateToplevel()
{
	QObject *sender = (QObject*)(QObject::sender());  // QTimer
	QWidget *toplevel = (QWidget*)(sender->parent());  // QTimer's parent which should be set to a required Toplevel
	// show and activate the toplevel
	toplevel->show();
	qApp->processEvents();
	usleep( 1000*GLOBALHOTKEYS_SHOWANDACTIVATETIMERWAITTIME );
	toplevel->raise();
	toplevel->activateWindow();
	activateWindow( toplevel->winId() );
}


void GlobalHotkeys::openRecentChat( int n )
{
	// hide the popup menu
	recentchatsmenuinactivitytimer->stop();
	recentchatsmenu->hide();
	// (re)open the chat with selected user(s)
	chat_manager->openPendingMsgs( recentchatsusers[n] );
	if( recentchatswindows.contains( n ) )  // if it was currently open chat, activate the window
	{
		// check if the chat window is still opened
		if( ! chat_manager->chats().isEmpty() )
		{
			// for each currently open chat
			foreach( ChatWidget *chat, chat_manager->chats() )
			{
				// find the window
				QWidget *win = chat;
				while( win->parent() )
					win = static_cast<QWidget *>(win->parent());
				if( win == recentchatswindows[n] )
				{
					// hide chat window first to avoid virtual desktop switching
					win->hide();
					// show and activate chat window
					QTimer *showandactivatetimer = new QTimer( win );
					connect( showandactivatetimer, SIGNAL(timeout()), this, SLOT(showAndActivateToplevel()) );
					showandactivatetimer->start( GLOBALHOTKEYS_SHOWANDACTIVATETIMERINTERVAL, true );
				}
			}
		}
	}
}


void GlobalHotkeys::recentchatsmenuinactivitytimerTimeout()
{
	Window activewindow; int revertto;
	XGetInputFocus( QX11Info::display(), &activewindow, &revertto );
	if( ! shown )
	{
		if( activewindow == recentchatsmenu->winId() )
		{
			shown = true;
		}
		else if( recentchatsmenu->isVisible() )
		{
			// hide the recentchatsmenu popup menu if it is inactive
			recentchatsmenuinactivitytimer->stop();
			recentchatsmenu->hide();
			shown = false;
		}
	}
	else
	{
		if( ( activewindow != recentchatsmenu->winId() ) )
		{
			if( recentchatsmenu->isVisible() )
			{
				// hide the recentchatsmenu popup menu if it is inactive
				recentchatsmenuinactivitytimer->stop();
				recentchatsmenu->hide();
			}
			shown = false;
		}
	}
}




bool HotkeyEdit::x11Event( XEvent *event )
{
	if( lastvalidvalue.isNull() )  // if lastvalidvalue is not initialized
	{
		lastvalidvalue = "";
	}
	if( ( event->type == KeyPress ) || ( event->type == KeyRelease ) )  // is it mouse KeyPress or KeyRelease event?
	{
		// event data
		uint keycode = event->xkey.keycode;
		KeySym keysym = XKeycodeToKeysym( QX11Info::display(), keycode, 0 );
		// result string
		QString hotkeystring = "";
		// get the modifiers
		bool mod_shift;
		bool mod_control;
		bool mod_alt;
		bool mod_altgr;
		bool mod_super;
		mod_shift   = ( ( event->xkey.state & GLOBALHOTKEYS_X11SHIFTMASK   ) != 0 );
		mod_control = ( ( event->xkey.state & GLOBALHOTKEYS_X11CONTROLMASK ) != 0 );
		mod_alt     = ( ( event->xkey.state & GLOBALHOTKEYS_X11ALTMASK     ) != 0 );
		mod_altgr   = ( ( event->xkey.state & GLOBALHOTKEYS_X11ALTGRMASK   ) != 0 );
		mod_super   = ( ( event->xkey.state & GLOBALHOTKEYS_X11SUPERMASK   ) != 0 );
		// checking events
		if( event->type == KeyPress )  // is it mouse KeyPress event?
		{
			// build hotkeystring
			if(
				( keysym != GLOBALHOTKEYS_X11LSHIFT   ) &&
				( keysym != GLOBALHOTKEYS_X11RSHIFT   ) &&
				( keysym != GLOBALHOTKEYS_X11LCONTROL ) &&
				( keysym != GLOBALHOTKEYS_X11RCONTROL ) &&
				( keysym != GLOBALHOTKEYS_X11LALT     ) &&
				( keysym != GLOBALHOTKEYS_X11RALT     ) &&
				( keysym != GLOBALHOTKEYS_X11ALTGR    ) &&
				( keysym != GLOBALHOTKEYS_X11LSUPER   ) &&
				( keysym != GLOBALHOTKEYS_X11RSUPER   )
			)
			{
				if(
						( ! mod_shift ) && ( ! mod_control ) && ( ! mod_alt ) && ( ! mod_altgr ) && ( ! mod_super ) &&
						( keysym == GLOBALHOTKEYS_CONFIGURATIONCLEARKEY )
					)
				{
					// clear key was pressed
					setText( "" );
					lastvalidvalue = "";
				}
				else
				{
					hotkeystring = "";
					hotkeystring += mod_shift   ? "Shift+"   : "";
					hotkeystring += mod_control ? "Control+" : "";
					hotkeystring += mod_alt     ? "Alt+"     : "";
					hotkeystring += mod_altgr   ? "AltGr+"   : "";
					hotkeystring += mod_super   ? "Super+"   : "";
					// keysym string or key code number
					QString keystring;
					if( keysym != NoSymbol )
					{
						keystring = XKeysymToString( keysym );
						if( keystring.isEmpty() )
						{
							// keycode instead of keysym string
							keystring.setNum( keycode );
						}
						else
						{
							// if the keysym is a single lowercase letter
							if( ( keystring.length() == 1 ) && ( keystring.at(0) >= 'a' ) && ( keystring.at(0) <= 'z' ) )
							{
								// make the letter uppercase (just to have a nice looking shortcut)
								keystring = keystring.upper();
							}
						}
					}
					else
					{
						// keycode instead of keysym string
						keystring.setNum( keycode );
					}
					hotkeystring += keystring;
					// set edit field text
					setText( hotkeystring );
				}
			}
			else
			{
				if( ( keysym == GLOBALHOTKEYS_X11LSHIFT   ) || ( keysym == GLOBALHOTKEYS_X11RSHIFT   ) )
					mod_shift   = true;
				if( ( keysym == GLOBALHOTKEYS_X11LCONTROL ) || ( keysym == GLOBALHOTKEYS_X11RCONTROL ) )
					mod_control = true;
				if( ( keysym == GLOBALHOTKEYS_X11LALT     ) || ( keysym == GLOBALHOTKEYS_X11RALT     ) )
					mod_alt     = true;
				if(   keysym == GLOBALHOTKEYS_X11ALTGR                                                 )
					mod_altgr   = true;
				if( ( keysym == GLOBALHOTKEYS_X11LSUPER   ) || ( keysym == GLOBALHOTKEYS_X11RSUPER   ) )
					mod_super   = true;
				hotkeystring = "";
				hotkeystring += mod_shift   ? "Shift+"   : "";
				hotkeystring += mod_control ? "Control+" : "";
				hotkeystring += mod_alt     ? "Alt+"     : "";
				hotkeystring += mod_altgr   ? "AltGr+"   : "";
				hotkeystring += mod_super   ? "Super+"   : "";
				// set edit field text
				setText( hotkeystring );
			}
		}
		else if( event->type == KeyRelease )
		{
			if( ( ! text().isEmpty() ) && ( text().at( text().length() - 1 ) == '+' ) )  // if the hotkey typing is not finished yet ("+" at the end)
			{
				if( ! (
					( keysym != GLOBALHOTKEYS_X11LSHIFT   ) &&
					( keysym != GLOBALHOTKEYS_X11RSHIFT   ) &&
					( keysym != GLOBALHOTKEYS_X11LCONTROL ) &&
					( keysym != GLOBALHOTKEYS_X11RCONTROL ) &&
					( keysym != GLOBALHOTKEYS_X11LALT     ) &&
					( keysym != GLOBALHOTKEYS_X11RALT     ) &&
					( keysym != GLOBALHOTKEYS_X11ALTGR    ) &&
					( keysym != GLOBALHOTKEYS_X11LSUPER   ) &&
					( keysym != GLOBALHOTKEYS_X11RSUPER   )
				) )
				{
					if( ( keysym == GLOBALHOTKEYS_X11LSHIFT   ) || ( keysym == GLOBALHOTKEYS_X11RSHIFT   ) )
						mod_shift   = false;
					if( ( keysym == GLOBALHOTKEYS_X11LCONTROL ) || ( keysym == GLOBALHOTKEYS_X11RCONTROL ) )
						mod_control = false;
					if( ( keysym == GLOBALHOTKEYS_X11LALT     ) || ( keysym == GLOBALHOTKEYS_X11RALT     ) )
						mod_alt     = false;
					if(   keysym == GLOBALHOTKEYS_X11ALTGR                                                 )
						mod_altgr   = false;
					if( ( keysym == GLOBALHOTKEYS_X11LSUPER   ) || ( keysym == GLOBALHOTKEYS_X11RSUPER   ) )
						mod_super   = false;
					hotkeystring = "";
					hotkeystring += mod_shift   ? "Shift+"   : "";
					hotkeystring += mod_control ? "Control+" : "";
					hotkeystring += mod_alt     ? "Alt+"     : "";
					hotkeystring += mod_altgr   ? "AltGr+"   : "";
					hotkeystring += mod_super   ? "Super+"   : "";
					// set edit field text
					if( ! hotkeystring.isEmpty() )
					{
						setText( hotkeystring );
					}
					else
					{
						// reset the text to the last valid value
						setText( lastvalidvalue );
					}
				}
			}
			else
			{
				lastvalidvalue = text();
			}
		}
		// don't forward the event
		return true;
	}
	else if( ( event->type == ButtonPress ) || ( event->type == ButtonRelease ) )  // is it mouse ButtonPress or ButtonRelease event?
	{
		if( event->xbutton.button == Button1 )  // is if left mouse button?
		{
			return false;  // forward the event to Qt
		}
		return true;  // don't forward the event
	}
	return false;  // forward the event to Qt
}


void HotkeyEdit::focusInEvent( QFocusEvent *event )
{
	lastvalidvalue = text();
	// important: call the default focusInEvent
	QLineEdit::focusInEvent( event );
}


void HotkeyEdit::focusOutEvent( QFocusEvent *event )
{
	if( lastvalidvalue.isNull() )  // if lastvalidvalue is not initialized
	{
		lastvalidvalue = "";
	}
	if( text().at( text().length() - 1 ) == '+' )  // if the hotkey typing is not finished yet ("+" at the end)
	{
		// reset the text to the last valid value
		setText( lastvalidvalue );
	}
	// important: call the default focusOutEvent
	QLineEdit::focusOutEvent( event );
}




void EmptyMsgHandler( QtMsgType type, const char *msg )
{
	messageshandled = 1;
}
