/* SocketFactory.java -- factory for client sockets.
   Copyright (C) 2003  Casey Marshall <rsdio@metastatic.org>

This program is free software; you can redistribute it and/or modify it
under the terms of the GNU General Public License as published by the
Free Software Foundation; either version 2 of the License, or (at your
option) any later version.

This program is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
General Public License for more details.

You should have received a copy of the GNU General Public License along
with this program; if not, write to the

   Free Software Foundation, Inc.,
   59 Temple Place, Suite 330,
   Boston, MA  02111-1307
   USA

Linking this library statically or dynamically with other modules is
making a combined work based on this library.  Thus, the terms and
conditions of the GNU General Public License cover the whole
combination.

As a special exception, the copyright holders of this library give you
permission to link this library with independent modules to produce an
executable, regardless of the license terms of these independent
modules, and to copy and distribute the resulting executable under terms
of your choice, provided that you also meet, for each linked independent
module, the terms and conditions of the license of that module.  An
independent module is a module which is not derived from or based on
this library.  If you modify this library, you may extend this exception
to your version of the library, but you are not obligated to do so.  If
you do not wish to do so, delete this exception statement from your
version.  */


package javax.net;

import java.io.IOException;
import java.net.InetAddress;
import java.net.Socket;
import java.net.UnknownHostException;
import java.security.Security;

/**
 * A factory for client sockets. The purpose of this class is to serve
 * as the superclass of server socket factories that produce client
 * sockets of a particular type, such as <i>Secure Socket Layer</i>
 * (<b>SSL</b>) sockets.
 *
 * @author Casey Marshall (rsdio@metastatic.org)
 */
public abstract class SocketFactory
{

  // Constructor.
  // -------------------------------------------------------------------

  /**
   * Default 0-arguments constructor.
   */
  protected SocketFactory()
  {
    super();
  }

  // Class methods.
  // -------------------------------------------------------------------

  /**
   * Returns the default socket factory. The type of factory
   * returned may depend upon the installation.
   *
   * @return The default socket factory.
   */
  public static synchronized SocketFactory getDefault()
  {
    try
      {
        String s = Security.getProperty("gnu.defaultSocketFactory");
        if (s != null)
          {
            Class c = Class.forName(s);
            return (SocketFactory) c.newInstance();
          }
      }
    catch (Exception e)
      {
      }
    return new VanillaSocketFactory();
  }

  // Instance methods.
  // -------------------------------------------------------------------

  /**
   * Returns an unbound client socket.
   *
   * @return The new, unbound socket.
   */
  public Socket createSocket() throws IOException
  {
    throw new UnsupportedOperationException();
  }

  /**
   * Creates a socket connected to a given host on a given port.
   *
   * @param host The hostname to connect to.
   * @param port The port on <i>host</i> to connect to.
   * @return A socket connected to <i>host</i> on <i>port</i>.
   * @throws IOException If a network error occurs.
   * @throws UnknownHostException If <i>host</i> cannot be resolved.
   */
  public abstract Socket createSocket(String host, int port) throws IOException, UnknownHostException;

  /**
   * Creates a socket connected to a given host on a given port,
   * connecting locally to the interface with the given address and port.
   *
   * @param host The hostname to connect to.
   * @param port The port on <i>host</i> to connect to.
   * @param localHost The address of the local interface to bind to.
   * @param localPort The local port to bind to.
   * @return A socket connected to <i>host</i> on <i>port</i>.
   * @throws IOException If a network error occurs.
   * @throws UnknownHostException If <i>host</i> cannot be resolved.
   */
  public abstract Socket createSocket(String host, int port, InetAddress localHost, int localPort) throws IOException, UnknownHostException;

  /**
   * Creates a socket connected to a given host on a given port.
   *
   * @param host The host address to connect to.
   * @param port The port on <i>host</i> to connect to.
   * @return A socket connected to <i>host</i> on <i>port</i>.
   * @throws IOException If a network error occurs.
   */
  public abstract Socket createSocket(InetAddress host, int port) throws IOException;

  /**
   * Creates a socket connected to a given host on a given port,
   * connecting locally to the interface with the given address and port.
   *
   * @param host The host address  to connect to.
   * @param port The port on <i>host</i> to connect to.
   * @param localHost The address of the local interface to bind to.
   * @param localPort The local port to bind to.
   * @return A socket connected to <i>host</i> on <i>port</i>.
   * @throws IOException If a network error occurs.
   */
  public abstract Socket createSocket(InetAddress hast, int port, InetAddress localHost, int localPort) throws IOException;
}
