#!/usr/bin/env python
# -*- coding: utf-8 -*-

################################################################################
##
## Copyright (C) 2009 Richard A. Johnson <rjohnson@kde.org>
##
##    This program is free software: you can redistribute it and/or modify
##    it under the terms of the GNU General Public License as published by
##    the Free Software Foundation, either version 3 of the License, or
##    (at your option) any later version.
##
##    This program is distributed in the hope that it will be useful,
##    but WITHOUT ANY WARRANTY; without even the implied warranty of
##    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
##    GNU General Public License for more details.
##
##    You should have received a copy of the GNU General Public License
##    along with this program.  If not, see <http://www.gnu.org/licenses/>.
##    
################################################################################

from PyQt4.QtCore import *
from PyQt4.QtGui import *

from PyKDE4.kdecore import *
from PyKDE4.kdeui import *

from subprocess import Popen, PIPE, STDOUT

import sys

class App(QObject):
    def __init__(self):
        QObject.__init__(self)
        
        # Wireless devices
        self.wifi_device = self.getWiFiDeviceInfo()
        self.wifi_power = self.getWiFiDevicePowerStatus()
        self.bluetooth_device = self.getBluetoothDeviceInfo()
        self.bluetooth_power = self.getBluetoothDevicePowerStatus()
        self.threeg_device = self.get3GDeviceInfo()
        self.threeg_power = self.get3GDevicePowerStatus()
        
        # Airplane Mode
        self.airplane_mode = self.isAirplaneMode()
        
        # Tray Icon
        self.tray_icon = KSystemTrayIcon(KIcon("network-wireless"))
        self.setTrayToolTip()
        self.setMenu()
        self.connect(self.tray_icon, SIGNAL("activated(QSystemTrayIcon::ActivationReason)"), self.activated)
	self.tray_icon.show()
        
    ####################
    ## UI CODE
    ####################
    #---- Tray Icon Clicked ----#
    def activated(self, activationReason):
        if activationReason == QSystemTrayIcon.Trigger:
	    if self.airplane_mode:
		msg = i18n("<b>Disable Airplane Mode?</b>\n\n"
		           "Disabling Airplane Mode will turn the power for\n"
			   "all wireless devices on, therefore putting your\n"
			   "system in a non-safe condition during flight.")
		btn = i18n("Disable Airplane Mode")
	    else:
		msg = i18n("<b>Enable Airplane Mode?</b>\n\n"
		           "Enabling Airplane Mode will turn the power for\n"
			   "all wireless devices off, therefore putting your\n"
		           "system in a safe condition during flight.")
		btn = i18n("Enable Airplane Mode")
	    yesButton = KGuiItem(btn)
	    confirm = KMessageBox.questionYesNo(None, msg, i18n("KDE Airplane Mode"), yesButton, KStandardGuiItem.cancel())
	    if confirm == KMessageBox.Yes:
		self.toggleAirplaneMode()
            
    #---- Notifications ----#
    def notifyUser(self, event, message):
        KNotification.event(event, message, QPixmap(), None,
                            KNotification.CloseOnTimeout,
                            KComponentData("kairmode", "kairmode",
                                           KComponentData.SkipMainComponentRegistration))
        
    #---- Tray Icon Tooltip ----#
    def setTrayToolTip(self):
        tooltip = QString()
        if self.wifi_device:
            tooltip.append(i18n("WiFi device (%1) is currently %2\n", self.wifi_device, self.wifi_power))
        if self.bluetooth_device:
            tooltip.append(i18n("Bluetooth device (%1) is currently %2\n", self.bluetooth_device, self.bluetooth_power))
        if self.threeg_device:
            tooltip.append(i18n("3G device (%1) is currently %2\n", self.threeg_device, self.threeg_power))

	if len(tooltip) == 0:
	    tooltip.append(i18n("No devices are present.\n"))

        if self.airplane_mode:
            tooltip.append(i18n("\nAirplane Mode Enabled"))
        elif not self.airplane_mode:
            tooltip.append(i18n("\nAirplane Mode Disabled"))

	self.tray_icon.setToolTip(tooltip)

    #---- Set Menu ----#
    def setMenu(self):
        self.menu = KMenu()
        if self.wifi_device:
            if self.wifi_power == "off" or not self.wifi_power: w = "on"
            elif self.wifi_power == "on": w = "off"
            self.menu.addAction(KIcon("network-wireless"), i18n("Turn wireless device (%1) %2", self.wifi_device, w), self.toggleWiFi)
        if self.bluetooth_device:
            if self.bluetooth_power == "off" or not self.bluetooth_power: b = "on"
            elif self.bluetooth_power == "on": b = "off"
            self.menu.addAction(KIcon("preferences-system-bluetooth"), i18n("Turn bluetooth device (%1) %2", self.bluetooth_device, b), self.toggleBluetooth)
        if self.threeg_device:
            if self.threeg_power == "off" or  not self.threeg_power: t = "on"
            elif self.threeg_power == "on": t = "off"
            self.menu.addAction(KIcon("phone"), i18n("Turn 3G device (%1) %2", self.threeg_device, t), self.toggle3G)
	if self.wifi_power or self.bluetooth_power or self.threeg_power:
	    self.menu.addSeparator()
	    if self.airplane_mode:
		self.menu.addAction(KIcon("preferences-system-performance"), i18n("Disable Airplane Mode"), self.toggleAirplaneMode)
	    else:
		self.menu.addAction(KIcon("preferences-system-performance"), i18n("Enable Airplane Mode"), self.toggleAirplaneMode)
	    self.menu.addSeparator()
        self.menu.addAction(KIcon("application-exit"), i18n("Quit"), self.toggleQuit)
        self.tray_icon.setContextMenu(self.menu)
        
    #---- Toggle Quit ----#
    def toggleQuit(self):
	sys.exit(app.quit())
    
    ####################
    ## COMMANDS
    ####################
    #---- Iwconfig (WiFi) ----#
    def getIwconfigInfo(self, flags):
        if flags:
            info = Popen('kdesudo -c "iwconfig %s"' % flags, shell=True, stdout=PIPE, stderr=STDOUT).stdout
        else:
            info = Popen('kdesudo -c "iwconfig"', shell=True, stdout=PIPE, stderr=STDOUT).stdout
        if not info:
            return None
        return info
    
    #---- Hciconfig (Bluetooth) ----#
    def getHciconfigInfo(self, flags):
        if flags:
            info = Popen('kdesudo -c "hciconfig %s"' % flags, shell=True, stdout=PIPE, stderr=STDOUT).stdout
        else:
            info = Popen('kdesudo -c "hciconfig"', shell=True, stdout=PIPE, stderr=STDOUT).stdout
        if not info:
            return None
        return info
    
    ####################
    ## DEVICE CODE
    ####################
    #---- Get WiFi Device ----#
    def getWiFiDeviceInfo(self):
	try:
	    device = [(x[0:x.find(" ")]) for x in self.getIwconfigInfo(flags=None) if ("ESSID" in x)][0]
	except IndexError:
	    return None
        return device
    
    #---- Get Bluetooth Device ----#
    def getBluetoothDeviceInfo(self):
	try:
	    device = [(x[0:x.find(" ")]) for x in self.getHciconfigInfo(flags=None) if ("Type" in x)][0].strip().split(":")[0]
	except IndexError:
	    return None
        return device
    
    #---- Get 3G Device ----#
    def get3GDeviceInfo(self):
        # TODO: Not implemented yet
        return None
    
    #---- Get WiFi Power Status ----#
    def getWiFiDevicePowerStatus(self):
        if not self.wifi_device: return None
	# The reason for the same thing over and over here is that some devices
	# use : as a separator and some use =. Also you can notice the removal
	# of an extra .split for devices that are off. The reason for this is
	# that there is no extra space in this case.
        try:
	    # power is on
            power = [(x[0:x.find("\n")]) for x in self.getIwconfigInfo(self.wifi_device) if ("Tx-Power" in x)][0].strip().split("  ")[1].split(":")[1]
        except IndexError:
	    try:
		# power is off
		power = [(x[0:x.find("\n")]) for x in self.getIwconfigInfo(self.wifi_device) if ("Tx-Power" in x)][0].strip().split(":")[1]
	    except IndexError:
		try:
		    # power is on
		    power = [(x[0:x.find("\n")]) for x in self.getIwconfigInfo(self.wifi_device) if ("Tx-Power" in x)][0].strip().split("  ")[1].split("=")[1]
		except IndexError:
		    try:
			# power is off
			power = [(x[0:x.find("\n")]) for x in self.getIwconfigInfo(self.wifi_device) if ("Tx-Power" in x)][0].strip().split("=")[1]
		    except IndexError:
			power = None
        if not power: return None
        if power != "off": return "on"
        return power
    
    #---- Get Bluetooth Power Status ----#
    def getBluetoothDevicePowerStatus(self):
        if not self.bluetooth_device: return None
        try:
            power = [(x[0:x.find(" ")]) for x in self.getHciconfigInfo(self.bluetooth_device) if ("UP" in x)][0].strip()
        except IndexError:
            try:
                power = [(x[0:x.find(" ")]) for x in self.getHciconfigInfo(self.bluetooth_device) if ("DOWN" in x)][0].strip()
            except IndexError:
                return None
        if power == "UP": return "on"
        elif power == "DOWN": return "off"
        return None
    
    #---- Get 3G Power Status ----#
    def get3GDevicePowerStatus(self):
        # TODO: Not implemented yet
        return None
    
    #---- Toggle WiFi ----#
    def toggleWiFi(self, am_selected=None):
        if self.wifi_power:
            if self.wifi_power == "on":
                next_state = "off"
                notify_event = "WiFiPowerOff"
            elif self.wifi_power == "off":
                next_state = "on"
                notify_event = "WiFiPowerOn"
        else:
            return
        toggle = Popen('kdesudo -c "iwconfig %s txpower %s"' % (self.wifi_device, next_state), shell=True, stdout=PIPE, stderr=PIPE)
        (stdout, stderr) = toggle.communicate()
        if not stderr:
            self.wifi_power = self.getWiFiDevicePowerStatus()
	    if not am_selected:
		self.notifyUser(notify_event, i18n("WiFi device (%1) powered %2", self.wifi_device, next_state))
        else:
            self.notifyUser(notify_event, i18n("ERROR: Unable to turn WiFi device (%1) %2", self.wifi_device, next_state))
        self.airplane_mode = self.isAirplaneMode()
        self.setTrayToolTip()
        self.setMenu()
    
    #---- Toggle Bluetooth ----#
    def toggleBluetooth(self, am_selected=None):
        if self.bluetooth_power:
            if self.bluetooth_power == "on":
                next_state = "down"
                next_state_name = "off"
                notify_event = "BluetoothPowerOff"
            elif self.bluetooth_power == "off":
                next_state = "up"
                next_state_name = "on"
                notify_event = "BluetoothPowerOn"
        else:
            return
        toggle = Popen('kdesudo -c "hciconfig %s %s"' % (self.bluetooth_device, next_state), shell=True, stdout=PIPE, stderr=PIPE)
        (stdout, stderr) = toggle.communicate()
        if not stderr:
            self.bluetooth_power = self.getBluetoothDevicePowerStatus()
	    if not am_selected:
		self.notifyUser(notify_event, i18n("Bluetooth device (%1) powered %2", self.bluetooth_device, next_state_name))
        else:
            self.notifyUser(notify_event, i18n("ERROR: Unable to turn Bluetooth device (%1) %2", self.bluetooth_device, next_state_name))
        self.airplane_mode = self.isAirplaneMode()
        self.setTrayToolTip()
        self.setMenu()
    
    #---- Toggle 3G ----#
    def toggle3G(self, am_selected=None):
        # TODO: Not implemented yet
        pass
    
    #---- Toggle Airplane Mode ----#
    @pyqtSignature("toggleAirplaneMode()")
    def toggleAirplaneMode(self):
        if not self.airplane_mode:
            if self.wifi_device and self.wifi_power == "on":
		self.toggleWiFi(True)
            if self.bluetooth_device and self.bluetooth_power == "on":
		self.toggleBluetooth(True)
            if self.threeg_device and self.threeg_power == "on":
		self.toggle3G(True)
            self.airplane_mode = self.isAirplaneMode()
            if self.airplane_mode:
                self.notifyUser("EnableAirplaneMode", i18n("Airplane Mode Enabled - All wireless devices powered off"))
            else:
                self.notifyUser("EnableAirplaneMode", i18n("ERROR: Attempting to enable Airplane Mode failed!"))
        else:
	    if self.wifi_device:
		if self.wifi_power == "off" or not self.wifi_power:
		    self.toggleWiFi(True)
	    if self.bluetooth_device:
		if self.bluetooth_power == "off" or not self.bluetooth_power:
		    self.toggleBluetooth(True)
	    if self.threeg_device:
		if self.threeg_power == "off" or not self.threeg_power:
		    self.toggle3G(True)
            self.airplane_mode = self.isAirplaneMode()
            if not self.airplane_mode:
                self.notifyUser("DisableAirplaneMode", i18n("Airplane Mode Disabled - All wireless devices powered on"))
            else:
                self.notifyUser("DisableAirplaneMode", i18n("ERROR: Attempting to disable Airplane Mode failed!"))
        self.setTrayToolTip()
        self.setMenu()
    
    #---- Determine if Airplane Mode ----#
    def isAirplaneMode(self):
        status = True
	if self.wifi_device and self.wifi_power == "on":
	    status = False
	if self.bluetooth_device and self.bluetooth_power == "on" and status == True:
	    status = False
	if self.threeg_device and self.threeg_power == "on" and status == True:
	    status = False
        return status
            
####################
## MAIN
####################
if __name__ == "__main__":
    appName = "kairmode"
    catalog = "kairmode"
    programName = ki18n("KDE Airplane Mode")
    version = "1.0"
    description = ki18n("A KDE system tray applet that allows the user to "
                        "enable or disable the power to any or all wireless "
                        "devices.")
    license = KAboutData.License_GPL_V3
    copyright = ki18n("2009 Richard A. Johnson")
    text = KLocalizedString()
    homePage = "http://www.kde.org"
    bugEmail = "rjohnson@kde.org"
    
    aboutData = KAboutData(appName, catalog, programName, version, description,
                           license, copyright, text, homePage, bugEmail)
    
    aboutData.addAuthor(ki18n("Richard A. Johnson"), ki18n("Author"))
    
    KCmdLineArgs.init(sys.argv, aboutData)
    
    app = KApplication()
    app.setQuitOnLastWindowClosed(False)
    applet = App()
    app.exec_() 
        
